<?php

App::uses('DiscountType', 'EvDiscount.Model');
App::uses('EvDiscountAppModel', 'EvDiscount.Model');

class Discount extends EvDiscountAppModel {

	public $displayField = 'code';

	public $belongsTo = array(
		'DiscountType' => array(
			'className' => 'EvDiscount.DiscountType'
		)
	);

	public $hasAndBelongsToMany = array(
		'Brand' => array(
			'className' => 'EvShop.Brand',
			'joinTable' => 'ev_discount_brands_discounts'
		),
		'Product' => array(
			'className' => 'EvShop.Product',
			'joinTable' => 'ev_discount_discounts_products'
		)
	);

	public $hasMany = array(
		// Define the association to the brands join table for easier querying.
		'BrandsDiscount' => array(
			'className' => 'EvDiscount.BrandsDiscount'
		),
		'DiscountsProduct' => array(
			'className' => 'EvDiscount.DiscountsProduct'
		)
	);

	public $validate = array(
		'code' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'unique' => array(
				'rule' => 'isUnique',
				'message' => 'Discount codes must be unique'
			)
		),
		'discount_type_id' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'amount' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'min_order' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		),
		'max_uses' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			)
		)
	);

/**
 * Construct and add in virtual fields
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		// Work out the remaining uses of a discount code for convinience.
		$this->virtualFields['remaining_uses'] = $this->escapeField('max_uses') . ' - ' . $this->escapeField('uses');
	}

/**
 * Query used to retrieve a record ready for edit
 *
 * @param int $id ID of row to edit
 * @param array $query The db query array - can be used to pass in additional parameters such as contain
 * @return array
 */
	public function readForEdit($id, $query = array()) {
		$query['contain'] = array(
			'Brand',
			'Product'
		);
		return parent::readForEdit($id, $query);
	}

/**
 * Calculates discounts for an order.
 *
 * @param array $cart
 * @param string $discountCode
 * @return float
 */
	public function calculateDiscount($cart, $discountCode) {
		$subtotal = $cart['Order']['subtotal'];

		// Check for a site setting to work out if the discount can be applied
		// after tax instead of before.
		if (Configure::read('SiteSetting.discount_min_amount_after_tax') == '1') {
			$minOrderTotal = $cart['Order']['total'];
		} else {
			$minOrderTotal = $subtotal;
		}

		$now = date('Y-m-d H:i:s');

		$discount = $this->find(
			'first',
			array(
				'contain' => array(
					'BrandsDiscount',
					'DiscountsProduct'
				),
				'conditions' => array(
					'Discount.code' => $discountCode,
					'Discount.min_order <=' => $subtotal,
					array(
						'OR' => array(
							'Discount.start_date' => null,
							'Discount.start_date <=' => $now
						)
					),
					array(
						'OR' => array(
							'Discount.end_date' => null,
							'Discount.end_date >=' => $now
						)
					),
					'Discount.remaining_uses >' => 0,
					'Discount.is_active' => true
				)
			)
		);

		$discountTotal = 0;

		if (!empty($discount)) {

			// Find the subtotals for each product in the basket (we may have
			// multiple rows for each variant of the product so need to workout
			// the sums here).
			$products = [];
			$result = Hash::combine($cart['OrderItem'], '{n}.id', '{n}.subtotal', '{n}.product_id');
			foreach ($result as $productId => $items) {
				$products[$productId] = array_sum($items);
			}

			if (!empty($discount['DiscountsProduct'])) {
				$products = array_intersect_key(
					$products,
					Hash::combine($discount['DiscountsProduct'], '{n}.product_id', '{n}.discount_id')
				);
				$subtotal = array_sum($products);
			}

			if (!empty($discount['BrandsDiscount'])) {
				$discountableProducts = $this->getDiscountableProductsByBrand(
					Hash::extract($discount['BrandsDiscount'], '{n}.brand_id'),
					array_keys($products)
				);
				$products = array_intersect_key($products, array_flip($discountableProducts));
				$subtotal = array_sum($products);
			}

			$discountTotal = static::calculate(
				$subtotal,
				$discount['Discount']['amount'],
				$discount['Discount']['discount_type_id']
			);
		}

		return $discountTotal;
	}

/**
 * Returns all products eligible for discount by brand
 *
 * @param array $brands
 * @param array $products
 * @return array
 */
	public function getDiscountableProductsByBrand($brands, $products) {
		$result = ClassRegistry::init('EvRelatedItems.RelatedItem')->find(
			'all',
			array(
				'conditions' => array(
					'RelatedItem.related_model_id' => $brands,
					'RelatedItem.related_model' => 'Brand',
					'RelatedItem.model_id' => $products,
					'RelatedItem.model' => 'Product'
				)
			)
		);
		return Hash::extract($result, '{n}.RelatedItem.model_id');
	}

/**
 * Calculates the discount of the specified amount.
 *
 * @param float $subtotal
 * @param float $discountAmount
 * @param int $discountType
 * @return float
 */
	public static function calculate($subtotal, $discountAmount, $discountType) {
		$discountTotal = 0;

		if ((int)$discountType === DiscountType::PERCENTAGE) {
			$discountTotal = $subtotal * $discountAmount / 100;
		} elseif ((int)$discountType === DiscountType::FIXED) {
			// Discount can never be more than the total payable.
			$discountTotal = $discountAmount > $subtotal ? $subtotal : $discountAmount;
		}

		return $discountTotal;
	}

}
