<?php
/* (c) Sam Biggins (sam@evoluted.net)
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Deployer;

use Deployer\Exception\RuntimeException;
use Deployer\Task\Context;

use function Evoluted\Deployer\GitLab\getDeployPath;
use function Evoluted\Deployer\VhostApi\notifyVhostApiManager;

require_once 'db.php';
require_once 'vhost_api.php';
require_once 'crontab.php';
require_once 'supervisord.php';
require_once 'gitlab-ci.php';

$cdHost = host('cd')
    ->hostname('cd.evoluted.net')
    ->user('deploy')
    ->stage('cd')
    ->set('deploy_path', getDeployPath());

desc('Creates the .env file setup for the environment, only runs for CD stage');
task('ci:setup:env', function () {
	writeln('<info>Writing .env file</info>');

	$db_name = get('cd_database_name');
	$db_user = get('cd_database_user');
	$db_pass = get('cd_database_password');
	$db_host = get('cd_database_host');
	$db_port = get('cd_database_port', '3306');

	switch (get('application_type')) {
        case 'cakephp':
        case 'cakephp2':
        case 'cakephp3':
		case 'cake70':
		case 'cake71':
		case 'cake72':
		case 'cake73':
		case 'cake74':
		case 'cake82':
		case 'cake83':
		case 'cake84':
		case 'cake85':
		case 'cake56apache':
				$envContents = "DB_NAME='${db_name}'
DB_USER='${db_user}'
DB_PASS='${db_pass}'
DB_HOST='${db_host}'

ENVIRONMENT='PREVIEW'
BRANCH='" . getenv('CI_COMMIT_REF_SLUG') . "'

PREVIEW_URL='" . getenv('CI_ENVIRONMENT_URL') . "'
";

			// If the site is a multiapp with separate databases add prefix
			if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
				$applicationPrefix = getenv('APPLICATION_PREFIX');

				$envContents .= "\nAPPLICATION_PREFIX=" . $applicationPrefix;
			}

			break;
		case 'laravel':
		case 'laravel70':
		case 'laravel71':
		case 'laravel72':
		case 'laravel73':
		case 'laravel74':
		case 'laravel80':
		case 'laravel81':
		case 'laravel82':
		case 'laravel83':
		case 'laravel84':
		case 'laravel85':
		case 'craftcms':
		case 'craftcms74':
		case 'craftcms80':
		case 'craftcms81':
		case 'craftcms82':
		case 'craftcms83':
		case 'craftcms84':
		case 'craftcms85':
			// Build a list of env paths to try in priority order
			$envPaths = [
				get('release_path') . '/.env.cd.' . getenv('CI_COMMIT_REF_SLUG'),
				get('release_path') . '/.env.cd',
			];
			if (getenv('APPLICATION_PREFIX')) {
				// If there's an application prefix prioritise that
                $envPaths = array_merge([
                    get('release_path') . '/.env.cd.' . getenv('CI_COMMIT_REF_SLUG') . '-' . getenv('APPLICATION_PREFIX'),
                    get('release_path') . '/.env.cd-' . getenv('APPLICATION_PREFIX'),
                ], $envPaths);
			}

			// Search for an existing env path
			$envPath = current($envPaths);
			writeln('<info>Looking for deploy specific .env file at "' . $envPath . '"</info>');
			while (!file_exists($envPath)) {
				$envPath = next($envPaths);
				if (empty($envPath)) {
					throw new RuntimeException(
						Context::get()->getHost()->getHostname(),
						'ci:setup:env',
						404,
						'No env.cd file in the repository',
						''
					);
				}
				writeln('<info>Looking for deploy specific .env file at "' . $envPath . '"</info>');
			}

			$envContents = file_get_contents($envPath);

			$applicationDomain = getDomainName();

			// Replace database information
			$envContents = preg_replace("/DB_DATABASE=REPLACE/i", "DB_DATABASE='${db_name}'", $envContents);
			$envContents = preg_replace("/DB_HOST=REPLACE/i", "DB_HOST='${db_host}'", $envContents);
			$envContents = preg_replace("/DB_PORT=REPLACE/i", "DB_PORT='${db_port}'", $envContents);
			$envContents = preg_replace("/DB_USERNAME=REPLACE/i", "DB_USERNAME='${db_user}'", $envContents);
			$envContents = preg_replace("/DB_PASSWORD=REPLACE/i", "DB_PASSWORD='${db_pass}'", $envContents);
			$envContents = preg_replace("/APPLICATION_DOMAIN=REPLACE/i", "APPLICATION_DOMAIN='" . $applicationDomain . "'", $envContents);
			// If the site is a multiapp with separate databases add to env file
			if (getenv('APPLICATION_PREFIX')) {
				$envContents = preg_replace("/APPLICATION_PREFIX=REPLACE/i", "APPLICATION_PREFIX='" . getenv('APPLICATION_PREFIX') . "'", $envContents);
			}
			break;
		case 'codeigniter-apache':
			// find / replace in database config file
			$path = get('release_path') . '/application/config/database.php.dist';

			writeln('<info>Looking for database config dist file at "' . $path . '"</info>');

			if (file_exists($path)) {
				$content = file_get_contents($path);
				$searchReplace = [
					'{__DB_HOST__}' => $db_host,
					'{__DB_USERNAME__}' => $db_user,
					'{__DB_PASSWORD__}' => $db_pass,
					'{__DB_DATABASE__}' => $db_name,
				];

				$newContent = escapeshellarg(str_replace(array_keys($searchReplace), array_values($searchReplace), $content));
				run('echo "' . base64_encode($newContent) . '" | base64 -d > ' . get('release_path') . '/application/config/database.php');
				writeln('<info>Wrote ' . get('release_path') . '/application/config/database.php file');
			} else {
				writeln('<error>Could not find config dist file</error>');
			}

			$envContents = '';
			break;
		case 'smarty':
			$envContents = "DB_NAME='${db_name}'
DB_USER='${db_user}'
DB_PASS='${db_pass}'
DB_HOST='${db_host}'

ENVIRONMENT='PREVIEW'
BRANCH='" . getenv('CI_COMMIT_REF_SLUG') . "'

PREVIEW_URL='" . getenv('CI_ENVIRONMENT_URL') . "'
";

			// If the site is a multiapp with separate databases add prefix
			if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
				$applicationPrefix = getenv('APPLICATION_PREFIX');

				$envContents .= "\nAPPLICATION_PREFIX=" . $applicationPrefix;
			}
			break;
		default:
			$envContents = '';
			break;
	}

	// Add .env file to deployment
	run('echo "' . base64_encode($envContents) . '" | base64 -d > ' . get('release_path') . "/.env");
})->onStage('cd');

desc('Rsync across assets');
task('ci:setup:assets', function () {});

desc('Configure Vhost and PHP-FPM');
task('ci:vhost:configure', function () {
	$hostname = getDomainName();
	// For legacy the application_type includes the PHP version and the type of application so splitting this out
    $applicationInformation = splitApplicationType(get('application_type'));

    $vhostBuilt = notifyVhostApiManager($hostname, $applicationInformation['applicationType'], $applicationInformation['phpVersion'], 'php-fpm');

	if (! $vhostBuilt['success']) {
		if (! is_array($vhostBuilt['errors'])) {
			writeln($vhostBuilt['errors']);
		} else {
			foreach ($vhostBuilt['errors'] as $error) {
				writeln($error);
			}
		}
		writeln($vhostBuilt['message']);

		throw new RuntimeException(
			Context::get()->getHost()->getHostname(),
			'ci:vhost:configure',
			500,
			'Error communicating with the Vhost API',
			''
		);
	}
	reloadWebserver();
})->onStage('cd');

desc('Removes the environemnt from the server, used by Gitlab CD to tidy up');
task('ci:destroy', function () {
	$deployPath = get('deploy_path');

	if (!preg_match('#^/var/www/cd/#', $deployPath)) {
		// Just a safety catch incase deploy path is set to / by accident or something
		// we would like to limit our damage
		return false;
	}

	run("rm -rf $deployPath");

	writeln('<error>WARNING: Removing vhost</error>');
	$hostname = getDomainName();
	if (file_exists('/etc/nginx/conf.d/' . $hostname . '.conf')) {
		run('rm /etc/nginx/conf.d/' . $hostname . '.conf');
	}

	if (file_exists('/etc/apache2/sites-enabled/' . $hostname . '.conf')) {
		run('rm /etc/apache2/sites-enabled/' . $hostname . '.conf');
	}

	reloadWebserver();
});

/**
 * Set up a root task, if this isn't defined then the ci:setup:env task doesn't run
 */
try {
    task('deploy:init');
} catch (\InvalidArgumentException $e) {
    task('deploy:init', function() { });
}

/**
 * Gets the domain name for the deployment
 * @return string 		Domain name without protocol
 */
function getDomainName() {
	$hostname = preg_replace('#^https?://#', '', getenv('CI_ENVIRONMENT_URL'));

	// If it is a multi-site with separate DB's and files etc domain needs to be prefixed
	if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
		// Check that the prefix isn't already appended
		if (!preg_match('/' . getenv('APPLICATION_PREFIX') . '\./', $hostname)) {
			$hostname = getenv('APPLICATION_PREFIX') . '.' . $hostname;
		}
	}

	return $hostname;
}

/**
 * Reloads the webserver, handles different applications webserver types
 */
function reloadWebserver() {
	// Based on the application type there may be different services to reload
	switch (get('application_type')) {
		case 'laravel70':
		case 'cake70':
			run('sudo service php7.0-fpm reload');
			break;
		case 'cake71':
		case 'laravel71':
			run('sudo service php7.1-fpm reload');
			break;
		case 'cake72':
		case 'laravel72':
			run('sudo service php7.2-fpm reload');
			break;
		case 'cake73':
		case 'laravel73':
			run('sudo service php7.3-fpm reload');
			break;
		case 'cake74':
        case 'laravel74':
            run('sudo service php7.4-fpm reload');
            break;
        case 'laravel80':
        case 'craftcms80':
            run('sudo service php8.0-fpm reload');
            break;
        case 'laravel81':
        case 'craftcms81':
            run('sudo service php8.1-fpm reload');
            break;
        case 'cake82':
        case 'laravel82':
        case 'craftcms82':
            run('sudo service php8.2-fpm reload');
        	break;
        case 'cake83':
        case 'laravel83':
        case 'craftcms83':
            run('sudo service php8.3-fpm reload');
        	break;
        case 'cake84':
        case 'laravel84':
        case 'craftcms84':
            run('sudo service php8.4-fpm reload');
        	break;
        case 'cake85':
        case 'laravel85':
        case 'craftcms85':
            run('sudo service php8.5-fpm reload');
        	break;
		case 'cake56apache':
		case 'smarty':
			run('sudo service apache2 reload');
			break;
	}
}

/* HOOKS START */
after('deploy:init', 'ci:database:provision');
after('ci:database:provision', 'ci:setup:env');

// On deployment
after('deploy:symlink', 'ci:vhost:configure');
after('deploy:symlink', 'crontab:update');
after('deploy:symlink', 'supervisor:configure');

// On environment destroy
after('ci:destroy', 'ci:database:teardown');
after('ci:destroy', 'crontab:destroy');
after('ci:destroy', 'supervisor:destroy');
/*  HOOKS END  */

return $cdHost;
