<?php

namespace Evoluted\Deployer\Db;

use PDO;
use PDOException;
use InvalidArgumentException;

/**
 * Get a valid database name based upon the project, application, branch and optional prefix
 *
 * @param string $project E.g. "photocentric"
 * @param string $branch E.g. "feature/something-splendid"
 * @param string $application E.g. "cake"
 * @param string $prefix
 * @param string $type The database type e.g. 'mysql', 'postgresql'
 * @return string A database name valid for this class's type
 */
function getDatabaseName($project, $branch, $application = '', $prefix = 'cd_', $type = 'mysql')
{
    if (!empty($application)) {
        $application .= '_';
    }

    $name = sprintf('%s%s_%s%s', $prefix, $project, $application, $branch);
    $name = str_replace('-', '_', $name);

    if (strlen($name) >= getMaximumDatabaseNameLength($type)) {
        $name = shortenDatabaseName($name, $project, $branch, $prefix, $type);
    }

    return $name;
}

/**
 * Shorten the supplied database name to make it valid for the type of database
 *
 * @param string $currentName
 * @param string $project
 * @param string $branch
 * @param string $prefix
 * @param string $type The database type e.g. 'mysql', 'postgresql'
 * @return string
 */
function shortenDatabaseName($currentName, $project, $branch, $prefix, $type = 'mysql')
{
    if (strlen($currentName) < getMaximumDatabaseNameLength($type)) {
        return $currentName;
    }

    // convert the full name to the base-32 representation of the CRC32 hash
    $hash = base_convert(crc32($currentName), 10, 32);
    $newName = sprintf('%s%s_%s', $prefix, $project, $hash);

    // With our remaining chars use a substring of the branch name
    $subStringBranchName = substr($branch, strlen($newName) - getMaximumDatabaseNameLength($type));
    $newName .= '_' . $subStringBranchName;

    return str_replace('-', '_', $newName);
}

/**
 * Get the maximum number of characters a database name can be
 *
 * @return int
 */
function getMaximumDatabaseNameLength($type)
{
    switch ($type) {
        case 'mysql':
        case 'postgresql':
            return 63;
    }

    return PHP_INT_MAX;
}

/**
 * Whether the database with $name exists
 *
 * @param PDO $connection
 * @param string $name
 * @param string $type E.g. mysql, postgresql
 * @return bool
 */
function databaseExists(PDO $connection, $name, $type = 'mysql')
{
    try {
        $stmt = null;

        switch ($type) {
            case 'mysql':
                $stmt = $connection->prepare('SHOW DATABASES LIKE :name');
                $stmt->execute([ ':name' => $name ]);
                break;
            case 'postgresql':
                $stmt = $connection->prepare('SELECT datname FROM pg_database WHERE datname LIKE :name');
                $stmt->execute([ ':name' => $name ]);
                break;
        }

        return count($stmt->fetchAll()) > 0;
    } catch (PDOException $e) { }

    return false;
}

/**
 * Create a database with $name
 *
 * @param PDO $connection
 * @param string $name
 * @param string $type E.g. mysql, postgresql Currently unused
 * @return bool Whether the database was created or not
 */
function createDatabase(PDO $connection, $name, $type = 'mysql')
{
    try {
        $stmt = $connection->query('CREATE DATABASE '.$name);

        return ($stmt !== false);
    } catch (PDOException $e) { }

    return false;
}

/**
 * Drop the database with $name
 *
 * @param PDO $connection
 * @param string $name
 * @param string $type E.g. mysql, postgresql Currently unused
 * @return bool Whether the database was dropped or not
 */
function dropDatabase(PDO $connection, $name, $type = 'mysql')
{
    try {
        $stmt = $connection->query('DROP DATABASE IF EXISTS '.$name);

        return ($stmt !== false);
    } catch (PDOException $e) { }

    return false;
}

/**
 * Get a PDO connection
 *
 * @param string $host E.g. 127.0.0.1
 * @param string $username E.g. root
 * @param string $password E.g. secret
 * @param string $type E.g. mysql, postgresql
 * @return PDO
 */
function getDatabaseConnection($host, $username, $password, $type = 'mysql')
{
    switch ($type) {
        case 'mysql':
            return new PDO('mysql:host=' . $host, $username, $password);
        case 'postgresql':
            return new PDO('pgsql:host=' . $host, $username, $password);
    }

    throw new InvalidArgumentException('Type '.$type.' not implemented');
}
