<?php
/* (c) Sam Biggins (sam@evoluted.net)
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
namespace Deployer;

use Deployer\Exception\RuntimeException;
use Deployer\Task\Context;

use function Evoluted\Deployer\Db\getDatabaseConnection;
use function Evoluted\Deployer\Db\getDatabaseName as getDbName;
use function Evoluted\Deployer\Db\databaseExists as dbExists;
use function Evoluted\Deployer\Db\createDatabase as createDb;
use function Evoluted\Deployer\Db\dropDatabase as dropDb;
use function Evoluted\Deployer\VhostApi\notifyVhostApiManager;

require_once 'db.php';
require_once 'vhost_api.php';
require_once 'crontab.php';
require_once 'supervisord.php';

desc('Create if does not exist');
task('ci:database:setup', function () {
	getOrCreateDatabaseName();
})->onStage('cd');

desc('Creates the .env file setup for the environment, only runs for CD stage');
task('ci:setup:env', function () {
	writeln('<info>Writing .env file</info>');

	$db_name = getDatabaseName();
	$db_user = get('cd_database_user');
	$db_pass = get('cd_database_password');
	$db_host = get('cd_database_host');

	switch (get('application_type')) {
        case 'cakephp':
        case 'cakephp2':
		case 'cake70':
		case 'cake71':
		case 'cake72':
		case 'cake73':
		case 'cake56apache':
				$envContents = "DB_NAME='${db_name}'
DB_USER='${db_user}'
DB_PASS='${db_pass}'
DB_HOST='${db_host}'

ENVIRONMENT='PREVIEW'
BRANCH='" . getenv('CI_COMMIT_REF_SLUG') . "'

PREVIEW_URL='" . getenv('CI_ENVIRONMENT_URL') . "'
";

			// If the site is a multiapp with separate databases add prefix
			if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
				$applicationPrefix = getenv('APPLICATION_PREFIX');

				$envContents .= "\nAPPLICATION_PREFIX=" . $applicationPrefix;
			}

			break;
		case 'laravel70':
		case 'laravel71':
		case 'laravel72':
		case 'laravel73':
		case 'laravel74':
			// Build a list of env paths to try in priority order
			$envPaths = [
				get('release_path') . '/.env.cd.' . getenv('CI_COMMIT_REF_SLUG'),
				get('release_path') . '/.env.cd',
			];
			if (getenv('APPLICATION_PREFIX')) {
				// If there's an application prefix prioritise that
                $envPaths = array_merge([
                    get('release_path') . '/.env.cd.' . getenv('CI_COMMIT_REF_SLUG') . '-' . getenv('APPLICATION_PREFIX'),
                    get('release_path') . '/.env.cd-' . getenv('APPLICATION_PREFIX'),
                ], $envPaths);
			}

			// Search for an existing env path
			$envPath = current($envPaths);
			writeln('<info>Looking for deploy specific .env file at "' . $envPath . '"</info>');
			while (!file_exists($envPath)) {
				$envPath = next($envPaths);
				if (empty($envPath)) {
					throw new RuntimeException(
						Context::get()->getHost()->getHostname(),
						'ci:setup:env',
						null,
						'No env.cd file in the repository',
						null
					);
				}
				writeln('<info>Looking for deploy specific .env file at "' . $envPath . '"</info>');
			}

			$envContents = file_get_contents($envPath);

			$applicationDomain = getDomainName();

			// Replace database information
			$envContents = preg_replace("/DB_DATABASE=REPLACE/i", "DB_DATABASE='${db_name}'", $envContents);
			$envContents = preg_replace("/DB_HOST=REPLACE/i", "DB_HOST='${db_host}'", $envContents);
			$envContents = preg_replace("/DB_USERNAME=REPLACE/i", "DB_USERNAME='${db_user}'", $envContents);
			$envContents = preg_replace("/DB_PASSWORD=REPLACE/i", "DB_PASSWORD='${db_pass}'", $envContents);
			$envContents = preg_replace("/APPLICATION_DOMAIN=REPLACE/i", "APPLICATION_DOMAIN='" . $applicationDomain . "'", $envContents);
			// If the site is a multiapp with separate databases add to env file
			if (getenv('APPLICATION_PREFIX')) {
				$envContents = preg_replace("/APPLICATION_PREFIX=REPLACE/i", "APPLICATION_PREFIX='" . getenv('APPLICATION_PREFIX') . "'", $envContents);
			}
			break;
		case 'codeigniter-apache':
			// find / replace in database config file
			$path = get('release_path') . '/application/config/database.php.dist';

			writeln('<info>Looking for database config dist file at "' . $path . '"</info>');

			if (file_exists($path)) {
				$content = file_get_contents($path);
				$searchReplace = [
					'{__DB_HOST__}' => $db_host,
					'{__DB_USERNAME__}' => $db_user,
					'{__DB_PASSWORD__}' => $db_pass,
					'{__DB_DATABASE__}' => $db_name,
				];

				$newContent = escapeshellarg(str_replace(array_keys($searchReplace), array_values($searchReplace), $content));
				run('echo "' . base64_encode($newContent) . '" | base64 -d > ' . get('release_path') . '/application/config/database.php');
				writeln('<info>Wrote ' . get('release_path') . '/application/config/database.php file');
			} else {
				writeln('<error>Could not find config dist file</error>');
			}

			$envContents = '';
			break;
		case 'smarty':
			$envContents = "DB_NAME='${db_name}'
DB_USER='${db_user}'
DB_PASS='${db_pass}'
DB_HOST='${db_host}'

ENVIRONMENT='PREVIEW'
BRANCH='" . getenv('CI_COMMIT_REF_SLUG') . "'

PREVIEW_URL='" . getenv('CI_ENVIRONMENT_URL') . "'
";

			// If the site is a multiapp with separate databases add prefix
			if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
				$applicationPrefix = getenv('APPLICATION_PREFIX');

				$envContents .= "\nAPPLICATION_PREFIX=" . $applicationPrefix;
			}
			break;
		default:
			$envContents = '';
			break;
	}

	// Add .env file to deployment
	run('echo "' . base64_encode($envContents) . '" | base64 -d > ' . get('release_path') . "/.env");
})->onStage('cd');

desc('Rsync across assets');
task('ci:setup:assets', function () {});

desc('Configure Vhost and PHP-FPM');
task('ci:vhost:configure', function () {
	$hostname = getDomainName();
	// For legacy the application_type includes the PHP version and the type of application so splitting this out
    $applicationInformation = splitApplicationType(get('application_type'));

    notifyVhostApiManager($hostname, $applicationInformation['applicationType'], $applicationInformation['phpVersion'], 'php-fpm');
	reloadWebserver();
})->onStage('cd');

desc('Creates a database for the deployment');
task('ci:database:create', function () {
	$databaseName = getDatabaseName();
	if (createDatabase($databaseName)) {
		// If this is not the staging branch, then we can
		// use the staging database to seed our new one if it exists
		writeln('<info>looking to seed the database</info>');
		if ($seedDatabaseName = getSeedDatabaseName()) {
			// Don't want to seed from the same database
			if ($seedDatabaseName != getDatabaseName()) {
				writeln('<info>Seeding from ' . $seedDatabaseName . '</info>');
				run("mysqldump --add-drop-table -u " . get('cd_database_user') . " -h " . get('cd_database_host') . " -p" . get('cd_database_password') . " " . $seedDatabaseName . " | mysql -u " . get('cd_database_user') . " -h " . get('cd_database_host') . " -p" . get('cd_database_password') . " " . getDatabaseName());
			}
		} else {
			writeln('<error>Cant find seed database</error>');
		}
	} else {
		writeln('<error>Could not create database.</error>');
	}
});

desc('Removes the environemnt from the server, used by Gitlab CD to tidy up');
task('ci:destroy', function () {
	// Get the database name to remove
	$databaseName = getDatabaseName();
	$deployPath = get('deploy_path');

	if (!preg_match('#^/var/www/cd/#', $deployPath)) {
		// Just a safety catch incase deploy path is set to / by accident or something
		// we would like to limit our damage
		return false;
	}

	run("rm -rf $deployPath");

	writeln('<error>WARNING: About to remove databse: ' . $databaseName . " and folder directory " . $deployPath);
	dropDatabase($databaseName);

	writeln('<error>WARNING: Removing vhost</error>');
	$hostname = getDomainName();
	if (file_exists('/etc/nginx/conf.d/' . $hostname . '.conf')) {
		run('rm /etc/nginx/conf.d/' . $hostname . '.conf');
	}

	if (file_exists('/etc/apache2/sites-enabled/' . $hostname . '.conf')) {
		run('rm /etc/apache2/sites-enabled/' . $hostname . '.conf');
	}

	reloadWebserver();

});


/**
 * Gets the domain name for the deployment
 * @return string 		Domain name without protocol
 */
function getDomainName() {
	$hostname = preg_replace('#^https?://#', '', getenv('CI_ENVIRONMENT_URL'));

	// If it is a multi-site with separate DB's and files etc domain needs to be prefixed
	if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
		// Check that the prefix isn't already appended
		if (!preg_match('/' . getenv('APPLICATION_PREFIX') . '\./', $hostname)) {
			$hostname = getenv('APPLICATION_PREFIX') . '.' . $hostname;
		}
	}

	return $hostname;
}

/**
 * Forms the database name from environment variables, does not check for existance
 * @param  $branchName 	Allows overriding the branch name to find the db name for a different branch
 * @return string 		Database name
 */
function getDatabaseName($branchName = null) {
    $dbType = get('cd_database_type', 'mysql');
	$projectName = getenv('CI_PROJECT_NAME');

	if ($branchName === null) {
		$branchName = getenv('CI_COMMIT_REF_SLUG');
	}

	// If the site is a multiapp with separate databases add prefix
	$applicationPrefix = '';
	if (getenv('SEPARATE_APPLICATIONS') && getenv('APPLICATION_PREFIX')) {
		$applicationPrefix = getenv('APPLICATION_PREFIX');
	}

	return getDbName($projectName, $branchName, $applicationPrefix, 'cd_', $dbType);
}

/**
 * Returns the same of the seed database to copy initial data from
 */
function getSeedDatabaseName() {
	if (get('seed_database', false)) {
		$seedDatabase = get('seed_database');
	} else {
		// If we don't have a seed database explicitly set then we want
		// to use the staging database if there is one
		$seedDatabase = getDatabaseName('staging');
	}

	$type = get('cd_database_type', 'mysql');

	if ($type === 'mysql' && databaseExists($seedDatabase)) {
		return $seedDatabase;
	}

	// If seed database doesn't exist then we can't use it
	return false;
}

/**
 * Gets the database name if it exists otherwise it creates the database
 * @return string Database name
 */
function getOrCreateDatabaseName() {
	$databaseName = getDatabaseName();

	if (databaseExists($databaseName)) {
		writeln('<info>DATABASE does exist</info>');
	} else {
		writeln('<comment>Creating database: ' . $databaseName . '</comment>' );
		invoke('ci:database:create');
	}

	return $databaseName;
}

/**
 * Checks if a given database exists
 * @param  string $databaseName Name of database to check
 * @return boolean              True is the database exists
 */
function databaseExists($databaseName) {
    $host = get('cd_database_host');
    $username = get('cd_database_user');
    $password = get('cd_database_password');
    $type = get('cd_database_type', 'mysql');

    return dbExists(getDatabaseConnection($host, $username, $password, $type), $databaseName);
}

/**
 * Creates a database of a given name
 * @param  string $databaseName Name of database to create
 * @return boolean              True is the database has been created
 */
function createDatabase($databaseName) {
    $host = get('cd_database_host');
    $username = get('cd_database_user');
    $password = get('cd_database_password');
    $type = get('cd_database_type', 'mysql');

    return createDb(getDatabaseConnection($host, $username, $password, $type), $databaseName);
}

/**
 * Creates a database of a given name
 * @param  string $databaseName Name of database to create
 * @return null
 */
function dropDatabase($databaseName) {
    $backupPath = '/var/www/cd/backups/mysql/' . getenv('CI_PROJECT_NAME');
    run("mkdir -p $backupPath");

    $dumpLocation = $backupPath . '/' . getenv('CI_COMMIT_REF_SLUG') . '_' . $databaseName . '.sql';

    // Before we drop the database lets take a backup and file it away just incase we need to restore it
    dumpDatabase($databaseName, $dumpLocation);

    $host = get('cd_database_host');
    $username = get('cd_database_user');
    $password = get('cd_database_password');
    $type = get('cd_database_type', 'mysql');

    return dropDb(getDatabaseConnection($host, $username, $password, $type), $databaseName);
}

/**
 * Dump given database into given file location
 *
 * @param string $databaseName Database to dump/backup
 * @param string $dumpLocation File location to dump file to
 */
function dumpDatabase($databaseName, $dumpLocation) {
    $user = get('cd_database_user');
    $host = get('cd_database_host');
    $password = get('cd_database_password');
    $type = get('cd_database_type', 'mysql');

    switch ($type) {
        case 'mysql':
            run('mysqldump --add-drop-table -u ' . $user . ' -h ' . $host . ' -p' . $password . ' ' . $databaseName . ' > ' . $dumpLocation);
            break;
        case 'postgresql':
            run('PGPASSWORD="' . $password . '" pg_dump -h ' . $host . ' -U ' . $user . ' -OFc -f ' . $dumpLocation . ' ' . $databaseName);
            break;
    }
}

/**
 * Reloads the webserver, handles different applications webserver types
 */
function reloadWebserver() {
	// Based on the application type there may be different services to reload
	switch (get('application_type')) {
		case 'laravel70':
		case 'cake70':
			run('sudo service php7.0-fpm reload');
			break;
		case 'cake71':
		case 'laravel71':
			run('sudo service php7.1-fpm reload');
			break;
		case 'cake72':
		case 'laravel72':
			run('sudo service php7.2-fpm reload');
			break;
		case 'cake73':
		case 'laravel73':
			run('sudo service php7.3-fpm reload');
			break;
        case 'laravel74':
            run('sudo service php7.4-fpm reload');
            break;
		case 'cake56apache':
		case 'smarty':
			run('sudo service apache2 reload');
			break;
	}
}

/* HOOKS START */
// On deployment
after('deploy:symlink', 'crontab:update');
after('deploy:symlink', 'supervisor:configure');

// On environment destroy
after('ci:destroy', 'crontab:destroy');
after('ci:destroy', 'supervisor:destroy');
/*  HOOKS END  */

