<?php

namespace Evoluted\Deployer\VhostApi;

use function Deployer\has;
use function Deployer\get;
use function Deployer\writeln;

/**
 * Notify the Vhost API about a (possibly new) hostname
 *
 * @param string $hostname
 * @param string $applicationType E.g. "laravel", "cake"
 * @param string $phpVersion E.g. "7.1"
 * @param string $serverType Not currently used
 * @return array Whether the request to the vhost API was successful
 */
function notifyVhostApiManager($hostname, $applicationType, $phpVersion, $serverType) {

	$vhostApiManagerHost = 'http://vhost-api.cd.evoluted.net';
	if (has('vhost_api_host')) {
		$vhostApiManagerHost = get('vhost_api_host');
	}

	writeln('Using API host: ' . $vhostApiManagerHost);

	if (has('application/public_path')) {
		$applicationPublicPath = get('application/public_path');
	} else {
		$applicationPublicPath = getApplicationPublicPath($applicationType);
	}

	$bodyArray = [
		'template' => 'nginx',
		'server_name' => $hostname,
		'root' => get('release_path') . DIRECTORY_SEPARATOR . $applicationPublicPath,
		'server_type' => 'php-fpm',
		'php_version' => $phpVersion
	];

	// Merge in any custom params added by the applications deployer config.
	if (has('vhost_parameters')) {
		$bodyArray = array_merge($bodyArray, get('vhost_parameters'));
	}

	writeln('vHost parameters: ' . json_encode($bodyArray));

	$curl = curl_init();

	writeln('Sending deploy request to ' . $hostname);

	curl_setopt_array($curl, array(
		CURLOPT_URL => $vhostApiManagerHost . "/api/vhost/create",
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_ENCODING => "",
		CURLOPT_MAXREDIRS => 10,
		CURLOPT_TIMEOUT => 30,
		CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
		CURLOPT_CUSTOMREQUEST => "POST",
		CURLOPT_POSTFIELDS => json_encode($bodyArray),
		CURLOPT_HTTPHEADER => array(
			"Accept: application/json",
			"Authorization: Bearer " . getVhostManagerApiAccessToken(),
			"Content-Type: application/json",
		),
	));

	$response = curl_exec($curl);
	$err = curl_error($curl);

	curl_close($curl);

	writeln('Response: ' . $response);

	if ($err) {
		return [
			'success' => false,
			'errors' => 'cURL',
			'message' => "cURL Error #:" . $err,
		];
	} else {
		$jsonResponse = json_decode($response, true);

		if (!empty($jsonResponse['success']) && $jsonResponse['success'] == true) {
			return [
				'success' => true,
			];
		}

		return [
			'success' => false,
			'errors' => $jsonResponse['errors'] ?? 'Unidentified error',
			'message' => $jsonResponse['message'] ?? '',
		];
	}
}

/**
 * Returns the API manager access token, uses one set in deploy.php if it is with the key vhostmanagerapi/access_token
 * otherwise uses default one
 * @return string
 */
function getVhostManagerApiAccessToken() {
    if (has('vhostmanagerapi/access_token')) {
        return get('vhostmanagerapi/access_token');
    }

	return getenv('VHOST_API_ACCESS_TOKEN');
}

/**
 * Gets the public directory path for a given application type
 *
 * @param $applicationType
 * @return string The correct relative path to the application public directory e.g. for Cakephp2 app/webroot, Laravel
 *         public
 */
function getApplicationPublicPath($applicationType) {
    switch ($applicationType) {
        case 'cake':
        case 'cakephp':
        case 'cakephp2':
            return 'app/webroot';
        case 'cakephp3':
            return 'webroot';
        case 'craftcms':
            return 'web';
        case 'laravel':
        default:
            return 'public';
    }
}
