# Deployer recipes

Evoluted specific recipes for use with the [deployer/deployer](https://packagist.org/packages/deployer/deployer) package.

## Quickstart

Make sure that `packages.evoluted.net` is a repository in your `composer.json` file:

```json
"repositories": [
	{
		"type": "composer",
		"url": "https://packages.evoluted.net"
	}
],
```

Then run `composer require --dev evoluted/deployer-recipes` to include this repository.

Create a `deploy.php` in the root of your project with the default set up:

```php
<?php

namespace Deployer;

require 'vendor/evoluted/deployer-recipes/recipe/vanilla_laravel.php'; // Application type
require 'vendor/evoluted/deployer-recipes/recipe/common.php'; // Settings common to our hosting setup

set('php_version', '7.4');
set('repository', 'git@git.evoluted.net:evoluted/my-project.git');
set('application_name', 'my-project');
set('application_type', 'laravel74');

require 'vendor/evoluted/deployer-recipes/recipe/cd.php'; // CI/CD pipelines
require 'vendor/evoluted/deployer-recipes/recipe/evoluted-aws-uat.php'; // deploy to our AWS infrastructure for the UAT branch
require 'vendor/evoluted/deployer-recipes/recipe/evoluted-aws-production.php'; // deploy to our AWS infrastructure for production

```

This enables common defaults, CI/CD integration for testing and feature branches, the AWS for production deployment and tweaks to the standard Deployer Laravel process.

## Recipes

### Common

Default settings for our hosting environments.

### CI / CD

The CI/CD pipeline, you will need an appropriate `.gitlab-ci.yml` file to make use of this.

Sets up triggers to create a virtual host and database based on the branch. Will also handle crontab.template files and supervisord integration.

If you need to change the seed database that branch databases use:

```php
set('cd_db_seed', 'my_database');
```

If you need to change the type of database used (e.g. from MySQL to PostgreSQL):

```php
set('cd_db_type', 'pgsql');
```

Valid values are `mysql`, `pgsql`, and `mongodb`.

You can also set the version:
```php
set('cd_db_version', '5.7');
```

Set the environment as follows:

```php
set('cd_db_environment', 'cd');
```

### UAT

If you would like to deploy your `uat` branch to the AWS UAT server (accessible to clients):

```php
require 'vendor/evoluted/deployer-recipes/recipe/evoluted-aws-uat.php';
```

This will set up a host when the `CI_COMMIT_REF_SLUG` environment variable is `uat` and will overwrite whatever host is currently set for the `cd` deployer stage. More often than not this will be the CI / CD host from the `cd.php` recipe.

### AWS

Sets up the AWS hosts as the production target. Running `dep deploy production` will deploy to our AWS infrastructure.

If you have set up an application system user rather than using the default `deploy` user, set the `aws_user` variable e.g.:

```php
set('aws_user', 'lta_office_furniture')
```

### Application types

* `cakephp2.php`
* `craftcms.php`
* `pyrocms.php`
* `smarty.php`
* `vanilla_laravel.php`

## Other hosts and settings

The recipes above are meant to help with the most common setups for projects. They do not stop you setting up hosts or tasks as you would normally.

So if you need to deploy to web6 instead of AWS for production, **don't** include the `evoluted-aws-production.php` file and set your host up as normal in your `deploy.php` file:

```php
host('web6.core.evoluted.net')
    ->user('root')
    ->stage('production')
    ->set('branch', 'master')
    ->set('deploy_path', '...')
    ->set('keep_releases', 5)
;
```
If you find yourself using the same host or settings over and over, consider creating a recipe specifically for those settings.

## Sentry and Cron

You can create supervisor and cron templates that will be processed and added to the server automatically (note: not on production servers, may need manual intervention).

* crontab.staging.template
* supervisor-jobs.production.template

Our tasks look for the templates in this order, matching against:
* The gitlab commit's branch name (works on CD pipelines)
* The stage as set in the deployer host (manual deploys, not always needed)
    * `->stage('beta')`

If a match isn't found, it falls back to `crontab.template` or `supervisor-jobs.template`. If none of these files exist it won't set up cron jobs or supervisor programmes.

## Artifacts - Asset compilation
If you'd like to compile assets on the pipeline (css/js/etc)

1. Include the `artifact_deployment` recipe in your deploy file
2. Create a store stage in your pipeline e.g.

```yaml
store:
  stage: store
  image: node:16
  script:
    - yarn
    - yarn production
  artifacts:
    name: $CI_PROJECT_PATH_SLUG
    paths:
      - public/js/
      - public/css/
      - public/fonts/
      - public/images/
    expire_in: 1 week
```

3. Make sure you aren't committing assets to the repo as your store stage will overwrite them.

### Other servers
The `ci:setup:upload` job is set to only run on the 'cd' stage, but you can include the following to override the stage and set your own:

```php
Deployer::get()->tasks->get('ci:setup:upload')->onStage('cd', 'uat', 'production');
```
