<?php
App::uses('EvCybertillAppModel', 'EvCybertill.Model');

class CybertillTransaction extends EvCybertillAppModel {

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'CybertillTransactionStatus' => array(
			'className' => 'EvCybertill.CybertillTransactionStatus'
		),
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'model' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
			),
		),
		'model_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
			),
		),
		'data' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
			),
		),
		'cybertill_transaction_status_id' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
			),
			'numeric' => array(
				'rule' => array('numeric'),
			),
		),
	);

	protected $_Cybertill;

/**
 * Find an extended version of the Cybertill class and if one exists, use that.
 *
 * {@inheritDoc}
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		App::uses('EvCybertillCybertill', 'Lib');
		if (class_exists('EvCybertillCybertill')) {
			$this->_Cybertill = new EvCybertillCybertill();
		}

		if (empty($this->_Cybertill)) {
			App::uses('Cybertill', 'EvCybertill.Lib');
			$this->_Cybertill = new Cybertill();
		}
	}

/**
 * Create a cybertill transaction record and put a cybertill transaction into the queue.
 * @param  array $transactionData          The data to be used to create the cybertill transaction
 * @param  string $transactionModelName    The name of the model to associate the cybertill transaction with
 * @param  string $transactionModelIdPath  An array path to the id in the transaction data to assocaited the transaction with
 * @return bool/array                      False if the creation of the transaction was unsuccessful. The Cybertill transaction otherwise.
 */
	public function createTransaction($transactionData, $transactionModelName, $transactionModelIdPath) {
		$data = [
			'model' => $transactionModelName,
			'model_id' => Hash::get($transactionData, $transactionModelIdPath),
			'data' => json_encode($transactionData),
			'cybertill_transaction_status_id' => $this->CybertillTransactionStatus->statuses['queued']
		];

		$this->create();
		$transactionResult = $this->save($data);

		if ($transactionResult) {
			$this->createTransactionJob($transactionResult['CybertillTransaction']['id']);
		}

		return $transactionResult;
	}

/**
 * Get the data from the Cybertill transaction to put a transaction through in Cybertill. Update
 * the status of the transaction accordingly.
 * @param  int $id  The id of the Cybertill transaction to put through Cybertill.
 * @return bool     Whether the Cybertill transaction was successful or not.
 */
	public function makeOrder($id) {
		$data = $this->readForOrder($id);
		if (!empty($data)) {
			$this->_updateCybertillTransactionStatus($id, $this->CybertillTransactionStatus->statuses['pending']);

			$this->_updateCybertillTransactionAttempts($id, $data['CybertillTransaction']['attempts']);

			$result = $this->_Cybertill->makeOrder($data['CybertillTransaction']['data']);

			if ($result['success'] === true) {
				$this->_updateCybertillTransactionStatus($id, $this->CybertillTransactionStatus->statuses['successful']);
			} else {
				if (!empty($result['orderCreated']) && $result['orderCreated'] === true) {
					$this->_updateCybertillTransactionStatus($id, $this->CybertillTransactionStatus->statuses['notDispatched']);
				} else {
					$this->_updateCybertillTransactionStatus($id, $this->CybertillTransactionStatus->statuses['failed']);
				}

				if (!empty($result['message'])) {
					$this->_updateCybertillTransactionFailureMessage($id, $result['message']);
				}

				if (!empty($notifyAdmin) && !empty($result['message'])) {
					if (empty($result['adminNotification']) ||
						(
							Configure::check('EvCybertill.adminNotifications.' . $result['adminNotification']) &&
							Configure::read('EvCybertill.adminNotifications.' . $result['adminNotification']) === true
						)
					) {
						$this->_notifyAdminOfTransactionFailure($data, $result['message']);
					}
				}
			}

			return $result;
		}

		return false;
	}

	public function createTransactionJob($id) {
		$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');
		$jobResult = $QueuedTask->createJob(
			'CybertillOrder',
			$id
		);

		if ($jobResult) {
			$this->_updateCybertillTransactionStatus($id, $this->CybertillTransactionStatus->statuses['queued']);
		}

		return $jobResult;
	}

/**
 * Get the data from the Cybertill Transaction which will be used when putting a transaction through Cybertill.
 * @param  int $id The Id to put through Cybertill.
 * @return array   The Cybertill Transaction data.
 */
	public function readForOrder($id) {
		return $this->find(
			'first',
			[
				'conditions' => [
					'CybertillTransaction.id' => $id
				]
			]
		);
	}

	public function afterFind($results, $primary = false) {
		foreach ($results as $resultKey => $result) {
			$results[$resultKey]['CybertillTransaction']['data'] = json_decode($result['CybertillTransaction']['data'], true);

			if ($result['CybertillTransaction']['cybertill_transaction_status_id'] == $this->CybertillTransactionStatus->statuses['successful']) {
				$results[$resultKey]['CybertillTransaction']['is_successful'] = true;
			} else {
				$results[$resultKey]['CybertillTransaction']['is_successful'] = false;
			}

			if ($result['CybertillTransaction']['cybertill_transaction_status_id'] == $this->CybertillTransactionStatus->statuses['failed']) {
				$results[$resultKey]['CybertillTransaction']['is_queueable'] = true;
			} else {
				$results[$resultKey]['CybertillTransaction']['is_queueable'] = false;
			}
		}

		return parent::afterFind($results, $primary);
	}

/**
 * Update the ID of a Cybertill transaction by a matching model/model ID
 * @param  string $model	The model for the related model entry
 * @param  int $modelId 	The Id for the related model entry
 * @param  int $cybertillTransactionId 	The Id created and  returned by Cybertill themselves
 * @return bool/array 		False if an error occured, result of cybertill update call otherwise
 */
	public function updateCybertillTransactionIdByOrderId($model, $modelId, $cybertillTransactionId) {
		$transaction = $this->find('first', [
			'conditions' => [
				'model' => $model,
				'model_id' => $modelId
			],
			'order' => [
				'created' => 'desc'
			]
		]);

		if (empty($transaction)) {
			return false;
		}

		return $this->_updateCybertillTransactionId($transaction[$this->alias]['id'], $cybertillTransactionId);
	}

/**
 * Update the ID of a Cybertill transaction.
 * @param  int $id       				The Id of the Cybertill transaction to update (index in our database).
 * @param  int $cybertillTransactionId 	The Id created and  returned by Cybertill themselves.
 * @return bool/array    				Where the udpate was successful or not.
 */
	protected function _updateCybertillTransactionId($id, $cybertillTransactionId) {
		$this->create();
		$this->id = $id;
		return $this->saveField('cybertill_transaction_id', $cybertillTransactionId);
	}

/**
 * Update the status of a Cybertill transaction.
 * @param  int $id       The Id of the Cybertill transaction to update.
 * @param  int $statusId The status Id to update to.
 * @return bool/array    Where the udpate was successful or not.
 */
	protected function _updateCybertillTransactionStatus($id, $statusId) {
		$this->create();
		$this->id = $id;
		return $this->saveField('cybertill_transaction_status_id', $statusId);
	}

/**
 * Increment the attempts of a Cybertill Transaction and make the last attempted time now.
 * @param  int  $id              The id of the Cybertill Transaction to update.
 * @param  int  $currentAttempts The amount of attempts the Cybertill Transaction currently has.
 * @return bool                  Whether the update was successful or not.
 */
	protected function _updateCybertillTransactionAttempts($id, $currentAttempts) {
		$this->create();
		$this->id = $id;

		if ($currentAttempts == null) {
			$currentAttempts = 0;
		}

		$resultAttempts = $this->saveField('attempts', ++$currentAttempts);
		$resultLastAttempted = $this->saveField('last_attempted', date('Y-m-d H:i:s'));

		if ($resultAttempts && $resultLastAttempted) {
			return true;
		}

		return false;
	}

/**
 * Update the failure message of the Cybertill transaction to record the reason why the last attempt failed.
 * @param  int       $id      The id of the Cybertill transaction to update.
 * @param  string    $message The message to update to.
 * @return boo/array          Whether the update was successful or not.
 */
	protected function _updateCybertillTransactionFailureMessage($id, $message) {
		$this->create();
		$this->id = $id;

		$resultFailureMessage = $this->saveField('failure_message', $message);

		if ($resultFailureMessage) {
			return true;
		}

		return false;
	}

	protected function _notifyAdminOfTransactionFailure($data, $message) {
		if (!empty(Configure::read('SiteSetting.general.admin_email')) && Configure::read('SiteSetting.general.site_title')) {
			$QueuedTask = EvClassRegistry::init('Queue.QueuedTask');

			$QueuedTask->createJob(
				'CustomEmail',
				array(
					'from' => array(
						Configure::read('SiteSetting.general.admin_email') => Configure::read('SiteSetting.general.site_title')
					),
					'to' => array(
						Configure::read('SiteSetting.general.admin_email') => 'Admin'
					),
					'subject' => __('Cybertill Transaction Failure #%s', array($data['CybertillTransaction']['model_id'])),
					'viewVars' => array(
						'cybertillTransaction' => $data,
						'failureMessage' => $message,
					),
					'helpers' => [
						'Html',
						'Number'
					],
					'template' => 'EvCybertill.CybertillTransactions/adminFailureNotification'
				)
			);
		}
	}

/**
 * An extension of the Order based readForView method which appends the
 * original unit price to each order item. This is required for order items
 * containing bulk pricing items as the clients wants to see the original
 * unit price rather than the discounted rate associated with each order item
 *
 * @param int $id The unique id of the order
 * @return array Contains the data relating to the requested order id
 */
	public function readForCybertillTransaction($id) {
		$Order = EvClassRegistry::init('EvCheckout.Order');
		$orderData = $Order->readForView($id);

		if (! empty($orderData['OrderData'])) {
			$orderOrderData = Hash::combine($orderData['OrderData'], '{n}.name', '{n}');

			if (
				isset($orderOrderData['original_price']) &&
				! empty($orderOrderData['original_price']['data'])
			) {
				$originalPrice = json_decode($orderOrderData['original_price']['data'], true);

				// apply the original price to the order item array
				foreach ($orderData['OrderItem'] as &$orderItemData) {
					// when available, assign the original unit price (inc tax) for each
					// order item in the order. this value is passed to cybertill when
					// an order is placed using the bulk pricing functionality
					if (
						isset($originalPrice[$orderItemData['model'] . '.' . $orderItemData['model_id']])
					) {
						$originalPriceData = $originalPrice[$orderItemData['model'] . '.' . $orderItemData['model_id']];
						$originalPriceExcTax = $originalPriceData['unit_price'];
						$originalPriceIncTax = $originalPriceData['unit_price_incTax'];
						$orderItemData['original_unit_price_exc_tax'] = $originalPriceExcTax;
						$orderItemData['original_unit_price_inc_tax'] = $originalPriceIncTax;
					}
				}
			}
		}

		return $orderData;
	}

}
