<?php

/**
 * A library that uses the cybertill api to make orders and sync stock
 * SOAP API docs: http://ct28790.c-pos.co.uk/current/docs/soap_documentation_v1_5/index.html
 */

class Cybertill {

	//The current loaded config (live/dev)
	protected $_config = null;

	//The current Soap Client
	protected $_client = null;

	//The current authorisation id returned from the client
	protected $_authId = null;

	//The current customer id, taken from config if avilable created from customer details
	protected $_customer = null;

	//The current order being used to make a cybertill order.
	protected $_order = [];

	// cybertill item cache (prevent repeated calls to _getItem)
	protected $_itemCache = [];

/**
 * Cybertill gift card balance statuses.
 *
 * The statuses are taken out of the CMS directly, they are not documented. To view the statuses, create or edit a gift
 * card in Cybertill and take the values from the dropdown.
 */
	const GIFT_CARD_STATUSES = [
		'created' => 0,
		'expired' => 1,
		'fully_redeemed' => 2,
		'issued' => 3,
		'refunded' => 5,
	];

/**
 * Check the balance of a gift card in Cybertill.
 *
 * @param string $code The code of the gift card to check.
 * @return array Formatted with api call success and gift card balance.
 */
	public function checkGiftCardBalance($code) {
		if (!$this->_setupClient()) {
			return [
				'success' => false,
				'balance' => 0,
			];
		}

		$balance = $this->_client->gift_card_check_balance($code);

		//Check gift card has the issued status
		if (!empty($balance->status) && $balance->status != self::GIFT_CARD_STATUSES['issued']) {
			return [
				'success' => false,
				'balance' => 0,
			];
		}

		if (!empty($balance->error)) {
			//There was an error getting the balance, return empty.
			CakeLog::write('cybertillApi', print_r($balance->error, true));
			return [
				'success' => false,
				'balance' => 0,
			];
		}

		if (!empty($balance->expiryDate)) {
			//Check that the expiry date is still valid
			$expiryDate = DateTime::createFromFormat('d-m-Y', $balance->expiryDate);
			$nowDate = new DateTime();

			if ($nowDate >= $expiryDate) {
				return [
					'success' => false,
					'balance' => 0,
				];
			}
		}

		if (empty($balance->currentBalance)) {
			return [
				'success' => false,
				'balance' => 0,
			];
		}

		//Current balance is formatted with currency so strip that out
		$currentBalance = floatval(preg_replace('/[^\d\.]+/', '', $balance->currentBalance));

		if ($currentBalance <= 0) {
			return [
				'success' => false,
				'balance' => 0,
			];
		}

		return [
			'success' => true,
			'balance' => $currentBalance,
		];
	}

/**
 * Make an order on the client's cybertill system.
 *
 * @param array $order An EvCheckout order array consisting of order info, order items, order totals and order data.
 * @return array|bool Returns transaction array. Contains success/failure and transaction info/errors.
 */
	public function makeOrder($order) {
		if (!$this->_setupClient()) {
			return [
				'success' => false,
				'message' => 'Unable to establish client',
				'adminNotification' => 'clientSetupFailure'
			];
		}

		$this->_setupCustomer();

		if (empty($this->_customer)) {
			return [
				'success' => false,
				'message' => 'Customer details missing',
				'adminNotification' => 'customerSetupFailure'
			];
		}

		$orderFailureMessage = '';

		$this->_setupOrder($order);

		$cybertillOrderItems = $this->_getItemsForOrder($order);

		if ($cybertillOrderItems['success'] === false) {
			$orderFailureMessage .= "\n" . $cybertillOrderItems['message'];
		} else {
			$this->_orderItems = $cybertillOrderItems['data'];
		}

		$cybertillOrderPayments = $this->_getPaymentsForOrder($order);

		if ($cybertillOrderPayments['success'] === false) {
			$orderFailureMessage .= "\n" . $cybertillOrderPayments['message'];
		} else {
			$this->_orderPayments = $cybertillOrderPayments['data'];
		}

		$cybertillOrderDelivery = $this->_getDeliveryForOrder($order);

		if ($cybertillOrderDelivery['success'] === false) {
			$orderFailureMessage .= "\n" . $cybertillOrderDelivery['message'];
		} else {
			$this->_orderDelivery = $cybertillOrderDelivery['data'];
		}

		$cybertillOrderDetails = $this->_getDetailsForOrder($order);

		if ($cybertillOrderDetails['success'] === false) {
			$orderFailureMessage .= "\n" . $cybertillOrderDetails['message'];
		} else {
			$this->_orderDetails = $cybertillOrderDetails['data'];
		}

		if (!empty($orderFailureMessage)) {
			if (
				Configure::check('EvCybertill.errorLogging.transactionFailure')
				&& Configure::read('EvCybertill.errorLogging.transactionFailure') === true
			) {
				CakeLog::write('cybertillApi', print_r($orderFailureMessage, true));
				CakeLog::write(
					'cybertillApi',
					"\n Transaction Data Sent: \n" .
					print_r($cybertillOrderDetails['data'], true) . "\n" .
					print_r($cybertillOrderItems['data'], true) . "\n" .
					print_r($cybertillOrderDelivery['data'], true) . "\n" .
					print_r($cybertillOrderPayments['data'], true)
				);
			}

			return [
				'success' => false,
				'message' => $orderFailureMessage,
				'adminNotification' => 'orderSetupFailure'
			];
		}

		$transaction = $this->_client->transaction_add_giftcard(
			$cybertillOrderDetails['data'],
			$cybertillOrderItems['data'],
			$cybertillOrderDelivery['data'],
			$cybertillOrderPayments['data']
		);

		if (!$transaction->success) {
			if (
				Configure::check('EvCybertill.errorLogging.transactionFailure')
				&& Configure::read('EvCybertill.errorLogging.transactionFailure') === true
			) {
				CakeLog::write('cybertillApi', print_r($transaction, true));
				CakeLog::write(
					'cybertillApi',
					"\n Transaction Data Sent: \n" .
					print_r($cybertillOrderDetails['data'], true) . "\n" .
					print_r($cybertillOrderItems['data'], true) . "\n" .
					print_r($cybertillOrderDelivery['data'], true) . "\n" .
					print_r($cybertillOrderPayments['data'], true)
				);
			}

			return [
				'success' => false,
				'message' => $transaction->errors->item->error,
				'adminNotification' => 'transactionFailure'
			];
		}

		if (empty($this->_config['despatch_user_id'])) {
			return [
				'success' => false,
				'message' => 'Dispatch user missing from config.',
				'orderCreated' => true,
				'adminNotification' => 'dispatchFailure'
			];
		}

		$models = $this->_config['models'];
		$orderModel = $models['order_model'];

		$CybertillTransaction = EvClassRegistry::init('EvCybertill.CybertillTransaction');
		$transactionUpdated = $CybertillTransaction->updateCybertillTransactionIdByOrderId(
			$orderModel,
			$this->_order['Order']['id'],
			$transaction->transaction->id
		);

		if ($transactionUpdated === false) {
			if (
				Configure::check('EvCybertill.errorLogging.transactionFailure')
				&& Configure::read('EvCybertill.errorLogging.transactionFailure') === true
			) {
				CakeLog::write(
					'cybertillApi',
					"\n Transaction Lookup Data: \n" .
					"Model: " . $orderModel . "\n" .
					"Model ID: " . $this->_order['Order']['id'] . "\n" .
					"Cybertill Transaction ID (remote): " . $transaction->transaction->id
				);
			}

			return [
				'success' => false,
				'message' => 'No matching transaction found.',
				'orderCreated' => true,
				'adminNotification' => 'transactionFailure'
			];
		}

		$dispatchOrder = [
			[
				'transactionId' => $transaction->transaction->id,
				'locationId' => $this->_config['location_id'],
				'consignmentRef' => $this->_config['order_details']['order_note'],
				'userId' => null
			]
		];

		$dispatched = $this->_client->transaction_despatch_orders(
			$this->_config['despatch_user_id'],
			$dispatchOrder
		);

		if ($dispatched->success) {
			return ['success' => true];
		}

		$dispatchedError = $dispatched->errors->item->error;

		if (
			Configure::check('EvCybertill.errorLogging.dispatchFailure')
			&& Configure::read('EvCybertill.errorLogging.dispatchFailure') === true
		) {
			if (CakePlugin::loaded('EvErrbit')) {
				$e = new Exception(print_r($dispatchedError, true));
				Errbit::notify($e);
			}

			CakeLog::write('cybertillApi', print_r($dispatchedError, true));
			CakeLog::write(
				'cybertillApi',
				"\n Transaction Data Sent: \n" .
				print_r($cybertillOrderDetails['data'], true) . "\n" .
				print_r($cybertillOrderItems['data'], true) . "\n" .
				print_r($cybertillOrderDelivery['data'], true) . "\n" .
				print_r($cybertillOrderPayments['data'], true)
			);
		}

		return [
			'success' => false,
			'message' => $dispatchedError,
			'orderCreated' => true,
			'adminNotification' => 'dispatchFailure'
		];
	}

/**
 * Pull information from Cybertill and update the local site
 *
 * Does not update Cybertill.
 *
 * @return bool If the update is successful and item(s) have been updated
 */
	public function pull() {
		if (!$this->_setupClient()) {
			return false;
		}

		if (empty($this->_config['models'])) {
			return false;
		}

		$models = $this->_config['models'];

		if (empty($models['item_model']) || empty($models['stock_model']) || empty($models['pricing_model'])) {
			return false;
		}

		$ItemModel = EvClassRegistry::init($models['item_model']);

		if ($ItemModel === false) {
			return false;
		}

		$batchLimit = Configure::read('EvCybertill.queries.pullBatchItemLimit');
		$batchLimit = !empty($batchLimit) ? $batchLimit : 500;
		$batchParameters = [
			'limit' => $batchLimit,
			'offset' => 0,
		];

		$items = $this->_getItemsToPull($ItemModel, $batchParameters);

		while (!empty($items)) {
			foreach ($items as $item) {
				if (!empty($item[$ItemModel->alias]['sku'])) {
					$this->pullItem($item[$ItemModel->alias]['sku'], $item);
				}
			}

			$batchParameters['offset'] += $batchLimit;

			$items = $this->_getItemsToPull($ItemModel, $batchParameters);
		}

		return true;
	}

/**
 * syncStock is deprecated since 2.3.1.0 and will be removed in 2.4.0.0, use pull()
 *
 * @return bool.
 * @deprecated Since 2.3.1.0
 */
	public function syncStock() {
		trigger_error(
			'syncStock is deprecated since 2.3.1.0 and will be removed in 2.4.0.0, use pull()',
			E_USER_DEPRECATED
		);

		return $this->pull();
	}

/**
 * Pull the stock and pricing of a single item from Cybertill to the local website
 *
 * Does not update Cybertill.
 *
 * @param string $sku The SKU of the item in Cybertill to get stock from.
 * @param array $item The item data from the website.
 * @return void.
 */
	public function pullItem($sku, $item) {
		if (!$this->_setupClient()) {
			return;
		}

		$cybertillItem = $this->_getItem($sku);

		if (!empty($cybertillItem['item']->productOptionPrice->priceStore)) {
			$this->_updatePrice($cybertillItem['item']->productOptionPrice, $item);
		}

		if (!empty($cybertillItem['item']->productOption->id)) {
			$stock = $this->_getItemStock($sku, $cybertillItem['item']->productOption->id);

			if (!empty($stock)) {
				$this->_updateStock($stock, $item);
			}
		}
	}

/**
 * syncItemStock() is deprecated since 2.3.1.0 and will be removed in 2.4.0.0, use pullItem()
 *
 * @param string $sku The SKU of the item in Cybertill to get stock from.
 * @param array $item The item data from the website.
 * @return void.
 * @deprecated Since 2.3.1.0
 */
	public function syncItemStock($sku, $item) {
		trigger_error(
			'syncItemStock() is deprecated since 2.3.1.0 and will be removed in 2.4.0.0, use pullItem()',
			E_USER_DEPRECATED
		);

		$this->pullItem($sku, $item);
	}

/**
 * Check the stock of a particular item by using the SKU provided.
 *
 * @param string $sku The SKU to be used to find the item to return the stock of
 * @return int|bool If an item and stock are found then stock will be returned otherwise false is returned
 */
	public function checkStock($sku) {
		if (empty($this->_client)) {
			$this->_setupClient();
		}

		if (!empty($this->_client)) {
			$productItem = $this->_getItem($sku);

			if ($productItem['success'] === true) {
				return ['success' => true, 'stock' => $this->_getItemStock($sku, $productItem['item']->productOption->id)];
			} else {
				return ['success' => false, 'message' => $productItem['message']];
			}
		}

		return ['success' => false, 'message' => 'unable to establish client'];
	}

/**
 * Check the Cybertill stored price(s) of a particular item using the provided SKU
 *
 * @param string $sku The product SKU to search for pricing for
 * @return array With a 'success' key and either a 'pricing' key if successful, 'message' if unsuccessful
 */
	public function checkPricing($sku) {
		if (empty($this->_client)) {
			$this->_setupClient();
		}

		if (!empty($this->_client)) {
			$productItem = $this->_getItem($sku);

			return [ 'success' => true, 'pricing' => $productItem['item']->productOptionPrice ];
		}

		return [ 'success' => false, 'message' => 'unable to establish client' ];
	}

/**
 * Set the stock of an item by using the SKU provided.
 *
 * @param string $sku         The SKU of the cybertill product to set the stock on.
 * @param int    $stockAmount The quantity of stock to set the product to have.
 * @return void.
 */
	public function setStock($sku, $stockAmount) {
		if ($this->_setupClient()) {
			$productItem = $this->_getItem($sku);

			if ($productItem['success'] === true) {
				$newStock = [
					[
						'itemId' => $productItem['item']->productOption->id,
						'locationId' => $this->_config['location_id'],
						'reasonText' => 'Evoluted Add Stock',
						'qty' => $stockAmount,
						'updateType' => 'set'
					]
				];

				$this->_client->stock_update($newStock);
			}
		}
	}

/**
 * Create the soap client to create orders or sync stock. Uses settings from the config, if a requried
 * config setting is missing then the connection won't be made.
 *
 * @return bool True if client is setup, false otherwise.
 */
	protected function _setupClient() {
		if ($this->_client !== null) {
			return true;
		}

		$this->_getConfig();

		if (!empty($this->_config['client_details'])) {
			$clientDetails = $this->_config['client_details'];
			$this->_client = new SoapClient($clientDetails['wsdl']);

			if (!empty($clientDetails['url']) && !empty($clientDetails['user_id'])) {
				$this->_authId = $this->_client->authenticate_get($clientDetails['url'], $clientDetails['user_id']);

				$this->_client = new SoapClient($clientDetails['wsdl'], ['login' => $this->_authId]);

				return true;
			}
		}

		if (Configure::check('EvCybertill.errorLogging.clientSetupFailure') && Configure::read('EvCybertill.errorLogging.clientSetupFailure') === true) {
			// A client hasn't been able to be made
			if (CakePlugin::loaded('EvErrbit')) {
				$e = new Exception("A client wasn't able to be established.");
				Errbit::notify($e);
			}
		}

		return false;
	}

/**
 * Read the config settings based on the current app environment. Allows for different testing settings if available.
 *
 * @return void.
 */
	protected function _getConfig() {
		if (Configure::read('app.environment') == 'PRODUCTION') {
			$this->_config = Configure::read('EvCybertill.live');
		} else {
			$this->_config = Configure::read('EvCybertill.dev');
		}
	}

/**
 * Attempt to get a customer using the provided customer id in the config. If an id is missing in the config
 * then attempt to create a customer using the customer details in the config. If a customer is found or
 * added then save the details.
 *
 * @return void.
 */
	protected function _setupCustomer() {
		if (!empty($this->_config['customer_id'])) {
			$this->_customer = $this->_client->customer_get($this->_config['customer_id']);
		} else {
			if (!empty($this->_config['customer_details'])) {
				$this->_customer = $this->_client->customer_add($this->_config['customer_details']);
			}
		}
	}

/**
 * Setup an order ready to be processed and sent to Cybertill.
 * Clear out an existing instance variables used when processing an order.
 *
 * @return void.
 */
	protected function _setupOrder($order) {
		$this->_order = $order;
		$this->_orderItems = [];
		$this->_orderPayments = [];
		$this->_orderDelivery = [];
		$this->_orderDetails = [];
		$this->_orderTotals = [];
	}

/**
 * Get all the Cybertill items from the provided order. Each item is checked if it exists and is available before being
 * added to the returned array. If an item is missing then the order items aren't returned and the Cybertill.
 * transaction won't take place.
 *
 * @param array $order An EvCheckout order.
 * @return array The Cybertill order items or null if an item can't be found.
 */
	protected function _getItemsForOrder($order) {
		$paths = $this->_config['paths'];

		if (empty($paths['items']) || empty($paths['sku'])) {
			return [
				'success' => false,
				'message' => 'Missing item or sku path in config'
			];
		}

		$items = Hash::get($this->_order, $paths['items']);

		$itemErrorMessage = '';

		foreach ($items as $item) {
			$orderItem = $this->_getItemForOrder($item, $paths['sku']);
			if ($orderItem['success'] === true) {
				$orderItems[] = $orderItem['item'];
			} else {
				$itemErrorMessage .= "\n" . $orderItem['message'];
			}
		}

		if (!empty($orderItems) && count($orderItems) == count($items)) {
			//All items added successfully. Check if any discount needs to be added
			$deliveryCybertillOrderItem = $this->_getDeliveryItemForOrder($order);

			if (!empty($this->_config['order_payments']['discount_label'])) {
				$discountAmount = $this->_getOrderTotalForOrder($this->_order, $this->_config['order_payments']['discount_label']);

				if (!empty($discountAmount)) {
					$orderItems = $this->_includeDiscountOnItems($orderItems, $discountAmount);

					$totalDiscountedPence = 0;
					foreach ($orderItems as $orderItem) {
						if (empty($orderItem['discountPrice'])) {
							continue;
						}

						$totalDiscountedPence += (($orderItem['itemPrice'] - $orderItem['discountPrice']) * $orderItem['salesQty']) * 100;
					}

					$discountAmountPence = abs($discountAmount) * 100;

					if ($totalDiscountedPence != $discountAmountPence) {
						$remainingDiscountPence = $discountAmountPence - $totalDiscountedPence;

						$remainingDiscount = 0;
						if ($remainingDiscountPence > 0) {
							$remainingDiscount = $remainingDiscountPence / 100;
						}

						$deliveryCybertillOrderItem['item']['itemPrice'] += $remainingDiscount;
					}
				}
			}

			if ($deliveryCybertillOrderItem['success'] === true) {
				$orderItems[] = $deliveryCybertillOrderItem['item'];
			} else {
				return [
					'success' => false,
					'message' => $deliveryCybertillOrderItem['message'],
				];
			}

			return [
				'success' => true,
				'data' => $orderItems,
			];
		}

		return [
			'success' => false,
			'message' => $itemErrorMessage,
		];
	}

/**
 * Get an item for a Cybertill order. Check if the item has stock available before trying to order it.
 * An Cybertill order item array is constructed if the item is found and has stock and returned to add
 * to the order items array.
 *
 * @param array  $orderItem An item provided by EvCheckout to add to the Cybertill order
 * @param string $skuPath   The path within the orderItem array where the sku is stored.
 * @return array The Cybertill order item or null if no item was found.
 */
	protected function _getItemForOrder($orderItem, $skuPath) {
		// Get the product the client wants us to add, we need this for the product ID.
		$sku = Hash::get($orderItem, $skuPath);

		// Check the item is in stock before we try and add it to an order.
		$stock = $this->checkStock($sku);

		$stockMessage = '';
		if ($stock['success'] === false) {
			$stockMessage = $stock['message'];
		}

		$productItem = $this->_getItem($sku);

		if ($productItem['success'] !== true) {
			return [
				'success' => false,
				'message' => $productItem['message'] . ' ' . $stockMessage
			];
		}

		$returnData = [
			'success' => true,
			'item' => [
				'itemId' => $productItem['item']->productOption->id,
				'salesQty' => $orderItem['quantity'],
				'note' => '',
				'vatRate' => null, // Item price passed inc tax
			]
		];

		// when available, use the original unit price. this will
		// be available in the array when a bulk discount has been used
		if (isset($orderItem['original_unit_price_inc_tax'])) {
			$returnData['item']['itemPrice'] = $orderItem['original_unit_price_inc_tax'];
			$returnData['item']['discountPrice'] = $orderItem['unit_price_inc_tax'];
		} else {
			$returnData['item']['itemPrice'] = $orderItem['unit_price_inc_tax'];
		}

		return $returnData;
	}

/**
 * Get an item from Cybertill using the SKU provided.
 *
 * @param string $sku The SKU of the item to get from Cybertill
 * @return object The Cybertill item or null if no item was found
 */
	protected function _getItem($sku) {
		if (array_key_exists($sku, $this->_itemCache)) {
			return $this->_itemCache[$sku];
		}

		try {
			$item = ['success' => true, 'item' => $this->_client->item_get(null, $sku, null, true)];
			if (count($this->_itemCache) >= 10) {
				array_pop($this->_itemCache);
			}

			$this->_itemCache[$sku] = $item;

			return $item;
		} catch (Exception $e) {
			if (Configure::check('EvCybertill.errorLogging.stockNotFound') && Configure::read('EvCybertill.errorLogging.stockNotFound') === true) {
				if (CakePlugin::loaded('EvErrbit')) {
					$e = new Exception("no product exists with sku: " . $sku);
					Errbit::notify($e);
				}

				CakeLog::write('cybertillApi', "no product exists with sku: " . $sku);
				CakeLog::write('missingProduct', $sku);
			}

			return ['success' => false, 'message' => 'no product exists with sku: ' . $sku];
		}
	}

/**
 * Get the stock for a specific Cybertill item. Items are found via their SKU's.
 * It is possible that if the item has been discontinued that an item would be found but no stock
 * exists for it. In this case then an item won't be returned.
 *
 * @param string $sku    The SKU of the item to get stock of
 * @param int    $itemId The Cybertill if of the item to get stock of
 * @return int The current stock in Cybertill of the item to get stock of
 */
	protected function _getItemStock($sku, $itemId) {
		$stock = null;
		if (!empty($itemId)) {
			try {
				$stock = $this->_client->stock_item($itemId, $this->_config['location_id']);
			} catch (Exception $e) {
				if (Configure::check('EvCybertill.errorLogging.stockNotFound') && Configure::read('EvCybertill.errorLogging.stockNotFound') === true) {
					if (CakePlugin::loaded('EvErrbit')) {
						$e = new Exception("no stock found for: " . $sku);
						Errbit::notify($e);
					}

					CakeLog::write('cybertillApi', "no stock found for: " . $sku);
					CakeLog::write('missingStock', $sku);
				}
			}
		}
		return $stock;
	}

/**
 * Include an discount from the current order on each item in the order. Discounts can't be added in full so need to be
 * split over each item. The discount is split proportionately across each item based on their value and quantity.
 *
 * @param array     $orderItems The items being added to the Cybertill order.
 * @param int|float $discount   The amount being discounted on the order.
 * @return array Discounted $orderItems.
 */
	protected function _includeDiscountOnItems($orderItems, $discount) {
		$discount = abs($discount);

		//Calculate sub total and quantity for the order.
		$orderSubTotal = 0;

		foreach ($orderItems as $orderItem) {
			$orderSubTotal += $orderItem['itemPrice'] * $orderItem['salesQty'];
		}

		if (empty($orderSubTotal)) {
			return $orderItems;
		}

		$totalItems = count($orderItems);
		$currentItem = 1;
		$currentAmountDiscounted = 0.00;

		//Split the discount
		foreach ($orderItems as &$orderItem) {
			if ($currentItem === $totalItems) {
				//On the last item so add or remove additional discount to round it off
				$itemDiscount = $discount - $currentAmountDiscounted;
				$unitDiscount = $itemDiscount / $orderItem['salesQty'];

				$unitDiscount = floor($unitDiscount * 100) / 100;
			} else {
				//Calculate proportion of discount on this item
				$rowTotal = $orderItem['itemPrice'] * $orderItem['salesQty'];

				$itemDiscountProportion = $rowTotal / $orderSubTotal;

				$itemDiscount = $discount * $itemDiscountProportion;

				$unitDiscount = $itemDiscount / $orderItem['salesQty'];

				$unitDiscount = floor($unitDiscount * 100) / 100;

				//Get the total row discount now that it is rounded.
				$currentAmountDiscounted += $unitDiscount * $orderItem['salesQty'];
			}

			//Add the discounted amount to the item, if it has already been discounted then apply additional discount.
			if (!isset($orderItem['discountPrice'])) {
				$orderItem['discountPrice'] = $orderItem['itemPrice'] - $unitDiscount;
			} else {
				$orderItem['discountPrice'] -= $unitDiscount;
			}

			$currentItem++;
		}

		return $orderItems;
	}

/**
 * Get the order details for a Cybertill order. The details are mainly taken from the config but the order total
 * is taken from the order created on the website.
 *
 * @param array $order An array containing the order to get the order total from
 * @return array The array containing the order details
 */
	protected function _getDetailsForOrder($order) {
		$orderDetails = $this->_config['order_details'];

		if (!empty($orderDetails)) {
			$orderTotal = 0;
			if (!empty($this->_orderPayments)) {
				foreach ($this->_orderPayments as $orderPayment) {
					$orderTotal += $orderPayment['total'];
				}
			} elseif (!empty($orderDetails['order_total_label'])) {
				//Get the order total from the order
				$orderDetails['order_total'] = $this->_getOrderTotalForOrder($order, $orderDetails['order_total_label']);

				if (!empty($orderDetails['order_total'])) {
					$orderTotal = $orderDetails['order_total'];
				}
			}

			return [
				'success' => true,
				'data' => [
					'websiteId' => $orderDetails['website_id'],
					'customerId' => $this->_customer->customer->id,
					'locationId' => $this->_config['location_id'],
					'orderTotal' => $orderTotal,
					'orderNote' => $orderDetails['order_note'],
					'customerOrderRef' => $orderDetails['customer_order_reference'],
				]
			];
		}
	}

/**
 * Get a total of an order to be used in the payment details and order details of the Cybertill
 * order. The specific total to get is based on the label passed through which itself is in the config.
 *
 * @param array  $order An array containing the order to get the order total from.
 * @param string $label The label of the total to return.
 * @return float The amount of total. If a tax inclusive price is available then it will return that.
 */
	protected function _getOrderTotalForOrder($order, $label) {
		if (!empty($this->_orderTotals[$label])) {
			return $this->_orderTotals[$label];
		}

		$orderTotal = null;

		if (!empty($this->_config['paths']['totals'])) {
			$totals = Hash::get($order, $this->_config['paths']['totals']);

			foreach ($totals as $total) {
				if ($total['name'] == $label) {
					if (!empty((float)$total['display_inc_tax'])) {
						$orderTotal = $total['display_inc_tax'];
					} else {
						$orderTotal = $total['amount'];
					}
				}
			}
		}

		$this->_orderTotals[$label] = $orderTotal;

		return $orderTotal;
	}

/**
 * Get data from an order based on the name of the data.
 *
 * @param array  $order An array containing the order to get the data from
 * @param string $name  The name of the data to get.
 * @return string|null The data found. Null if no data found.
 */
	protected function _getOrderDataForOrder($order, $name) {
		if (empty($this->_config['paths']['data'])) {
			return null;
		}

		$data = Hash::get($order, $this->_config['paths']['data']);

		//Get the first set of data that matches the name
		foreach ($data as $datum) {
			if ($datum['name'] == $name) {
				return $datum['data'];
			}
		}

		return null;
	}

/**
 * Get the delivery details for a Cybertill order. The details are based on the user and as we use one user for all orders
 * we can't change it based on the actual delivery address of the order.
 *
 * @param array $order An array containing the order to get the delivery details from
 * @return array The array containing the payment details
 */
	protected function _getDeliveryForOrder($order) {
		$deliveryDetails = $this->_config['order_delivery'];

		$deliveryAmount = 0;
		if (!empty($deliveryDetails['delivery_label'])) {
			$deliveryAmount = $this->_getOrderTotalForOrder($order, $deliveryDetails['delivery_label']);
		}

		return [
			'success' => true,
			'data' => [
				'isCollection' => true,
				'addressId' => $this->_customer->addresses->item->id,
				'serviceId' => null,
				'tariffId' => null,
				'tariff' => $deliveryAmount,
				'vatRate' => null,
				'recipient' => (!empty($deliveryDetails['recipient'])) ? $deliveryDetails['recipient'] : '',
				'dateRequired' => '',
				'when' => null,
				'instructions' => (!empty($deliveryDetails['instructions'])) ? $deliveryDetails['instructions'] : '',
				'giftMessage' => (!empty($deliveryDetails['giftMessage'])) ? $deliveryDetails['giftMessage'] : '',
				'giftReceipt' => false
			]
		];
	}

/**
 * Create an item that represents delivery. This item should be included so that the full customer payments can be used.
 *
 * @param array $order An array containing the order to get the delivery details from
 * @return array 'success' => true/false if item received, 'item' => Cybertill item data.
 */
	protected function _getDeliveryItemForOrder($order) {
		if (empty($this->_config['order_delivery']['delivery_item_sku'])) {
			return [
				'success' => false,
				'message' => 'Delivery item SKU is not defined',
			];
		}

		$itemSkuPath = $this->_config['paths']['sku'];

		if (empty($itemSkuPath)) {
			return [
				'success' => false,
				'message' => 'SKU path is missing for delivery item',
			];
		}

		$deliveryAmount = 0;
		if (!empty($this->_config['order_delivery']['delivery_label'])) {
			$deliveryAmount = $this->_getOrderTotalForOrder($order, $this->_config['order_delivery']['delivery_label']);
		}

		$deliveryOrderItem = [
			'quantity' => 1,
			'unit_price_inc_tax' => $deliveryAmount,
		];

		$deliveryOrderItemData = [
			$itemSkuPath => $this->_config['order_delivery']['delivery_item_sku'],
		];

		$deliveryOrderItem = array_merge_recursive(
			$deliveryOrderItem,
			Hash::expand($deliveryOrderItemData)
		);

		return $this->_getItemForOrder(
			$deliveryOrderItem,
			$itemSkuPath
		);
	}

/**
 * Get the payment details for a Cybertill order. By default the payment is recorded as a card payment unless
 * specified otherwise in the config.
 *
 * @param array $order An array containing the order to get the payment details from
 * @return array The array containing the payment details
 */
	protected function _getPaymentsForOrder($order) {
		$payments = [];
		$paymentDetails = $this->_config['order_payments'];

		if (!empty($paymentDetails['gift_card_label']) && !empty($paymentDetails['gift_card_data_name'])) {
			//Get gift card amount
			$giftCardAmount = $this->_getOrderTotalForOrder($this->_order, $paymentDetails['gift_card_label']);

			//Get gift card code
			$giftCardCode = null;
			$giftCardData = $this->_getOrderDataForOrder($this->_order, $paymentDetails['gift_card_data_name']);
			if ($giftCardData !== null) {
				$giftCardData = json_decode($giftCardData, true);

				if (!empty($giftCardData['code'])) {
					$giftCardCode = $giftCardData['code'];
				}
			}

			if (!empty($giftCardAmount) && !empty($giftCardCode)) {
				$payments[] = [
					'type' => !empty($paymentDetails['gift_card_type']) ? $paymentDetails['gift_card_type'] : 12,
					'total' => abs($giftCardAmount),
					'cardNumber' => $giftCardCode,
				];
			}
		}

		if (!empty($paymentDetails['total_label'])) {
			//Get the payment total from the order
			$totalAmount = $this->_getOrderTotalForOrder($order, $paymentDetails['total_label']);

			$payments[] = [
				'type' => (!empty($paymentDetails['type'])) ? $paymentDetails['type'] : 2,
				'total' => $totalAmount,
			];
		}

		return [
			'success' => true,
			'data' => $payments,
		];
	}

/**
 * Update the stock in the website database using the stock from Cybertill.
 *
 * @param int 	$cybertillStock The amount of stock currently in Cybertill
 * @param array $item           The item to update
 * @return bool Whether the update was successful or not
 */
	protected function _updateStock($cybertillStock, $item) {
		// Update the inventory with the stock from cybertill
		$StockModel = EvClassRegistry::init($this->_config['models']['stock_model']);

		$stock = $StockModel->find(
			'first',
			[
				'conditions' => [
					'id' => $item['Inventory']['id']
				]
			]
		);

		//Item exists on site so we can update inventory
		$stock[$StockModel->alias]['stock'] = $cybertillStock->item->stock;

		return $StockModel->save($stock);
	}

/**
 * Update the pricing on the website database using the pricing from Cybertill
 *
 * @param int $cybertillPricing The pricing information currently in Cybertill
 * @param array $item The item to update
 * @return bool Whether the update was successful or not
 */
	protected function _updatePrice($cybertillPricing, $item) {
		if (empty($this->_config['models']['pricing_model']) || empty($this->_config['models']['item_model'])) {
			return false;
		}

		$PricingModel = EvClassRegistry::init($this->_config['models']['pricing_model']);
		$ItemModel = EvClassRegistry::init($this->_config['models']['item_model']);

		$pricing = $PricingModel->find(
			'first',
			[
				'conditions' => [
					'variant_id' => $item[$ItemModel->alias]['id'],
				],
			]
		);

		if (empty($pricing)) {
			return false;
		}

		$pricing = $pricing[0];

		// only persist to the database if the prices are updated
		$updatedPrices = false;
		$priceMapping = [
			'priceRrp' => 'rrp',
			'priceStore' => 'price',
			'priceTrade' => 'trade_price',
		];

		foreach ($priceMapping as $cybertillField => $modelField) {
			if (
				!empty($cybertillPricing->{$cybertillField}) &&
				$cybertillPricing->{$cybertillField} != $pricing[$PricingModel->alias][$modelField]
			) {
				$pricing[$PricingModel->alias][$modelField] = $cybertillPricing->{$cybertillField};
				$updatedPrices = true;
			}
		}

		if ($updatedPrices) {
			$this->_setSaleProductFlag($item, $ItemModel, $cybertillPricing->priceRrp > 0);
			return $PricingModel->save($pricing, [
				'removeTax' => true, // Cybertill prices are after tax
			]);
		}

		return true;
	}

/**
 * Get the items to be updated during a Cybertill pull.
 *
 * @param Model $ItemModel The item model to use.
 * @param array $params    Query parameters.
 * @return array.
 */
	protected function _getItemsToPull($ItemModel, $params = []) {
		return $ItemModel->find('all', $params);
	}

/**
 * Update the products is_sale_product flag
 *
 * @param array $item Variant
 * @param Model $model Variant model
 * @param bool $isSaleProduct True if product is sale product
 * @return void
 */
	protected function _setSaleProductFlag($item, $model, $isSaleProduct) {
		if (empty($item[$model->alias]['product_id'])) {
			return;
		}

		$model->Product->save([
			'id' => $item[$model->alias]['product_id'],
			'is_sale_product' => $isSaleProduct,
		], ['validate' => false, 'callbacks' => false]);
	}
}
