<?php

App::uses('HttpSocket', 'Network/Http');

/**
 * Exchange
 *
 * Exchange rate wrapper class, also provides on the fly conversion support
 *
 * @author Rick Mills <rick@evoluted.net>
 */
class Exchange {

    public $api_key;
    public $base_currency;
    public $cmd;
    public $api_url = 'http://www.exchangerate-api.com/';

    /**
     * Set up our global variables that'll be accessible from the child api class.
     */
    public function __construct()
    {
        $this->base_currency = Configure::read('EvCurrency.base_currency');
        $this->api_key = Configure::read('EvCurrency.api_key');
        $this->cmd = new HttpSocket();
    }

    /**
     * Calculate Price
     *
     * Does a simple currency conversion based on the current price in the base currency, and the rate
     * to convert it at.
     *
     * @param  decimal $rate The exchange rate to convert into
     * @param  decimal $price The base price we're converting
     * @return decimal The converted price
     */
    public function calculatePrice($rate, $price)
    {
        return $rate * $price;
    }

    /**
     * Calculate All Prices
     *
     * Uses the calculatePrice method to do a bulk set of conversions at once.
     *
     * @param  array $rates An array of the exchange rates to convert into
     * @param  decimal $price The base price we're converting
     * @return array An array of the converted prices
     */
    public function calculateAllPrices($rates, $price)
    {
        $prices = array();

        foreach ($rates as $code => $rate) {
            $prices[$code] = $this->calculatePrice($rate, $price);
        }

        return $prices;
    }

     /**
     * Get
     *
     * Used to build the query and retrieve a rate. Not accessible outside of this class.
     *
     * @param  string $rate The currency code to get the rate for.
     * @return string The currency exchange rate for the provided currency code.
     */
    private function get($rate) {

        $params['k'] = $this->api_key;

        return $this->cmd->get($this->api_url . $this->base_currency . '/' . $rate . '/' . '1.00', $params)->body;
    }

    /**
     * Get Rates
     *
     * Returns an array of exchange rates against your base currency.
     *
     * @param  array|string  $rates array of currency codes to find rates for (or a singluar string if only puling one rate)
     * @return array Returns an array of rate values with the currency code as the key for each rate
     */
    public function getRates($rates = array()) {

        if (! is_array($rates)) {
            $rates = array($rates);
        }

        $rate_values = array();

        foreach ($rates as $rate) {
            $rate_values[$rate] = $this->get($rate);
        }
        return $rate_values;
    }
}