# EvCore

Base plugin for all Evoluted New Media CakePHP based websites.

## Features

### readForViewOrFail

To retrieve a record from a model for view you can use:-

	$data = $this->Model->readForViewOrFail($id);

If the record isn't found it will throw a `NotFoundException`.

### User Management

#### User Groups

By default there are user groups included in EvCore. There are:
	* User, users who have access to frontend authorised pages but not the admin area.
	* Administrator, users who have access to the frontend and admin area but can't remove protected content.
	* Super User, users who have complete access to frontend and admin areas.

Access to the various areas of a site are dependent on the level of the user group that the user belongs to. Additional user groups can be added with various custom levels to create a site with hierarchial user access.

#### User Group Permissions

If you need user access that isn't hierarchial then you will need to use user group permissions. To view the permissions for a user group then you will need to switch the config setting `enable_user_group_permissions` to true. This will inject the permissions tab onto the user group admin edit form.

Unless custom permissions have been added, user group permissions are based on public controller actions so that super admins and admins have access to every controller action and users have access to every controller but are denied access to admin actions. By default the user group permissions reflect the user group level restrictions. Permissions are hierarchial so that allowing access to a parent permission provides access to the child permissions. Similarly if a parent is denied access then the children are also denied access. Exceptions can be created by allowing the permission for a parent and denying the permission for the child, similarly if you deny the parent and allow the child. If a permission isn't set then it will inherit it's access from the parent. If the parent has no permission then it is assumed denied.

If new controllers and actions have been created since the permissions were first generated, then they will be missing from the database and not able to be accessed. To generate the permissions for them run the following command:

	Console/cake EvCore.Permissions sync

If you want to also enable default permissions to be created/updated with the sync then run the following command:

	Console/cake EvCore.Permissions sync -u

If you have set custom user group permissions then it is possible that this will overwrite them.

If you are upgrading EvCore on an older project that didn't have user group permissions before and are finding yourself locked out of every restricted page then ensure you have user group permission tables by running the following command:

	Console/cake Migrations.migration run all --plugin EvCore

This will add the tables and set up the default user permissions as described above. You should now be able to access restricted pages.

#### Adding User Groups

Here is a cut down example migration from Beta Climbing Designs:

```php
// app/Config/Migration/1521541866_add_blogger_user_group.php
App::uses('EvCorePermissionsShell', 'Console/Command');

public function after($direction) {
	if ($direction == 'up') {
		$UserGroup = EvClassRegistry::init('EvCore.UserGroup');

		//Save new user groups
		$UserGroupData = [
			[
				'id' => 4,
				'name' => 'Blogger',
				'level' => 15,
				'is_active' => true,
			]
		];
		$UserGroup->saveMany($UserGroupData);

		//Save default user groups as aro records.
		$aroData = [
			[
				'parent_id' => null,
				'model' => 'UserGroup',
				'foreign_key' => 4
			]
		];
		EvClassRegistry::init('Aro')->saveMany($aroData);

		//Pull in all the controller actions to be used as aco records.
		$PermissionsShell = new EvCorePermissionsShell();
		$PermissionsShell->params = [
			'update-default-permissions' => true,
		];
		$PermissionsShell->startup();
		$PermissionsShell->sync();
	}
	return true;
}
```

The migration above requires the PermissionsShell to be extended

```php
// app/Console/Command/EvCorePermissionsShell.php
App::uses('PermissionsShell', 'EvCore.Console/Command');

class EvCorePermissionsShell extends PermissionsShell {

/**
 * Sync permissions
 *
 * @return void
 */
	public function sync() {
		parent::sync();
		if (empty($this->params['skip-sync'])) {
			// Pull in all the controller actions to be used as aco records.
			$aclExtrasShell = new AclExtrasShell();
			$aclExtrasShell->startup();
			$aclExtrasShell->aco_sync();
		}

		if (!empty($this->params['update-default-permissions'])) {
			$this->_setModel('UserGroup');

			$this->_setForeignKey(4);
			$this->out("Updating default permissions for " . $this->_model . ' ' . $this->_foreignKey);
			$this->_allowBloggerPermissions($this->_model, $this->_foreignKey);
		}

		return null;
	}

/**
 * Sets permissions for bloggers
 *
 * @param Array $model    User Group model
 * @param Int $foreignKey Foreign Key to be user
 * @return void
 */
	protected function _allowBloggerPermissions($model, $foreignKey) {
		$this->_adminFunctionsForAll();
		$this->_allowBlog();
	}

/**
 * Allows blog for the current foreign key
 *
 * @return void
 */
	protected function _allowBlog() {
		$this->_allowAdminGenericRouteMethods('controllers/EvBlogBlogPosts/', [
			'ev_tiny_mce_image'
		]);
	}

/**
 * Allows basic admin functions
 *
 * @return void
 */
	protected function _adminFunctionsForAll() {
		$this->_denyAdminPermission($this->_model, $this->_foreignKey);

		// Personal account
		$this->_allowArray([
			'route' => 'controllers/EvCore/Admin/',
			'actions' => ['admin_index'],
		]);

		// Basic Self Control
		$this->_allowArray([
			'route' => 'controllers/EvCoreUsers/',
			'actions' => ['admin_account', 'admin_logout', 'admin_switch'],
		]);

		// Basic Self Control
		$this->_allowAdminGenericRouteMethods('controllers/EvTemplates/');

		// TinyMCE plugin
		$this->_allowArray([
			'route' => 'controllers/EvTinyMceImage/ImagePicker/',
			'actions' => ['admin_select'],
		]);

		// Ajax
		$this->_allowAdminTemplateAjax($this->_model, $this->_foreignKey);
	}


}

```

#### Admin User Events

When a new users is created or udpated, if they are active and they have admin permissions, an event is fired to signal that an admin user has been added/updated.

An email can be sent to the new admin user if the config setting `new_admin_users_password_reset` has been enabled. This hooks into the password reset functionality by creating a password reset code and setting the reset code expiry date. The email is sent using a different template than the normal password reset email but directs users to the same callback.

### Google reCAPTCHA

In a bid to tackle User Enumeration attacks, Google reCAPTCHA support is available on login and registration forms. Below you will find an example implementation. Use of Google reCAPTCHA on login and registration forms is controlled through the EvCore config file (`EvCore/Config/config.php`). As default `userReCaptcha` is set to `false` and does not automatically run out of the box. Set `EvCore.userReCaptcha.login` or `EvCore.userReCaptcha.register` to `false` to use one of the two implementations available.

	<?php
		$config = [
			'EvCore' => [
				...
				// define project specific Google reCAPTCHA details below. reCAPTCHA elements
				// will automatically show on login and registration forms when login or
				// registration interactions fail by the number of allowed attempts defined
				'userReCaptcha' => [
					// replace with site specific secret
					'secret' => '',

					// replace with site specific siteKey
					'siteKey' => '',

					// the total number of failed login attempts before reCAPTCHA displays
					'showAfterFailedLoginAttempts' => 5,

					// the total number of failed registration attempts before reCAPTCHA displays
					'showAfterFailedRegisterAttempts' => 5,
				],
				...
			],
		];

Site specific Google reCAPTCHA keys can be obtained from [https://www.google.com/recaptcha/admin#list](https://www.google.com/recaptcha/admin#list)

A note to bear in mind when adding the functionality to existing sites, you will need to add the reCAPTCHA form element manually (see snippet below).

	if (isset($showReCaptcha) && $showReCaptcha === true):
        echo $this->Form->recaptcha('User.g-recaptcha-response', [
            'data-sitekey' => Configure::read('EvCore.userReCaptcha.siteKey')
        ]);
    endif;

### Extended Paginator Helper

Functionality is available out of the box to echo prev and next canonical tags in site head elements

```<?php echo $this->Paginator->canonical(); ?>```

### Virtual fields in queries

Virtual fields can be added to models to provide fields in query results that don't exist in the database. Sometimes these are added to specific queries and used data from joins. This can cause issues if the virtual field is not removed once the query has been performed. To make sure that the virtual field are removed, pass the virtualFields you want to add to the query into a query parameter array with the key `virtualFields`. These will be added temporarily and tidied up automatically.

## Admin

### Form Fields

Form fields are assigned to a CakePHP variable through the `_adminFormFields` method in `EvCoreController` based on the current model's database schema. Form fields are then added to the `admin_form` Scaffold template in `core-cms`. The `$formFields` variable is then assigned as a CakePHP template variable, typically, through the `admin_edit` Controller action.

### Form Tabs

Tabs can be added to admin forms in a number of ways. Firstly by extending the `admin_form` admin template Scaffold, or by assigning them to the `$formFields` variable in your controller.

`core-cms` is configured to convert the `tabs` key of a `$formFields` variable to individual tabs, rendering each set of fields within the `tabs` array through the `EvForm` Helper.

For example, we would like to add 'foo' and 'bar' text input forms fields to a tab with name 'Tab 1'. The key value of each nested `tabs` is used as the tab title.

	<?php
		$formFields = [
			'ModelAlias.name' => [
				'label' => 'Name',
				'type' => 'text'
			],
			...
			'tabs' => [
				'Tab 1' => [
					'ModelAlias.foo' => [
						'label' => 'Foo',
						'type' => 'text'
					],
					'ModelAlias.bar' => [
						'label' => 'Bar',
						'type' => 'text'
					]
				]
			]
		];
	?>

### Pages

Pages have an internal_title field. This is used as an admin only display field so that pages can found easier in the CMS. For example:

	Title: The Bamboo Philosophy
	Internal Title: About Us

This makes it clear that this page is the about us page. This internal title can also be passed into `assignPage` in place of the page id. This is preferred as the internal title can be set manually to be consistent whereas ids will likely change between environments.

#### Exporting Pages

Using the `EvMigrationUtil` library, pages can be exported as a migration. This currently exports the page and its data including any custom fields. The template will be created as part of the migration if it doesn't already exist. To export to a migration use:

	app/Console/cake EvMigrationUtil.Migration export EvCore.Page page_internal_title

Replace `page_internal_title` with the internal title of the page. If the page already exists when the migration is run it will not be updated.

### S3 Uploads

You can upload documents and images directly to Amazon's S3 service by configuring the `s3Upload` settings. Files uploaded this way will store the AWS file URL rather than the filename in the relevant database tables.

To use this the Composer package `aws/aws-sdk-php` is required.

## Common JS Libraries

As common js code used in the admin is updated to the component format, it will be added to EvCore under webroot/js/components. Any of these components can be activated in a view. An example include would be:

	$this->Html->script('EvCore.components/cloneable', ['inline' => false, 'once' => true]);

### Cloneable

This is a replacement for the jQuery.cloneable.js from core-cms. It is useful when an unknown number of fields is required. Your form must be set up in this format:

	<div class="row" data-cloneable-container>

		<?php
			// This will be a hidden field submitted along with the data that contains a
			// comma separated list of all deleted item ids. This will need to be handled in your controller
		    echo $this->Form->hidden(
		        'Model.to_delete',
		        array(
		            'data-cloneable-removed-ids' => '',
		            'default' => '0'
		        )
		    );
		?>

		<?php
			// Create as many of the below rows as required using a 'Model.index.fieldName' naming convention.
			// There should always be at least one row. If it shouldn't be visible to begin with add a hidden class to the row
			// and the component will handle showing it when the add button is clicked.
		?>
		<div data-cloneable-row>
			<?php
				// The ID field is marked with 'data-cloneable-id-field' so it's value can be added to the delete list on remove
				echo $this->Form->hidden('Model.0.id', [
					'data-cloneable-id-field' => '',
					'default' => 0
				]);

				// There can be any number of other fields, they will be cloned and cleared by the plugin
				// Make sure to use the 'Model.index.fieldName' naming convention for fields.
			?>

			<?php // The remove button can be anywhere within the cloneable row div ?>
			<div class='col-xs-1'>
				<div data-cloneable-remove><i class='fa fa-times'></i></div>
			</div>
		</div>

		<div class="spacer"></div>

		<?php // The add button can be anywhere within the cloneable container div ?>
		<div class="col-xs-6 col-sm-offset-2 col-md-offset-3 col-lg-offset-2">
			<div class='btn' data-cloneable-add>Add Attribute</div>
		</div>
	</div>

If a row needs to maintain its value when cloned (e.g. for setting up a relationship) then add a `data-cloneable-const` attribute to the input. This will cause it not to be cleared on clone.

If confirmation is required for removing a row, add a `data-cloneable-confirm-delete` flag to the cloneable container.

## Minify Html

HTML is minified by default. To disable this set `app.disable_minify_html` to `true` in bootstrap:-

```
Configure::write('app.disable_minify_html', true);
```

## Creating Permissions

### An example of how it was used in DeltaTrust

```
public function sync() {
	// See app/Console/Command/EvCorePermissionsShell.php for the rest of this method

	$this->_setModel('UserGroup');
	$this->_updateForeignKey(2);
	$this->_deltaStaffPermissions();
}

protected function _deltaStaffPermissions() {
    $this->_adminFunctionsForAll();

    $this->_allowPages();
    $this->_allowNews();
    $this->_allowBlog();
}

protected function _adminFunctionsForAll() {
    $this->_denyAdminPermission($this->_model, $this->_foreignKey);

    // Personal account
    $this->_allowArray([
        'route' => 'controllers/EvCore/Admin/',
        'actions' => ['admin_index'],
    ]);

    // Basic Self Control
    $this->_allowArray([
        'route' => 'controllers/EvCoreUsers/',
        'actions' => ['admin_account', 'admin_logout', 'admin_switch'],
    ]);

    // TinyMCE plugin
    $this->_allowArray([
        'route' => 'controllers/EvTinyMceImage/ImagePicker/',
        'actions' => ['admin_select'],
    ]);

    // Ajax
    $this->_allowAdminTemplateAjax($this->_model, $this->_foreignKey);
}

protected function _allowPages() {
    $this->_allowAdminGenericRouteMethods('controllers/EvCorePages/', ['ev_tiny_mce_image']);
}

protected function _allowNews() {
    $this->_allowAdminGenericRouteMethods('controllers/News/NewsPosts/', ['ev_tiny_mce_image']);
}

protected function _allowBlog() {
    $this->_allowAdminGenericRouteMethods('controllers/EvBlogBlogPosts/', ['ev_tiny_mce_image']);
}
```
