<?php

App::uses('CakeTime', 'Utility');
App::uses('CakeEmail', 'Network/Email');
App::uses('AuthComponent', 'Controller/Component');
App::uses('CakeSession', 'Model/Datasource');

App::uses('BasicPermissions', 'EvCore.Lib');

/**
 * Business logic for user accounts
 */
class User extends AppModel {

	public $displayField = "email";

	public $order = array('email' => 'ASC');

	public $belongsTo = array(
		'UserGroup'
	);

	public $validate = array(
		'user_group_id' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'User group cannot be blank'
			),
		),
		'email' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Email cannot be left blank'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'You must enter a valid email address'
			),
			'customUnique' => array(
				'rule' => 'customUnique',
				'message' => 'A user with that email address already exists'
			)
		),
		'password' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Password cannot be left blank',
				'on' => 'create'
			)
		),
		'confirm_password' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'You must confirm your password',
				'on' => 'create'
			),
			'matches' => array(
				'rule' => array('checkMatches', 'password'),
				'message' => 'The passwords do not match'
			)
		)
	);

/**
 * Constructor. Binds the model's database table to the object.
 *
 * @param bool|int|string|array $id Set this ID for this model on startup,
 * can also be an array of options, see above.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		$this->virtualFields['name'] = 'TRIM(CONCAT(IFNULL(' . $this->alias . '.first_name, \'\'), " ", IFNULL(' . $this->alias . '.last_name, \'\')))';
	}

/**
 * Ensure the password is encrypted if present before it's saved to the DB. If a new user is being created and a
 * password hasn't been provided to be encrypted then generate a new random password to provide the user instead.
 *
 * @param array $options beforeSave options.
 * @return void.
 */
	public function beforeSave($options = array()) {
		if (!empty($this->data[$this->alias]['password'])) {
			$this->data[$this->alias]['password'] = AuthComponent::password($this->data[$this->alias]['password']);
		} elseif (empty($this->id)) {
			// This is a new user and no password has been specified - set a random password
			$this->data[$this->alias]['password'] = $this->data[$this->alias]['confirm_password'] = AuthComponent::password($this->_generateRandomPassword());
		} else {
			unset($this->data[$this->alias]['password']);
		}
	}

	public function afterSave($created, $options = array()) {
		$Permissions = new BasicPermissions(
			EvClassRegistry::init('EvCore.UserGroup')
		);

		if (!empty($this->data['User']['id'])) {
			$user = $this->find(
				'first',
				[
					'conditions' => [
						'User.id' => $this->data['User']['id']
					],
					'contain' => [
						'UserGroup',
					]
				]
			);
		}

		//Check if an admin user has been created
		if (!empty($user['User']['is_active']) && !empty($user['UserGroup']) && $Permissions->hasAdminPermission($user)) {
			if ($created) {
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCore.Model.User.createdActiveAdminUser', $this, array(
						'userId' => $user['User']['id'],
					))
				);
			} else {
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCore.Model.User.updatedActiveAdminUser', $this, array(
						'userId' => $user['User']['id'],
					))
				);
			}
		}

		return parent::afterSave($created, $options);
	}

/**
 * Validate password rules. Checks if password validation functions have been set in the config and if they exist then
 * user them to get the password validation rules. If not fallback on the admin user password validation rules.
 * @param int   $id            User ID.
 * @param int   $minCharacters Minimum password length.
 * @param array $data          Data that is being saved.
 * @return array Validation rules
 */
	public function validatePassword($id = null, $minCharacters = null, $data = null) {
		if (! empty($id)) {
			$existingPassword = $this->fieldOnly('password', array('id' => $id));
		}

		if (!empty($data['User']['user_group_id'])) {
			if (Configure::check('EvCore.validatePassword')) {
				$methods = Configure::read('EvCore.validatePassword');
				if (isset($methods[$data['User']['user_group_id']]) && method_exists($this, $methods[$data['User']['user_group_id']])) {
					$rules = $this->{$methods[$data['User']['user_group_id']]}($id, $minCharacters);
				}
			}
		}

		if (empty($rules)) {
			$rules = $this->_validatePasswordAdminUser($id, $minCharacters);
		}

		if (! empty($existingPassword)) {
			$rules['password']['existing'] = array(
				'rule' => array('customNotMatchExisting', $existingPassword),
				'message' => 'The new password must be different to the previous password used'
			);
		}

		return $rules;
	}

/**
 * Get the password validation rules for a frontend user.
 * @param  int    $id            Id of an existing user
 * @param  int    $minCharacters The minimum length of characters a password must be.
 * @return array                 Validation rules
 */
	protected function _validatePasswordFrontendUser($id = null, $minCharacters = null) {
		$rules = array(
			'password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Password cannot be left blank',
					'on' => 'create'
				),
				'minlength' => array(
					'rule' => array('customMinPasswordLength', $minCharacters)
				),
				'alphaNumeric' => array(
					'rule' => 'customAlphaNumericSpecial',
					'message' => 'Your password must contain at least one letter, one number and one special character (!@#$&*)'
				),
			),
			'confirm_password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'You must confirm your password'
				),
				'matches' => array(
					'rule' => array('checkMatches', 'password'),
					'message' => 'The passwords do not match'
				)
			)
		);

		return $rules;
	}

/**
 * Get the password validation rules for an admin user.
 * @param  int    $id            Id of an existing user
 * @param  int    $minCharacters The minimum length of characters a password must be.
 * @return array                 Validation rules
 */
	protected function _validatePasswordAdminUser($id = null, $minCharacters = null) {
		$rules = array(
			'password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Password cannot be left blank',
					'on' => 'create'
				),
				'minlength' => array(
					'rule' => array('customMinPasswordLength', $minCharacters)
				),
				'alphaNumeric' => array(
					'rule' => 'customAlphaNumericSpecial',
					'message' => 'Your password must contain at least one alphabetic, one numeric character and one special character (!@#$&*)'
				),
				'case' => array(
					'rule' => 'customLowercaseAndUppercase',
					'message' => 'Your password must contain at least one lowercase and one uppercase character'
				),
				'dictionary' => array(
					'rule' => 'customPasswordDictionary',
					'message' => 'Your password should be unique and cannot include commonly used pass-phrases'
				),
			),
			'confirm_password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'You must confirm your password'
				),
				'matches' => array(
					'rule' => array('checkMatches', 'password'),
					'message' => 'The passwords do not match'
				)
			)
		);

		return $rules;
	}

/**
 * Validation rules called from admin_account and account actions
 */
	public function validateAccount() {
		$rules = $this->validate;

		unset($rules['password']);
		unset($rules['confirm_password']);

		return $rules;
	}

/**
 * Validation rules called from the admin_edit action
 */
	public function validateEdit() {
		$rules = $this->validate;

		// add in a permissions rule to check that the user being edited cannot be
		// granted higher permissions than the current auth user
		$rulesUserGroupPermissions = array(
			'user_group_id' => array(
				'permissionLevel' => array(
					'rule' => 'validatePermissionLevel',
					'message' => 'The user group selection was invalid'
				)
			)
		);

		return array_merge_recursive($rules, $rulesUserGroupPermissions);
	}

	/**
	 * Validation rules for a login form submission
	 *
	 * @return array
	 */
	public function validateLogin() {
		return array(
			'email' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Email cannot be left blank.'
				),
				'email' => array(
					'rule' => 'email',
					'message' => 'You must enter a valid email address'
				)
			),
			'password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Password cannot be left blank'
				)
			)
		);
	}

/**
 * Validation rules to check the supplied reCAPTCHA submission
 *
 * @return array
 */
	public function validateReCaptcha() {
		return array(
			'g-recaptcha-response' => array(
				'recaptcha' => array(
					'rule' => array('customValidateReCaptcha'),
					'message' => 'Please complete the reCAPTCHA field'
				)
			)
		);
	}

	/**
	 * custom unique check, need to check to see if user is guest user first
	 *
	 * @param array $check email field to check
	 * @return  bool
	 */
	public function customUnique($check) {
		$conditions = array(
			$this->alias . '.email' => $check['email'],
			$this->alias . '.is_guest_user' => 0
		);

		if ($this->id > 0) {
			$conditions[] = $this->alias . '.id != ' . $this->id;
		}

		$count = $this->find('count', array('conditions' => $conditions));

		return ($count >= 1 ? false : true);
	}

	/**
	 * Returns true if an email already exists
	 *
	 * @param array $check email field to check
	 * @return true if email doesn't exists, otherwise false
	 */
	public function checkIsRegistered($check) {
		App::uses('User', 'Model');

		$value = array_pop($check);

		$user = $this->findByEmail($value);

		return empty($user);
	}

	/**
	 * Gets the user and all associated data out that is needed for login
	 * @param  int $id  The ID of the user
	 * @return array    User array
	 */
	public function getUserForLogin($id) {
		$fullUser = $this->find('first', array(
				'conditions' => array(
					$this->alias . '.id' => $id
				),
				'contain' => array(
					'UserGroup'
				)
			)
		);

		return $fullUser;
	}

	/**
	 * Sets a password reset code for a user.
	 *
	 * @param integer $userId
	 * @return boolean - returns false if the user cannot be found
	 */
	public function resetPassword($userId) {
		$user = $this->findById($userId);

		if (!empty($user)) {
			$code = md5(uniqid());
			$this->id = $userId;
			$this->save([
				'password_reset_code' => $code,
				'password_reset_code_expires' => gmdate('Y-m-d H:i:s', strtotime('+30 minutes'))
			]);

			App::uses('EvUserPasswordResetListener', 'EvCore.Lib/Event');
			$this->getEventManager()->attach(new EvUserPasswordResetListener());
			$event = new CakeEvent('Model.User.passwordReset', $this, array(
					'Model' => $this->alias,
					'User' => $user,
					'code' => $code,
					'url' => Router::url(
						array(
							'plugin' => false,
							'controller' => 'ev_core_users',
							'action' => 'password_reset_callback',
							$code
						),
						true
					)
				)
			);
			$this->getEventManager()->dispatch($event);

			return true;
		}

		return false;
	}

	/**
	 * register a new user
	 *
	 * @param   array   data from request->data
	 * @return  bool
	 */
	public function register($data) {
		$activationType = Configure::read('SiteSetting.users.activation');

		$data[$this->alias]['is_active'] = 0;
		if (
			$activationType === 'auto' ||
			(isset($data[$this->alias]['is_guest_user']) && (bool)$data[$this->alias]['is_guest_user'] === true)
		) {
			$data[$this->alias]['is_active'] = 1;
		}

		if (
			isset($data[$this->alias]['is_guest_user']) &&
			(bool)$data[$this->alias]['is_guest_user'] === true
		) {
			$data[$this->alias]['password'] = $data[$this->alias]['confirm_password'] = $this->_generateRandomPassword();
		}

		// if a user group id has been requested then check that it is allowed
		$allowedUserGroupIds = Configure::read('EvCore.allowed_user_group_registration_ids');
		if (
			! empty($data[$this->alias]['user_group_id']) &&
			in_array($data[$this->alias]['user_group_id'], $allowedUserGroupIds)
		) {
			$userGroupId = $data[$this->alias]['user_group_id'];
		} else {
			// no user_group_id was requested, or the requested value is not allowed
			$userGroupId = Configure::read('EvCore.default_user_group_registration_id');
		}

		$data[$this->alias]['user_group_id'] = $userGroupId;

		// add the additional validation rules for the password
		$this->validate = array_merge_recursive(
			$this->validate,
			$this->validatePassword(
				null,
				null,
				$data
			)
		);

		$return = $this->saveAssociated($data, ['deep' => true]);

		if ($return) {
			// dispatch new user event
			$this->getEventManager()->dispatch(
				new CakeEvent('Model.User.newUser', $this, array(
					'Model' => $this->alias,
					'user' => $data
				))
			);

			if (! isset($data[$this->alias]['is_guest_user']) || intval($data[$this->alias]['is_guest_user']) === 0) {
				if ($activationType == 'auto') {

					// dispatch auto activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.autoActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data
						))
					);

				} elseif ($activationType == 'email') {

					$code = hash('sha1', $this->id . '-' . $data[$this->alias]['email'] . '-' . time());
					$this->saveField('verification_code', $code);

					// dispatch email activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.emailActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data,
							'url' => Router::url(
								array(
									'plugin' => false,
									'controller' => 'ev_core_users',
									'action' => 'verify_email',
									$code
								),
								true
							)
						))
					);
				} elseif ($activationType == 'manual') {

					// dispatch email activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.manualActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data,
							'url' => Router::url(
								array(
									'admin' => true,
									'plugin' => false,
									'controller' => 'ev_core_users',
									'action' => 'edit',
									$this->id
								),
								true
							)
						))
					);
				}
			}
		}

		return $return;
	}

	/**
	 * Generate a random password for users. Used when creating a guest user account. Defaults to password length of 15
	 * to match the requirements for admin users.
	 * @param  int    $passwordLength The length of password to create, values lower than 4 will not meet requirements
	 * @return string                 The password string generated
	 */
	protected function _generateRandomPassword($passwordLength = 15) {
		//The sets of characters to randomly select from. Split due to requirements of at least 1 character from each string.
		$characters = [
			'0123456789',
			'abcdefghijklmnopqrstuvwxyz',
			'ABCDEFGHIJKLMNOPQRSTUVWXYZ',
			'!@#$&*'
		];

		$randomString = '';
		for ($i = 0; $i < $passwordLength; $i++) {
			//Get a character from each set of characters as we loop through
			$characterString = $characters[($i % 4)];
			$characterStringLength = strlen($characterString) - 1;

			//Select a random character from the current set of characters
			$randomString .= $characterString[rand(0, $characterStringLength)];
		}

		return str_shuffle($randomString);
	}

	/**
	 * Verify a user account from the code given in email
	 *
	 * @param   int     User ID
	 * @param   string  Verification code
	 * @return  bool
	 */
	public function verifyFromEmail($id, $code) {
		$return = $this->updateAll(
			array(
				'User.is_active' => 1,
				'User.verification_code' => null,
				'modified' => '"' . CakeTime::format('Y-m-d H:i:s', time()) . '"'
			),
			array(
				'User.id' => $id,
				'User.verification_code' => $code
			)
		);

		// dispatch email activate event
		$this->getEventManager()->dispatch(
			new CakeEvent('Model.User.emailVerified', $this, array(
				'Model' => $this->alias,
				'User' => $this->readForView($id)
			))
		);

		return $return;
	}

	/**
	 * Checks whether the required level of user group is less than or equal to the current auth user
	 *
	 * @param array $check containing user group to check against
	 * @return bool based on whether the items validate correctly
	 */
	public function validatePermissionLevel($check) {
		if (! empty($check['user_group_id'])) {
			$requiredLevel = $this->UserGroup->fieldOnly(
				'level',
				array('UserGroup.id' => $check['user_group_id'])
			);

			$authUserLevel = CakeSession::read('Auth.User.UserGroup.level');

			return (! empty($authUserLevel) && $authUserLevel <= $requiredLevel);
		}

		return false;
	}

/**
 * Custom validation rule to check for at least one occurance of an alphabetic character, at least one
 * occurance of a numeric character and at least one special character
 */
	public function customAlphaNumericSpecial($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		return preg_match('/(?=.*\d)(?=.*[a-zA-Z])(?=.*[!@#$&*])/', $value);
	}

/**
 * Custom validation rule to check that the requested password doesn't include
 * and values found in the password dictionary
 *
 * @param array $check The field to validate against
 * @return bool
 */
	public function customPasswordDictionary($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		// query the database to check that the password does not contain
		// and dictionary words
		return ! (bool)EvClassRegistry::init('EvCore.PasswordDictionary')->fieldOnly('id', [
			'LOCATE(password, ?)' => [$value]
		]);
	}

/**
 * Custom validation rule to check for at least one occurance of an lowercase character and at least one
 * occurance of a uppercase character
 */
	public function customLowercaseAndUppercase($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		return preg_match('/(?=.*[a-z])(?=.*[A-Z])/', $value);
	}

/**
 *  Custom validation rule to check that the newly supplied password is different to the previously stored password
 */
	public function customNotMatchExisting($check, $existing) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		// has the requested password value - mirroring the functionality used by the Auth component
		$value = Security::hash($value, 'sha1', true);

		return (empty($existing) || $value != $existing);
	}

/**
 * Custom validation method for checking password length depending on user group level (this can be
 * overridden by passing a minLength).
 * @param array $check Field to validate
 * @param int $minLength Minimum password length
 * @return bool Always returns true as the method directly invalidates the field where relevant
 */
	public function customMinPasswordLength($check, $minLength = null) {
		$field = key($check);
		$password = array_pop($check);
		if ($minLength === null) {
			$permissions = new BasicPermissions($this->UserGroup);
			$minLength = 8;
			// We need to determine the user group so that we can figure out the mimimum password
			// length; admin users will need a more secure password.
			if (!empty($this->id)) {
				$userGroup = $this->find('first', [
					'contain' => 'UserGroup',
					'conditions' => [$this->escapeField() => $this->id]
				]);
			} elseif (!empty($this->data[$this->alias]['user_group_id'])) {
				$userGroup = $this->UserGroup->findById($this->data[$this->alias]['user_group_id']);
			}
			if (!empty($userGroup) && $permissions->hasAdminPermission($userGroup)) {
				$minLength = 15;
			}
		}

		if (!Validation::minLength($password, $minLength)) {
			// Invalidate the field and set an error message with the minimum characters required.
			$this->invalidate($field, __d('ev_core', 'Your password must be a minimum of %d characters long', $minLength));
		}

		// Always return true as we've already invalidated the field if the rule failed in order to
		// inject an error message. If we don't do this we end up with multiple validation error
		// messages which we don't want.
		return true;
	}
}
