<?php

App::uses('Model', 'Model');
App::uses('Document', 'EvCore.Model');
App::uses('Image', 'EvCore.Model');
App::uses('CakeEvent', 'Event');

/**
 * EvCore model for Cake.
 *
 * All logic which is default to the Evoluted CMS is in here
 *
 * @package       app.Model
 */
class EvCoreAppModel extends Model {

	// Switch off automatic recursize queries app wide.
	public $recursive = -1;

	public $displayField = "name";

	// Name used in the admin theme for the model (override in your models or
	// leave as null to default to the model name).
	public $displayName = null;

	/**
	 * Define the number of images associated with the model.
	 *
	 * @var  integer|array
	 *
	 * It is now possible to have unlimited image slots
	 * Image Slots
	 * 		-1 = 	Unlimited Image Slots
	 *		0  = 	No Image Slots
	 *		X  = 	X Amount of Image Slots
	 *
	 * This can be done in 2 ways
	 * 		1. If the model only has one set of images
	 *			"public $imageSlots = 0;"
	 *
	 *		2. If there are multiple image areas to a page
	 *			public $imageSlots = array(
	 *				'main' => 4,
	 *				'listing' => 1,
	 *			);
	 *
	 *		public $imageSlots = array(
	 *			'main' => array(
	 *				'fields' => array(
	 *					'caption' => array(
	 *						'label' => 'Alt', //This array is passed through to FormHelper::input so can be used to  add class etc.
	 *					),
	 *					'button_text'
	 *				),
	 *				'slots' => -1,
	 *				'validate' => array(
	 *					'filename' => array(
	 *						'rule' => array('isBelowMaxSize', 1024, false),
	 *						'message' => 'File is too large'
	 *					)
	 *				)
	 *			)
	 *		);
	 *
	 *
	 */
	public $imageSlots = 0;

	/**
	 * Define the number of files/documents associated with the model.
	 *
	 * @see  AppModel::imageSlots
	 * @var  integer|array
	 */
	public $documentSlots = 0;

	// Define the menu content in this model belong in when using the
	// Navigatable behaviour.
	public $parentMenu = null;

	/**
	 * store when the behaviours have been checked so it doesn't happen again
	 * if bindModel is called
	 *
	 */
	public $behaviorsChecked = false;

	/**
	 * Returns a list of all events that will fire in the model during it's lifecycle.
	 *
	 * @return array
	 */
	public function implementedEvents() {
		$events = parent::implementedEvents();
		$events['Model.beforeBeforeSave'] = array('callable' => 'beforeBeforeSave', 'passParams' => true);
		return $events;
	}

	/**
	 * redefine and call parent so we can append our behavior checking code
	 * allows us to check for overriding behaviors without needing to overriding any actual code
	 */
	/**
	 * Create a set of associations.
	 *
	 * @return void
	 */
	protected function _createLinks() {
		parent::_createLinks();

		// Evoluted Amend. Loop and check for any plugins in behaviours
		// Check these haven't been overwritten
		if (! empty($this->actsAs) && ! $this->behaviorsChecked) {
			$this->actsAs = EvClassRegistry::checkOverridingItems('behaviors', $this->actsAs);
			$this->behaviorsChecked = true;
		}
	}

	/**
	 * Setup the model
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		// Use the given alias for models if the given alias doesn't match the class name.
		// This will usually be either the base model alias if initialised through EvClassRegistry
		// as an override (e.g. EvCore.User will have an alias of User) or a relationship alias.
		if (is_array($id) && isset($id['alias']) && isset($id['class'])) {
			$alias = $id['alias'];

			if (isset($id['plugin'])) {
				$alias = $id['plugin'] . '.' . $id['alias'];
			}

			if ($id['class'] !== $alias) {
				$this->alias = $id['alias'];
			}
		}

		parent::__construct($id, $table, $ds);

		if ($this->displayName == null) {
			$this->displayName = $this->alias;
		}

		$this->initialiseAssociateModels();
	}

	/**
	 * Initialises the associated models such as Image, Document and Block
	 */
	public function initialiseAssociateModels() {
		// Set up model associations for images automagically based on the
		// image slots.
		$this->_associateModels('EvCore.Image', $this->imageSlots);

		// Set up model associations for documents automagically based on the
		// document slots.
		$this->_associateModels('EvCore.Document', $this->documentSlots);
	}

	/**
	 * Automagically associate attachments and blocks with the model based on the
	 * number of slots defined. This method is called from the constructor.
	 *
	 * @param  string $modelAlias either 'Document', 'Image' or 'Block'
	 * @param  mixed  $modelSlots  array or integer defining the number of slots required
	 */
	protected function _associateModels($modelAlias, $modelSlots, $type = 'attachment_type') {
		list($plugin, $modelName) = pluginSplit($modelAlias);

		if (is_array($modelSlots)) {
			foreach ($modelSlots as $section => $slots) {
				if ($slots != 0) {
					$model = ($section == 'main') ? $modelName : Inflector::camelize($section) . $modelName;

					$conditions = array(
						'model' => $this->name,
						$type => $model
					);

					// check for custom conditions
					if (! empty($slots['conditions'])) {
						$conditions = array_merge(
							$conditions,
							$slots['conditions']
						);
					}

					$this->hasMany[$model] = array(
						'className' => $modelAlias,
						'foreignKey' => 'model_id',
						'conditions' => $conditions,
						'dependent' => true,
						'cascade' => true
					);
				}
			}

		} elseif ($modelSlots != 0) {

			$this->hasMany[$modelName] = array(
				'className' => $modelAlias,
				'foreignKey' => 'model_id',
				'conditions' => array(
					'model' => $this->name,
					$type => $modelName
				),
				'dependent' => true,
				'cascade' => true
			);

		}

		return;
	}

/**
 * Normalise attachment slots. Converts attachment slots into a standard array format.
 *
 * @param string $type Attachment type, e.g. 'document' or 'image'
 * @return array Attachment slots
 */
	public function normaliseAttachmentSlots($type = 'document') {
		$normalisedSlots = [];
		$attachmentSlots = $type . 'Slots';
		$defaultModel = Inflector::camelize($type);
		if (is_array($this->$attachmentSlots)) {
			foreach ($this->$attachmentSlots as $section => $slots) {
				$model = $section === 'main' ? $defaultModel : Inflector::camelize($section) . $defaultModel;
				if (is_array($slots)) {
					$normalisedSlots[$model] = $slots;
				} else {
					$normalisedSlots[$model] = [
						'slots' => $slots,
					];
				}
			}
		} elseif (! empty($this->$attachmentSlots)) {
			$normalisedSlots[$defaultModel] = [
				'slots' => $this->$attachmentSlots,
			];
		}

		return $normalisedSlots;
	}

	/**
	 * Override in your on Model to customise
	 */
	public function validate() {
		return $this->validate;
	}

	/**
	 * define a generate paginateCount to get round cake doing a count on a model
	 * and that models behaviours / callbacks being triggered. Was causing slow downs with this like
	 * MetaData plugin and bringing out every single meta data record in the system!
	 * Simply do the count and set the conditions but disable callbacks
	 *
	 * @param array $conditions
	 * @param boolean $recursive
	 * @param array $extra
	 * @return array
	 */
	public function paginateCount($conditions = null, $recursive = 0, $extra = array()) {
		$params = array_merge(
			array(
				'conditions' => $conditions,
				'callbacks' => false
			),
			$extra
		);
		// If we're containing associates we only want to keep one-to-one relations that we may be
		// enforcing conditions on (for example an admin filter) to prevent issues with unknown
		// columns in the generated query. We remove other contains as these come at a cost to
		// performance.
		if (!empty($params['contain'])) {
			$params['contain'] = array_filter($params['contain'], function ($val) {
				$belongsTo = array_keys($this->belongsTo);
				$hasOne = array_keys($this->hasOne);
				return in_array($val, $belongsTo + $hasOne);
			});
		}

		return $this->find(
			'count',
			$params
		);
	}

/**
 * Called before each find operation. Return false if you want to halt the find
 * call, otherwise return the (modified) query data.
 *
 * @param array $query Data used to execute this query, i.e. conditions, order, etc.
 * @return mixed true if the operation should continue, false if it should abort; or, modified
 *  $query to continue with new $query
 * @link https://book.cakephp.org/2.0/en/models/callback-methods.html#beforefind
 */
	public function beforeFind($query) {
		$parentQuery = parent::beforefind($query);

		if ($parentQuery === false) {
			return $parentQuery;
		} elseif (is_array($parentQuery)) {
			//Parent modified query
			$query = $parentQuery;
		}

		if (!empty($query['virtualFields'])) {
			$this->_originalFindVirtualFields = $this->virtualFields;

			foreach ($query['virtualFields'] as $field => $sql) {
				$this->virtualFields[$field] = $sql;
			}

			unset($query['virtualFields']);
		}

		return $query;
	}

/**
 * Called after each find operation. Can be used to modify any results returned by find().
 * Return value should be the (modified) results.
 *
 * @param mixed $results The results of the find operation
 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
 * @return mixed Result of the find operation
 * @link https://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
 */
	public function afterFind($results, $primary = false) {
		$result = parent::afterfind($results, $primary);

		if (!empty($this->_originalFindVirtualFields)) {
			$this->virtualFields = $this->_originalFindVirtualFields;
		}

		return $results;
	}

/**
 * Before validate callback
 * @param array $options Callback options
 * @return bool True if data validates, false if it fails
 */
	public function beforeValidate($options = []) {
		// Attach any validation rules for the model's document and image slots.
		$attachmentSlots = array_merge(
			$this->normaliseAttachmentSlots(),
			$this->normaliseAttachmentSlots('image')
		);
		foreach ($attachmentSlots as $model => $attachment) {
			if (! empty($attachment['validate'])) {
				foreach ($attachment['validate'] as $field => $rules) {
					$this->$model->validator()->add($field, $rules);
				}
			}
		}

		return parent::beforeValidate($options);
	}

	/**
	 * Model pre-save logic
	 */
	public function beforeSave($options = array()) {
		// remove trailing empty paragraph tag from content
		if (isset($this->data[$this->alias]['content']) && !empty($this->data[$this->alias]['content'])) {

			$this->data[$this->alias]['content'] = preg_replace("/(<p[^>]*>[\s|&nbsp;]*<\/p>\n?)*$/", '', $this->data[$this->alias]['content']);
		}

		return true;
	}

/**
 * Redefined saveAll to add in beforeBeforeSave callback allows related data to be modified and afterAfterSave
 * callback to add functionality once all data has been saved.
 *
 * @param array $data Record data to save. This can be either a numerically-indexed array (for saving multiple
 *     records of the same type), or an array indexed by association name.
 * @param array $options Options to use when saving record data, See $options above.
 * @return mixed If atomic: True on success, or false on failure.
 *    Otherwise: array similar to the $data array passed, but values are set to true/false
 *    depending on whether each record saved successfully.
 * @see Model::saveAll.
 */
	public function saveAll($data = array(), $options = array()) {
		$saveCommon = $this->_saveCommon($data, $options);
		$data = $saveCommon['data'];
		$options = $saveCommon['options'];

		$saveResult = parent::saveAll($data, $options);

		if (!empty($saveResult)) {
			$this->_saveAfterCommon($data, $options, $saveResult);
		}

		return $saveResult;
	}

/**
 * Redefined saveMany to add in beforeBeforeSave callback allows related data to be modified and afterAfterSave
 * callback to add functionality once all data has been saved.
 *
 * @param array $data Record data to save. This should be a numerically-indexed array
 * @param array $options Options to use when saving record data, See $options above.
 * @return mixed If atomic: True on success, or false on failure.
 *    Otherwise: array similar to the $data array passed, but values are set to true/false
 *    depending on whether each record saved successfully.
 * @throws PDOException
 * @see Model::saveMany().
 */
	public function saveMany($data = array(), $options = array()) {
		$saveCommon = $this->_saveCommon($data, $options);
		$data = $saveCommon['data'];
		$options = $saveCommon['options'];

		$saveResult = parent::saveMany($data, $options);

		if (!empty($saveResult)) {
			$this->_saveAfterCommon($data, $options, $saveResult);
		}

		return $saveResult;
	}

/**
 * Redefined saveAssociated to add in beforeBeforeSave callback allows related data to be modified and afterAfterSave
 * callback to add functionality once all data has been saved.
 *
 * @param array $data Record data to save. This should be an array indexed by association name.
 * @param array $options Options to use when saving record data, See $options above.
 * @return mixed If atomic: True on success, or false on failure.
 *    Otherwise: array similar to the $data array passed, but values are set to true/false
 *    depending on whether each record saved successfully.
 * @throws PDOException
 * @see Model::saveAssociated().
 */
	public function saveAssociated($data = array(), $options = array()) {
		$saveCommon = $this->_saveCommon($data, $options);
		$data = $saveCommon['data'];
		$options = $saveCommon['options'];

		$saveResult = parent::saveAssociated($data, $options);

		if (!empty($saveResult)) {
			$this->_saveAfterCommon($data, $options, $saveResult);
		}

		return $saveResult;
	}

/**
 * Common functionality for save[All|Associated|Many]. Fires beforeBeforeSave event/method.
 *
 * @param array $data The data being saved.
 * @param array $options The save options being used.
 * @return array An array containing the data to be saved and the options to save with.
 */
	protected function _saveCommon($data, $options) {
		if (empty($data)) {
			$data = $this->data;
		}

		//Record if the data is being created or updated for afterAfterSaves.
		$this->_createdOnSave = empty($data[$this->alias][$this->primaryKey]) && empty($data[$this->primaryKey]);

		/*
		 * manual callbacks for beforeBeforeSave for models and behaviors
		 * cannot be properly implemented by using events system
		 */
		if (method_exists($this, 'beforeBeforeSave')) {
			$data = $this->beforeBeforeSave($data, $options);

			if ($data === false) {
				return false;
			}
		}

		// loop and trigger any behaviors with the method
		$loadedBehaviors = $this->Behaviors->loaded();
		foreach ($loadedBehaviors as $behavior) {
			if (method_exists($this->Behaviors->{$behavior}, 'beforeBeforeSave')) {
				$data = $this->Behaviors->{$behavior}->beforeBeforeSave($this, $data, $options);
				if ($data === false) {
					return false;
				}
			}
		}

		return compact(array('data', 'options'));
	}

/**
 * Common functionality for save[All|Associated|Many]. Fires afterAfterSave event/method.
 *
 * @param array $data       The data that was saved.
 * @param array $options    The save options that were used to save.
 * @param array $saveResult The result of the save, which models save or failed.
 * @return void.
 */
	protected function _saveAfterCommon($data, $options, $saveResult) {
		/*
		 * manual callbacks for afterAfterSave for models and behaviors
		 * cannot be properly implemented by using events system
		 */
		if (method_exists($this, 'afterAfterSave')) {
			$this->afterAfterSave($this->_createdOnSave, $data, $options);
		}

		// loop and trigger any behaviors with the method
		$loadedBehaviors = $this->Behaviors->loaded();
		foreach ($loadedBehaviors as $behavior) {
			if (method_exists($this->Behaviors->{$behavior}, 'afterAfterSave')) {
				$this->Behaviors->{$behavior}->afterAfterSave($this, $this->_createdOnSave, $data, $options);
			}
		}
	}

	/**
	 * get the list of brands for drop down menu
	 *
	 * @param 	array 	array of query params
	 * @return 	array
	 */
	public function getForDropDown($query = array()) {
		if ($this->hasField('is_active')) {
			$query['conditions'][$this->alias . '.is_active'] = 1;
		}

		if ($this->hasField('sequence')) {
			$query = $this->_addOrderToQuery($query, $this->alias . '.sequence ASC');
		}

		if (!empty($this->displayField) && $this->hasField($this->displayField)) {
			$query = $this->_addOrderToQuery($query, $this->alias . '.' . $this->displayField . ' ASC');
		}

		$query['callbacks'] = false;

		return $this->find(
			'list',
			$query
		);
	}

/**
 * Add order params to a query array. If the query doesn't have any order currently set then add it as an array.
 * If order is set then check if it is an array or a string before appending the new order.
 *
 * @param array $query Current query array.
 * @param string $order The field and direction to order the query by.
 * @return array The query with the order added.
 */
	protected function _addOrderToQuery($query, $order) {
		if (empty($query['order'])) {
			$query['order'][] = $order;
		} else {
			if (is_array($query['order']) && !in_array($order, $query['order'])) {
				$query['order'][] = $order;
			} elseif (strpos($query['order'], $order) === false) {
				$query['order'] .= ', ' . $order;
			}
		}

		return $query;
	}

	/**
	 * convience wrapper replacement for field()
	 * field() still passes to callbacks (behaviors as well) - in most instances
	 * this is undesired. Emulate this without callbacks
	 *
	 * @param 	string 		The field we wish to retrieve
	 * @param 	array 		Array of conditions to search on.
	 * @param 	string 		Sql Order Condition
	 * @return 	string 		Value from the first returned result
	 */
	public function fieldOnly($field, $conditions = array(), $order = '') {
		$data = $this->find(
			'first',
			array(
				'fields' => $field,
				'conditions' => $conditions,
				'order' => $order,
				'callbacks' => false
			)
		);
		return Hash::get($data, $this->alias . '.' . $field); // will return null if no $data found
	}

	/**
	 * Base code for retrieving a single record for edit or view
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	protected function _getItem($id, $query = array()) {
		$query = array_merge(
			array(
				'getRoutes' => true,
				'getMenuItem' => true,
				'customFields' => true,
				'customFieldsSyntaxFormat' => false
			),
			$query
		);

		$alias = $this->alias;

		if (!empty($query['key'])) {
			// Search using an alternative key
			$queryKey = $query['key'];
			unset($query['key']);

			$query['conditions'][$alias . '.' . $queryKey] = $id;
		} else {
			$query['conditions'][$alias . '.id'] = $id;
		}

		if ($this->hasField('is_removed')) {
			$query['conditions'][$this->alias . '.is_removed'] = false;
		}

		if ($this->hasBehavior('EvTemplates.Template')) {
			if (!empty($queryKey)) {
				$templateId = intval($this->fieldOnly('template_id', array($alias . '.' . $queryKey => $id)));
			} else {
				$templateId = intval($this->fieldOnly('template_id', array($alias . '.id' => $id)));
			}

			if ($templateId > 0) {
				$this->setupAttachmentRelationships($templateId);
			}
		}

		$query = array_merge_recursive($query, $this->_containAssociateModel('Image', $this->imageSlots));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Document', $this->documentSlots));

		$data = $this->find('first', $query);

		// We only want to get the route if it has Routable and is also on Edit pages
		if ($query['getRoutes'] && $this->hasBehavior('Routable.Routable')) {
			$data['Route'] = $this->getRoute($data); // method on Routable Behaviour
		}

		// We only want to get the menu item if it has Navigatable and is also on Edit pages
		if ($query['getMenuItem'] && $this->hasBehavior('EvNavigation.Navigatable')) {
			$data['Menu'] = $this->getMenuItem($data); // method on Navigatable Behaviour
		}

		return $data;
	}

	protected function _containAssociateModel($modelAlias, $modelSlots) {
		$query = array();

		if (is_array($modelSlots)) {
			foreach ($modelSlots as $section => $slots) {
				if ($slots != 0) {
					$model = ($section == "main") ? $modelAlias : Inflector::camelize($section) . $modelAlias;
					$query['contain'][$model] = array(
						'order' => 'sequence'
					);
				}
			}
		} elseif (is_numeric($modelSlots) && $modelSlots != 0) {
			$query['contain'][$modelAlias] = array(
				'order' => 'sequence'
			);
		}

		return $query;
	}

	/**
	 * Wrapper for tge containAssociateModel method, is protected so provide array of alias => slots
	 * to be able to call method
	 *
	 * @param 	array
	 * @return 	array
	 */
	public function containSlotAssociations($slots) {
		$query = array();
		if (! empty($slots)) {
			foreach ($slots as $alias => $slots) {
				$query = array_merge_recursive($query, $this->_containAssociateModel($alias, $slots));
			}
		}

		return $query;
	}

	/**
	 * Query used to retrieve a record ready for edit
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $params = array()) {
		$params['readForEdit'] = true;
		return $this->_getItem($id, $params);
	}

/**
 * Query used to retrieve a record ready for view.
 *
 * By default calls readForEdit but can be extended to bring out more
 * complicated data.
 *
 * Use `readForViewOrFail` instead of this method to throw an exception if no record is found.
 *
 * @param int|string $id Primary key
 * @param array $params The db query array - can be used to pass in additional parameters such as contain
 * @return array
 */
	public function readForView($id, $params = array()) {
		if ($this->hasField('is_active')) {
			$params['conditions'][$this->escapeField('is_active')] = true;
		}

		$params['getRoutes'] = false;
		$params['getMenuItem'] = false;
		$params['customFieldsSyntaxFormat'] = true;
		$params['readForView'] = true;

		return $this->_getItem($id, $params);
	}

/**
 * Retrieve a record for view or throw a not found exception.
 * @param int|string $id Primary key
 * @param array $params Query array
 * @return array Data
 */
	public function readForViewOrFail($id, $params = array()) {
		$data = $this->readForView($id, $params);
		if (empty($data)) {
			throw new NotFoundException(InflectorExt::camelToHumanize($this->name) . ' not found');
		}

		return $data;
	}

	/**
	 * Query used to retrieve a record ready for admin view.
	 *
	 * By default calls readForEdit but can be extended to bring out more
	 * complicated data.
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForAdminView($id, $params = array()) {
		return $this->readForEdit($id, $params);
	}

	/**
	 * Query used to retrieve a listing of records ready for template
	 *
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForIndex($query = array()) {
		if ($this->hasField('is_active')) {
			$query['conditions'][$this->alias . '.is_active'] = true;
		}

		if ($this->hasField('is_removed')) {
			$query['conditions'][$this->alias . '.is_removed'] = false;
		}

		$query = array_merge_recursive($query, $this->_containAssociateModel('Image', $this->imageSlots));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Document', $this->documentSlots));

		return $this->find('all', $query);
	}

	/**
	 * Retrieve records for re-ordering
	 * @param array $params Additional query parameters
	 * @return array
	 */
	public function readForReorder($params = []) {
		if ($this->hasField('orderno')) {
			$params['order'] = $this->alias . '.orderno';
		} elseif ($this->hasField('sequence')) {
			$params['order'] = $this->alias . '.sequence';
		} else {
			$params['order'] = $this->alias . '.lft';
		}

		if ($this->hasField('parent_id')) {
			if ($params['order'] === 'lft') {
				$data = $this->find('threaded', $params);
			} else {
				$params['conditions'][] = [
					'OR' => [
						$this->alias . '.parent_id IS NULL',
						$this->alias . '.parent_id' => 0
					]
				];

				$data = $this->find('all', $params);
				$this->_getChildren($data, $params['order']);
			}
		} else {
			$data = $this->find('all', $params);
		}

		return $data;
	}

	/**
	 * Recursively finds all children for reorder - Takes array by reference
	 * @param array $data Data to find children of, passed by reference
	 * @param string $orderKey
	 * @return void
	 */
	protected function _getChildren(&$data, $orderKey) {
		foreach ($data as &$row) {
			$children = $this->find('all', array(
				'conditions' => array(
					'parent_id' => $row[$this->alias]['id']
				),
				'order' => $orderKey
			));
			$this->_getChildren($children, $orderKey);
			$row[$this->alias]['children'] = $children;
		}
		return;
	}

	/**
	 * Checks if the record is protected
	 * Removes associated images and documents
	 *
	 * @param boolean $cascade true if the delete is to cascade to children
	 * @return boolean true if the opperation was successful, otherwise false
	 * @see Model::beforeDelete()
	 */
	public function beforeDelete($cascade = true) {
		if ($this->hasField('is_protected') && $this->field('is_protected')) {
			return false;
		}

		return parent::beforeDelete($cascade);
	}

	/**
	 * Returns a list of dependant records (usually displayed on the delete dialog)
	 *
	 * @param integer $id ID of the row we are checking dependencies for
	 */
	public function findDependents($id) {
		$dependents = array();

		$models = array_merge_recursive($this->hasOne, $this->hasMany);

		foreach ($models as $model => $attr) {
			//If model is set as not dependent then dont need to look in it for dependents
			if ((isset($attr['dependent']) && !$attr['dependent']) || (isset($attr['cascade']) && $attr['cascade'])) {
				continue;
			}

			$params = array();

			//If default conditions are set on the relationship we need to listen to them
			if (isset($attr['conditions'])) {
				$params['conditions'] = $attr['conditions'];
			}

			$params['conditions'][$model . "." . $attr['foreignKey']] = $id;

			$found = $this->$model->find('all', $params);

			foreach ($found as $item) {
				$value = $item[$model][$this->$model->displayField];
				$dependents[$model][$item[$model]['id']] = $value;
			}
		}

		return $dependents;
	}

	/**
	 * Custom validate rule, checks if two fields are identical
	 * (useful for comparing password and confirm_password)
	 *
	 * @param array $check Field to trigger the check
	 * @param array $otherField Other field to compare with
	 */
	public function checkMatches($check, $otherField) {
		$value = array_pop($check);

		return $value == $this->data[$this->alias][$otherField];
	}

	/**
	 * Checks if a combination of field values are unique in the DB
	 *
	 * @param $check array field the check is triggered by
	 * @param $fields array list of fields that form part of the unique key
	 *
	 * @return boolean true if combination doesn't exists in the database, otherwise false
	 */
	public function checkCombinationUnique($check, $fields) {
		$fieldToTest = key($check);
		$value = array_pop($check);

		// build up the list of fields and their values into a condition array
		$conditions = array(
			$this->alias . "." . $fieldToTest => $value
		);

		foreach ($fields as $field) {
			$conditions[$this->alias . "." . $field] = $this->data[$this->alias][$field];
		}

		// don't include the current row in the duplicate check
		if (isset($this->data[$this->alias][$this->primaryKey]) && !empty($this->data[$this->alias][$this->primaryKey])) {
			$conditions[$this->alias . "." . $this->primaryKey . " <>"] = $this->data[$this->alias][$this->primaryKey];
		}

		// if we find a match, the combination already exists
		$found = $this->find('first', array(
			'conditions' => $conditions
		));

		return empty($found);
	}

	/**
	 * Toggles a field on or off.
	 *
	 * @param string $field field to toggle
	 * @param integer $id ID of row
	 *
	 * @return result of updateAll()
	 */
	public function toggleField($field, $id = null) {
		$id = empty($id) ? $this->id : $id;

		$field = $this->escapeField($field);

		$fields = array($field => "NOT $field");
		$conditions = array($this->escapeField() => $id);

		$result = $this->updateAll($fields, $conditions);

		// Dispatch an event to allow behaviors to act on this
		$Event = new CakeEvent('Model.afterToggle', $this, array(
			'model' => $this,
			'id' => $id,
			'field' => $field
		));
		$this->getEventManager()->dispatch($Event);

		return $result;
	}

	/**
	 * Custom delete method that checks whether the current model has an is_removed flag
	 * to mark as true rather than delete before actually deleting content
	 * @param  int  $id
	 * @param  boolean $cascade
	 * @return boolean
	 */
	public function delete($id = null, $cascade = true) {
		if ($this->hasField('is_removed')) {

			$this->id = $id;
			$this->saveField('is_removed', true);

			if ($this->hasField('is_active')) {
				$this->saveField('is_active', false);
			}

			$result = true;

		} else {
			$result = parent::delete($id, $cascade);
		}

		return $result;
	}

	/**
	 * Gets the next order number in a sequence.
	 *
	 * @param array $params
	 * @return integer
	 */
	public function getNextSequenceNo($params = array()) {
		$defaults = array(
			'order' => 'sequence DESC',
			'limit' => 1
		);

		$params = array_merge($params, $defaults);

		$result = $this->find('first', $params);

		return $result ? $result[$this->alias]['sequence'] + 1 : 0;
	}

	public function getLastQuery() {
		$log = $this->getDataSource()->getLog(false, false);
		// return the first element of the last array (i.e. the last query)
		return $log['log'][(count($log['log']) - 1)];
	}

	/**
	 * § Behaviours
	 */

	/**
	 * Helper function to check if the model has a particular behaviour
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function hasBehavior($behavior) {
		if (empty($this->actsAs)) {
			return false;
		}

		$setup = $this->setupBehavior($behavior);
		extract($setup);

		$behaviorKeys = array_keys($this->actsAs);
		$behaviorValues = array_values($this->actsAs);
		$behaviorClassNames = Hash::extract($this->actsAs, '{s}.className');

		return in_array($behavior, array_merge($behaviorKeys, $behaviorValues, $behaviorClassNames), true);
	}

	/**
	 * Helper function to remove a behaviour from a model
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function removeBehavior($behavior) {
		$setup = $this->setupBehavior($behavior);
		extract($setup);

		if ($this->hasBehaviour($behavior)) {
			unset($this->actsAs[$behavior]);

			$this->Behaviors->unload($behavior);
		}

		return true;
	}

	/**
	 * Helper function to add a behaviour to a model
	 *
	 * @param string $behaviour name of the behaviour
	 * @param array $config (optional) behaviour's config
	 * @param boolean $overrideBehaviour (optional) pass as true to reload the behaviour if already loaded
	 * @return boolean
	 */
	public function addBehavior($behavior, $config = array(), $overrideBehavior = false) {
		$setup = $this->setupBehavior($behavior, $config);
		extract($setup);

		if ($overrideBehavior === true || ! $this->hasBehaviour($behavior)) {
			$this->actsAs[$behavior] = $config;
			$this->Behaviors->load($behavior, $config);
		}

		return true;
	}

	/**
	 * backwards compatibility methods
	 */
	public function hasBehaviour($behavior) {
		return $this->hasBehavior($behavior);
	}

	public function removeBehaviour($behavior) {
		return $this->removeBehavior($behavior);
	}

	public function addBehaviour($behavior, $config = array(), $overrideBehavior = false) {
		return $this->addBehavior($behavior, $config, $overrideBehavior);
	}

	/**
	 * setup a behavior
	 *
	 * @param 	string 	Behavior to load
	 * @param 	array 	(optionval) Behaviors config
	 * @return  array  	array containing behavior / array config
	 */
	public function setupBehavior($behavior, $config = array()) {
		// check for a plugin / overriding file
		$origBehavior = $behavior;

		if (! empty($config['className'])) {
			$behavior = $config['className'];
		}

		$behavior = EvClassRegistry::findOverrideBehavior($behavior);
		$config['className'] = $behavior;

		// as we have set the className, we need to chop up the original
		// helper call if it was a plugin to set the alias correctly.
		$behavior = EvClassRegistry::getNewArrayKey($origBehavior);

		return array(
			'behavior' => $behavior,
			'config' => $config
		);
	}

	/**
	 * get a constant from the object
	 *
	 * @param 	string 	$constant 	The constant to retrieve
	 * @return 	mixed
	 */
	public function getConstant($constant) {
		return constant(get_class($this) . '::' . $constant);
	}

/**
 * Validates the given reCAPTCHA field against the Google reCAPTCHA API
 *
 * @param array $check Contains the field to validate against
 * @return bool
 */
	public function customValidateReCaptcha($check) {
		$userReCaptchaSecret = Configure::read('EvCore.userReCaptcha.secret');

		$recaptcha = new \ReCaptcha\ReCaptcha($userReCaptchaSecret, new \ReCaptcha\RequestMethod\CurlPost());
		$resp = $recaptcha->verify(current($check), $_SERVER['REMOTE_ADDR']);

		return $resp->isSuccess();
	}

/**
 * Merge two sets of query parameters together. The new parameters being added should take precedence over the current
 * parameters. This means that any new parameters should replace any parameters already defined and any query
 * parameters that are dependent on being ordered (order, group by, etc.) are placed before the current parameters.
 *
 * The following query parameters are merged:
 * - fields:     Provided as a list or as field => field name pairings, merged recursively.
 * - table:      Provided as a string, current value is replaced by the new value.
 * - alias:      Provided as a string, current value is replaced by the new value.
 * - joins:      Provided as an array of joins, merged recursively, similar joins are kept.
 * - conditions: Provided as a list or as field => condition pairings, merged recursively.
 * - order:      Provided as a string or as field => direction pairings. Array merged, duplicates removed, new orders
 *               take precedence over current orders. If the current order is a string then the merged order will also
 *               be a string even if the new order is an array. Only simple orders are supported, conditional orders
 *               will need to be merged manually.
 * - group:      Provided as a string or as a list of fields. Array merged, duplicates removed, new groups will take
 *               precedence over current groups. If the current group is a string then the merged group will also be a
 *               string even if the new group is an array. Only simple groups are supported, conditional groups will
 *               need to be merged manually.
 * - having:     Provided as a list of as field => condition pairings, merged recursively.
 * - limit:      Provided as an integer, current value is replaced by the new value.
 * - offset:     Provided as an integer, current value is replaced by the new value.
 * - page:       Provided as an integer, current value is replaced by the new value.
 * - recursive:  Provided as an integer, current value is replaced by the new value.
 * - callbacks:  Provided as a boolean or string, current value is replaced by the new value.
 * - contain:    Provided as an array, merged recursively.
 *
 * Single values are merged manually so that if any custom query parameters are passed through and require being merged
 * with their own requirements, they aren't merged here.
 *
 * @param array $currentParams The current query parameters.
 * @param array $newParams     The new parameters to be merged in.
 * @return array Merged query parameters.
 */
	public function mergeQueryParams($currentParams, $newParams) {
		//FIELDS
		if (!empty($newParams['fields'])) {
			//Merge fields normally
			if (empty($currentParams['fields'])) {
				$currentParams['fields'] = $newParams['fields'];
			} else {
				$currentParams['fields'] = array_merge_recursive(
					$currentParams['fields'],
					$newParams['fields']
				);
			}
		}

		unset($newParams['fields']);

		//TABLE
		if (isset($newParams['table'])) {
			//Table is a single value so just replace the current value.
			$currentParams['table'] = $newParams['table'];
		}

		unset($newParams['table']);

		//ALIAS
		if (isset($newParams['alias'])) {
			//Alias is a single value so just replace the current value.
			$currentParams['alias'] = $newParams['alias'];
		}

		unset($newParams['alias']);

		//JOINS
		if (!empty($newParams['joins'])) {
			//Joins are merged normally. This makes new joins appended to the current joins even if they are similar.
			if (empty($currentParams['joins'])) {
				$currentParams['joins'] = $newParams['joins'];
			} else {
				$currentParams['joins'] = array_merge_recursive(
					$currentParams['joins'],
					$newParams['joins']
				);
			}
		}

		unset($newParams['joins']);

		//CONDITIONS
		if (!empty($newParams['conditions'])) {
			//Merge conditions normally
			if (empty($currentParams['conditions'])) {
				$currentParams['conditions'] = $newParams['conditions'];
			} else {
				$currentParams['conditions'] = array_merge_recursive(
					$currentParams['conditions'],
					$newParams['conditions']
				);
			}
		}

		unset($newParams['conditions']);

		//ORDER
		if (!empty($newParams['order'])) {
			//Convert the new order parameter into an array if it has been provided as a string.
			if (is_string($newParams['order'])) {
				$newOrder = explode(',', $newParams['order']);

				$convertedOrder = [];
				foreach ($newOrder as $order) {
					$order = trim($order);
					list($orderField, $orderDirection) = explode(' ', $order, 2);

					$convertedOrder[$orderField] = $orderDirection;
				}

				$newParams['order'] = $convertedOrder;
			}

			if (empty($currentParams['order'])) {
				$currentParams['order'] = $newParams['order'];
			} else {
				//Convert the current order parameter into an array if it has been provided as a string.
				$convertOrderToString = false;
				if (is_string($currentParams['order'])) {
					$convertOrderToString = true;
					$currentOrder = explode(',', $currentParams['order']);

					$convertedOrder = [];
					foreach ($currentOrder as $order) {
						$order = trim($order);
						list($orderField, $orderDirection) = explode(' ', $order, 2);

						$convertedOrder[$orderField] = $orderDirection;
					}

					$currentParams['order'] = $convertedOrder;
				}

				/*
				 * Merge the current parameters into the new parameters. The parameters are merged this way so that the
				 * new parameters take precedence. The current parameters are only merged in if they are different to
				 * avoid any of the current parameters overriding the new parameters.
				 */
				$currentParams['order'] = array_merge(
					$newParams['order'],
					array_diff_key(
						$currentParams['order'],
						$newParams['order']
					)
				);

				if ($convertOrderToString) {
					$convertedOrder = [];
					foreach ($currentParams['order'] as $field => $direction) {
						$convertedOrder[] = $field . ' ' . $direction;
					}
					$currentParams['order'] = implode(', ', $convertedOrder);
				}
			}
		}

		unset($newParams['order']);

		//GROUP
		if (!empty($newParams['group'])) {
			if (empty($currentParams['group'])) {
				$currentParams['group'] = $newParams['group'];
			} else {
				//Convert the current group parameter into an array if it has been provided as a string.
				$convertGroupToString = false;
				if (is_string($currentParams['group'])) {
					$convertGroupToString = true;
					$currentParams['group'] = explode(',', $currentParams['group']);
				}

				/*
				 * Merge the current parameters into the new parameters. The parameters are merged this way so that the
				 * new parameters take precedence. The current parameters are only merged in if they are different to
				 * avoid any of the current parameters overriding the new parameters.
				 */
				$currentParams['group'] = array_merge(
					$newParams['group'],
					array_diff(
						$currentParams['group'],
						$newParams['group']
					)
				);

				if ($convertGroupToString) {
					$currentParams['group'] = implode(', ', $currentParams['group']);
				}
			}
		}

		unset($newParams['group']);

		//HAVING
		if (!empty($newParams['having'])) {
			//Merge having normally
			if (empty($currentParams['having'])) {
				$currentParams['having'] = $newParams['having'];
			} else {
				$currentParams['having'] = array_merge_recursive(
					$currentParams['having'],
					$newParams['having']
				);
			}
		}

		unset($newParams['having']);

		//LIMIT
		if (!empty($newParams['limit'])) {
			//Limit is a single value so just replace the current value.
			$currentParams['limit'] = $newParams['limit'];
		}

		unset($newParams['limit']);

		//OFFSET
		if (!empty($newParams['offset'])) {
			//Offset is a single value so just replace the current value.
			$currentParams['offset'] = $newParams['offset'];
		}

		unset($newParams['offset']);

		//PAGE
		if (!empty($newParams['page'])) {
			//Page is a single value so just replace the current value.
			$currentParams['page'] = $newParams['page'];
		}

		unset($newParams['page']);

		//RECURSIVE
		if (isset($newParams['recursive'])) {
			//Recursive is a single value so just replace the current value.
			$currentParams['recursive'] = $newParams['recursive'];
		}

		unset($newParams['recursive']);

		//CALLBACKS
		if (!empty($newParams['callbacks'])) {
			//Callbacks is a single value so just replace the current value.
			$currentParams['callbacks'] = $newParams['callbacks'];
		}

		unset($newParams['callbacks']);

		//LOCK
		if (!empty($newParams['lock'])) {
			//Lock is a single value so just replace the current value.
			$currentParams['lock'] = $newParams['lock'];
		}

		unset($newParams['lock']);

		//CONTAIN
		if (!empty($newParams['contain'])) {
			//Merge contain normally
			if (empty($currentParams['contain'])) {
				$currentParams['contain'] = $newParams['contain'];
			} else {
				$currentParams['contain'] = array_merge_recursive(
					$currentParams['contain'],
					$newParams['contain']
				);
			}
		}

		unset($newParams['contain']);

		/*
		 * We have removed all known query params after they have been merged, if there are any remaining query params
		 * then merge them in normally as they might be required custom query params or non-query params used before
		 * the query is performed (readForView, getRoutes, etc.)
		 */
		if (!empty($newParams)) {
			$currentParams = array_merge_recursive($currentParams, $newParams);
		}

		return $currentParams;
	}
}
