<?php
/**
 * Copyable Behavior class file.
 *
 * Adds ability to copy a model record, including all hasMany and hasAndBelongsToMany
 * associations. Relies on Containable behavior, which this behavior will attach
 * on the fly as needed.
 *
 * HABTM relationships are just duplicated in the join table, while hasMany and hasOne
 * records are recursively copied as well.
 *
 * Usage is straightforward:
 * From model: $this->copy($id); // id = the id of the record to be copied
 * From controller: $this->MyModel->copy($id);
 *
 * @filesource
 * @author          Jamie Nay
 * @copyright       Jamie Nay
 * @license         http://www.opensource.org/licenses/mit-license.php The MIT License
 * @link            http://github.com/jamienay/copyable_behavior
 *
 * @author          Mike Barlow
 * @link            https://github.com/snsmurf/copyable_behavior
 */
class CopyableBehavior extends ModelBehavior {

/**
 * Behavior settings
 *
 * @access public
 * @var array
 */
	public $settings = [];

/**
 * Array of contained models.
 *
 * @access public
 * @var array
 */
	public $contain = [];

/**
 * The full results of Model::find() that are modified and saved as a new copy.
 *
 * @access public
 * @var array
 */
	public $record = [];

/**
 * Default values for settings.
 *
 * - recursive: whether to copy hasMany and hasOne records,
 * - recurse_level: how many levels to include hasMany and hasOne records,
 * - habtm: whether to copy hasAndBelongsToMany associations,
 * - stripFields: fields to strip during copy process,
 * - append_display_field: The string to append to the display field, false to not append anything.
 *
 * @access protected
 * @var array
 */
	protected $_defaults = array(
		'recursive' => true,
		'recurse_level' => 1,
		'habtm' => true,
		'stripFields' => array('id', 'created', 'modified', 'lft', 'rght', 'dir'),
		'append_display_field' => false,
		'copyImages' => true,
		'copyDocuments' => true,
	);

/**
 * The model that is being copied.
 *
 * @var null|obj Null if not set, otherwise model object.
 */
	public $mainModel = null;

/**
 * The id of the record that is being copied.
 *
 * @var int
 */
	public $mainModelId = null;

/**
 * The id of the new record once copied.
 *
 * @var int
 */
	public $newModelId = null;

/**
 * Data to modify the record with before it is copied.
 *
 * @var array
 */
	protected $_copyData = [];

/**
 * The images that are currently saved on the main model.
 *
 * @var array
 */
	protected $_originalImages = [];

/**
 * The documents that are currently saved on the main model.
 *
 * @var array
 */
	protected $_originalDocuments = [];

/**
 * Configuration method.
 *
 * @param obj   $Model  Model object.
 * @param array $config Config array.
 * @access public
 * @return bool
 */
	public function setup(Model $Model, $config = array()) {
		$this->settings[$Model->alias] = array_merge($this->_defaults, $config);

		return true;
	}

/**
 * Copy a record. Associations can be copied with a recurseLevel > 0.
 *
 * @param obj $Model        Model object.
 * @param int $id           Integer model ID.
 * @param int $recurseLevel The amount of levels to include hasMany and hasOne records.
 * @access public
 * @return bool|array False if the copy failed. The new record if copy successful.
 */
	public function copy($Model, $id, $recurseLevel = 1) {
		$this->mainModel = $Model;

		$this->mainModelId = $id;

		$this->generateContain($this->mainModel, $recurseLevel);

		$this->_checkAttachmentCopy($this->mainModel, $this->mainModelId);

		$this->_setupRecord($this->mainModel, $this->mainModelId);

		if (empty($this->record)) {
			return false;
		}

		if (!$this->_convertData($this->mainModel)) {
			return false;
		}

		if ($recurseLevel > 1) {
			$options = ['deep' => true];
		} else {
			$options = [];
		}

		$copyResult = $this->_copyRecord($this->mainModel, $options);

		if ($copyResult) {
			$this->newModelId = $this->mainModel->id;

			$this->_copyAttachments($this->mainModel, $this->newModelId);
		}

		return $copyResult;
	}

/**
 * Wrapper method that combines the results of _recursiveChildContain() with the models' HABTM associations.
 *
 * @param obj $Model        Model object.
 * @param int $recurseLevel The amount of levels to include hasMany and hasOne records.
 * @access public
 * @return array An array of contains to copy along with the main record.
 */
	public function generateContain($Model, $recurseLevel) {
		if (!$this->_verifyContainable($Model)) {
			return false;
		}

		$this->contain = array();

		if ($this->settings[$Model->alias]['habtm']) {
			$this->contain = array_keys($Model->hasAndBelongsToMany);
		}

		$this->contain = array_merge($this->_recursiveChildContain($Model, $recurseLevel), $this->contain);

		return $this->contain;
	}

/**
 * Strips primary keys and other unwanted fields from hasOne and hasMany records.
 *
 * @param obj   $Model  Model object.
 * @param array $record The record to copy.
 * @access protected
 * @return array $record The converted record.
 */
	protected function _convertChildren($Model, $record) {
		if (!empty($Model->hasMany)) {
			foreach ($Model->hasMany as $modelAlias => $hasManyConfig) {
				if (!isset($record[$modelAlias]) || empty($record[$modelAlias])) {
					continue;
				}

				//Convert the data for each child
				foreach ($record[$modelAlias] as $childIndex => $childData) {
					$record[$modelAlias][$childIndex] = $this->_stripFields($Model->{$modelAlias}, $childData);

					if (array_key_exists($hasManyConfig['foreignKey'], $childData)) {
						unset($record[$modelAlias][$childIndex][$hasManyConfig['foreignKey']]);
					}

					$record[$modelAlias][$childIndex] = $this->_convertChildren(
						$Model->{$modelAlias},
						$record[$modelAlias][$childIndex]
					);
				}
			}
		}

		if (!empty($Model->hasOne)) {
			foreach ($Model->hasOne as $modelAlias => $hasOneConfig) {
				if (!isset($record[$modelAlias]) || empty($record[$modelAlias])) {
					continue;
				}

				$record[$modelAlias] = $this->_stripFields($Model->{$modelAlias}, $record[$modelAlias]);

				if (isset($record[$modelAlias][$hasOneConfig['foreignKey']])) {
					unset($record[$modelAlias][$hasOneConfig['foreignKey']]);
				}

				$record[$modelAlias] = $this->_convertChildren($Model->{$modelAlias}, $record[$modelAlias]);
			}
		}

		return $record;
	}

/**
 * Strips primary and parent foreign keys (where applicable) from $this->record in preparation for saving.
 *
 * @param object $Model Model object.
 * @access protected
 * @return array $this->record.
 */
	protected function _convertData($Model) {
		$this->record[$Model->alias] = $this->_stripFields($Model, $this->record[$Model->alias]);
		$this->record = $this->_convertHabtm($Model, $this->record);
		$this->record = $this->_convertChildren($Model, $this->record);
		return $this->record;
	}

/**
 * Loops through any HABTM results in $this->record and plucks out the join table info, stripping out the join
 * table primary key and the primary key of $Model. This is done instead of a simple collection of IDs of the
 * associated records, since HABTM join tables may contain extra information (sorting order, etc).
 *
 * @param obj   $Model  Model object.
 * @param array $record The record that is being copied.
 * @access private
 * @return array Modified $record.
 */
	protected function _convertHabtm($Model, $record) {
		if (!$this->settings[$Model->alias]['habtm']) {
			return $record;
		}

		foreach ($Model->hasAndBelongsToMany as $relationshipAlias => $relationship) {
			if (!isset($record[$relationshipAlias]) || empty($record[$relationshipAlias])) {
				continue;
			}

			list($withPlugin, $withClassName) = pluginSplit($relationship['with']);

			$joinInfo = Hash::extract($record[$relationshipAlias], '{n}.' . $withClassName);

			if (empty($joinInfo)) {
				continue;
			}

			foreach ($joinInfo as $joinKey => $joinVal) {
				$joinInfo[$joinKey] = $this->_stripFields($Model, $joinVal);

				if (array_key_exists($relationship['foreignKey'], $joinVal)) {
					unset($joinInfo[$joinKey][$relationship['foreignKey']]);
				}
			}

			$record[$relationshipAlias] = $joinInfo;
		}

		return $record;
	}

/**
 * Performs the actual creation and save.
 *
 * @param obj   $Model Model object
 * @param array $options The save options.
 * @access protected
 * @return mixed
 */
	protected function _copyRecord($Model, $options = []) {
		$this->_modifyRecord($Model);

		$Model->create();
		$Model->set($this->record);
		return $Model->saveAssociated($Model->data, $options);
	}

/**
 * Generates a contain array for Containable behavior by recursively looping through $Model->hasMany and
 * $Model->hasOne associations.
 *
 * @param obj $Model        Model object.
 * @param int $recurseLevel The amount of levels to include hasMany and hasOne records.
 * @access protected
 * @return array
 */
	protected function _recursiveChildContain($Model, $recurseLevel = false) {
		$contain = [];

		if (isset($this->settings[$Model->alias]) && !$this->settings[$Model->alias]['recursive']) {
			return $contain;
		}

		if ($recurseLevel > 1) {
			$children = array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));
			foreach ($children as $child) {
				$contain[$child] = $this->_recursiveChildContain($Model->{$child}, $recurseLevel - 1);
			}
		} else {
			return array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));

		}

		return $contain;
	}

/**
 * Strips unwanted fields from $record, taken from the 'stripFields' setting.
 *
 * @param obj   $Model  Model object.
 * @param array $record The record that is being copied.
 * @access protected
 * @return array
 */
	protected function _stripFields($Model, $record) {
		foreach ($this->settings[$this->mainModel->alias]['stripFields'] as $field) {

			// check if we 're defining model.field for stripping fields
			if (strpos($field, '.') !== false) {

				$ex = explode('.', $field);
				$stripModel = $ex['0'];
				$stripField = $ex['1'];

				// the models for the fields we are stripping don't match
				if ($stripModel != $Model->alias) {

					continue;
				}
			} else {
				$stripField = $field;
			}

			// check we have it and strip
			if (array_key_exists($stripField, $record)) {
				unset($record[$stripField]);
			}
		}

		return $record;
	}

/**
 * Attaches Containable if it's not already attached.
 *
 * @param obj $Model Model object.
 * @access protected
 * @return bool
 */
	protected function _verifyContainable($Model) {
		if (!$Model->Behaviors->attached('Containable')) {
			return $Model->Behaviors->attach('Containable');
		}

		return true;
	}

/**
 * Set data that can be used during the copy to modify the existing record before it is copied.
 *
 * @param obj   $Model The model object.
 * @param array $data  The data to make available during copying.
 * @return null.
 */
	public function setCopyData($Model, $data) {
		$this->_copyData = $data;
	}

/**
 * Modify the record that is being copied. Any modification of the record data that happens here is
 * applied to the new copy.
 *
 * @param obj $Model The model oject being copied.
 * @return void.
 */
	protected function _modifyRecord($Model) {
		// Option to append a string to the end of the display field.
		if (!empty($this->settings[$Model->alias]['append_display_field'])) {
			$this->record[$Model->alias][$Model->displayField] = $this->record[$Model->alias][$Model->displayField] . $this->settings[$Model->alias]['append_display_field'];
		}

		if (!empty($this->_copyData)) {
			$this->record = Hash::merge(
				$this->record,
				$this->stripAllIds($Model, $this->_copyData)
			);
		}
	}

/**
 * Setup a record to be copied. The record of the model with the provided id is found. If the
 * behavior is set up to add HABTM, HasMany and HasOne associations then they are also found
 * here.
 *
 * @param obj $Model  The model object being copied.
 * @param int $id     The model id of the record being copied.
 * @param array  $params Custom parameters to use when finding the record.
 * @return array The found record. Also stored in $this->record.
 */
	protected function _setupRecord($Model, $id, $params = []) {
		$defaultParams = [
			'conditions' => [
				$this->mainModel->alias . '.id' => $id,
			],
			'contain' => $this->contain,
		];

		$params = Hash::merge($defaultParams, $params);

		$this->record = $this->mainModel->find(
			'first',
			$params
		);

		return $this->record;
	}

/**
 * Remove the images from the contains array used in the _setupRecord() query.
 * All image slots are removed from the contains.
 *
 * @param obj $Model The model object being copied.
 * @return void.
 */
	protected function _removeImageContains($Model) {
		$this->_removeAttachmentTypeContains($Model, 'image');
	}

/**
 * Remove the documents from the contains array used in the _setupRecord() query.
 * All document slots are removed from the contains.
 *
 * @param obj $Model The model object being copied.
 * @return void.
 */
	protected function _removeDocumentContains($Model) {
		$this->_removeAttachmentTypeContains($Model, 'document');
	}

/**
 * Remove attachments from the contains array used in the _setupRecord() query.
 * All attachment slots are removed from the contains. Attachments can't be copied like other
 * assets as they are linked to assets which need duplicating and moving. This is handled
 * after the record has been copied.
 *
 * @param obj    $Model          The model object being copied.
 * @param string $attachmentType The attachment type to remove, e.g. image, document, variantImage, etc.
 * @return void.
 */
	protected function _removeAttachmentTypeContains($Model, $attachmentType) {
		$slots = $Model->normaliseAttachmentSlots($attachmentType);

		if (!empty($slots)) {
			foreach ($slots as $attachmentName => $attachmentSlot) {
				if (isset($this->contain[$attachmentName])) {
					unset($this->contain[$attachmentName]);
				} else {
					$attachmentSlotKey = array_search($attachmentName, $this->contain);
					if ($attachmentSlotKey !== false) {
						unset($this->contain[$attachmentSlotKey]);
					}
				}
			}
		}
	}

/**
 * Check to see if any attachments need copying and remove their models from the _setupRecord()
 * query. If attachments are to be copied then their original records are stored to be used for
 * when the record has been copied.
 *
 * @param obj $Model   The model object being copied.
 * @param int $modelId The model id of the record being copied.
 * @return void.
 */
	protected function _checkAttachmentCopy($Model, $modelId) {
		if (!empty($this->settings[$Model->alias]['copyImages'])) {
			$this->_storeOriginalImages($Model);
		}

		$this->_removeImageContains($Model);

		if (!empty($this->settings[$Model->alias]['copyDocuments'])) {
			$this->_storeOriginalDocuments($Model);
		}

		$this->_removeDocumentContains($Model);
	}

/**
 * Store the original images of the record so that they can be accessed after the record has been
 * copied.
 *
 * @param obj $Model The model object being copied.
 * @return void.
 */
	protected function _storeOriginalImages($Model) {
		$this->_storeOriginalAttachments($Model, 'Image');
	}

/**
 * Store the original documents of the record so that they can be accessed after the record has
 * been copied.
 *
 * @param obj $Model The model object being copied.
 * @return void.
 */
	protected function _storeOriginalDocuments($Model) {
		$this->_storeOriginalAttachments($Model, 'Document');
	}

/**
 * Store the original attachments of the record so that they can be accessed after the record
 * has been copied.
 *
 * @param obj    $Model          The model object being copied.
 * @param string $attachmentType The attachment type to store. E.g. Image, Document.
 * @param array  $params         Custom parameters to use when finding the original attachments.
 * @return void.
 */
	protected function _storeOriginalAttachments($Model, $attachmentType, $params = []) {
		$Attachment = EvClassRegistry::init('EvCore.' . $attachmentType);

		$defaultParams = [
			'conditions' => [
				$attachmentType . '.model' => $Model->alias,
				$attachmentType . '.model_id' => $this->mainModelId,
			]
		];

		$params = Hash::merge($defaultParams, $params);

		$this->{'_original' . $attachmentType . 's'} = $Attachment->find('all', $params);
	}

/**
 * Copy the attachments that were stored before the record was copied to the new record.
 *
 * @param obj $Model      The model object being copied.
 * @param int $newModelId The id of the new record.
 * @return void.
 */
	protected function _copyAttachments($Model, $newModelId) {
		if (!empty($this->_originalImages)) {
			$this->_copyImages($Model, $newModelId);
		}

		if (!empty($this->_originalDocuments)) {
			$this->_copyDocuments($Model, $newModelId);
		}
	}

/**
 * Copy the images that were stored before the record was copied to the new record.
 * Duplicate the images into a new folder of the new record's id.
 *
 * @param obj $Model      The model object being copied
 * @param int $newModelId The id of the new record.
 * @return void.
 */
	protected function _copyImages($Model, $newModelId) {
		$Image = EvClassRegistry::init('EvCore.Image');

		foreach ($this->_originalImages as $originalImage) {
			$newImage = [];
			$newImage['Image'] = $this->_stripFields($Model, $originalImage['Image']);

			$newImage['Image']['model_id'] = $newModelId;

			$Image->create();
			$Image->save($newImage);
			$newImageId = $Image->id;

			// Duplicate the entire original image directory
			$originalImageDirectory = new Folder(WWW_ROOT . 'files' . DS . 'image' . DS . $originalImage['Image']['id']);
			$newImageDirectory = WWW_ROOT . 'files' . DS . 'image' . DS . $newImageId;

			$originalImageDirectory->copy($newImageDirectory);
		}
	}

/**
 * Copy the documents that were stored before the record was copied to the new record.
 * Duplicate the documents into a new folder of the new record's id.
 *
 * @param obj $Model      The model object being copied
 * @param int $newModelId The id of the new record.
 * @return void.
 */
	protected function _copyDocuments($Model, $newModelId) {
		$Document = EvClassRegistry::init('EvCore.Document');

		foreach ($this->_originalDocuments as $originalDocument) {
			$newDocument = [];
			$newDocument['Document'] = $this->_stripFields($Model, $originalDocument['Document']);

			$newDocument['Document']['model_id'] = $newModelId;

			$Document->create();
			$Document->save($newDocument);
			$newDocumentId = $Document->id;

			// Duplicate the entire original image directory
			$originalDocumentDirectory = new Folder(WWW_ROOT . 'files' . DS . 'document' . DS . $originalDocument['Document']['id']);
			$newDocumentDirectory = WWW_ROOT . 'files' . DS . 'document' . DS . $newDocumentId;

			$originalDocumentDirectory->copy($newDocumentDirectory);
		}
	}

/**
 * Given an array of data, strip all Ids so when saved using the original model all relationships and items
 * associated with them are duplicated.
 *
 * @param object $Model The root model
 * @param array $data Either a single record with contains or an array of records
 * @return array The data with all relationships removed
 */
	public function stripAllIds($Model, $data) {
		if (!empty($data[$Model->alias])) {
			// This is a single record with contains
			$data[$Model->alias] = $this->_stripIdsFromRecord($Model, $data[$Model->alias], false, false);
			$data = $this->_stripRelatedIdsFromRecord($Model, $data);
		} else {
			// This is an array of records
			foreach ($data as $key => $record) {
				if (!empty($record[$Model->alias])) {
					// This is an array of primary records
					$data[$key][$Model->alias] = $this->_stripIdsFromRecord($Model, $record[$Model->alias], false, false);
					$data[$key] = $this->_stripRelatedIdsFromRecord($Model, $record);
				} else {
					// This is an array of contained records
					$data[$key] = $this->_stripIdsFromRecord($Model, $record, false, true);
				}
			}
		}

		return $data;
	}

/**
 * Strip the ids from a record
 *
 * @param object $Model The Model representing the data
 * @param array $data A record of type model
 * @param string $foreignKey The name of the foreign key that link this model to another. It will be stripped.
 * @param bool $stripRelated Whether related items in this record should also be stripped
 * @return array The original $data with ids stripped
 */
	protected function _stripIdsFromRecord($Model, $data, $foreignKey, $stripRelated) {
		unset(
			$data[$Model->primaryKey],
			$data['created'],
			$data['modified']
		);

		if (!empty($foreignKey)) {
			unset($data[$foreignKey]);
		}

		if ($stripRelated) {
			$data = $this->_stripRelatedIdsFromRecord($Model, $data);
		}

		return $data;
	}

/**
 * Works through the hasOne, hasMany and hasAndBelongsToMany relationships and removes any links between them.
 * HABTM records are also duplicated without the ids pointing to the current record so data kept on the link can
 * also be saved to the copy
 *
 * @param object $Model The model type
 * @param array $data A record of the model type
 * @return array The $data with related models stripped of their ids
 */
	protected function _stripRelatedIdsFromRecord($Model, $data) {
		foreach ($Model->hasOne as $key => $hasOne) {
			if (!empty($data[$key])) {
				$RelatedModel = $Model->$key;
				$data[$key] = $this->_stripIdsFromRecord($RelatedModel, $data[$key], $hasOne['foreignKey'], true);
			}
		}

		foreach ($Model->hasMany as $key => $hasMany) {
			if (!empty($data[$key])) {
				$RelatedModel = $Model->$key;
				foreach ($data[$key] as $hasManyKey => $hasManyRecord) {
					$data[$key][$hasManyKey] = $this->_stripIdsFromRecord($RelatedModel, $hasManyRecord, $hasMany['foreignKey'], true);
				}
			}
		}

		foreach ($Model->hasAndBelongsToMany as $key => $habtm) {
			if (!empty($data[$key])) {
				$RelatedModel = $Model->$key;
				$ThroughModel = EvClassRegistry::init($habtm['with']);
				foreach ($data[$key] as $habtmKey => $habtmValue) {
					$data[$key][$habtmKey][$ThroughModel->alias] = $this->_stripIdsFromRecord($ThroughModel, $habtmValue[$ThroughModel->alias], $habtm['foreignKey'], false);
				}
			}
		}

		return $data;
	}
}
