<?php

App::uses('BaseMigrationModelUtil', 'EvMigrationUtil.Lib');

class PageMigrationModelUtil extends BaseMigrationModelUtil {

/**
 * Exports a page and creates a migration
 *
 * @param string $internalTitle The id passed on the command line
 * @return bool True on success
 */
	public function getExportMigration($internalTitle) {
		$migrationUp = PHP_EOL;
		$migrationDown = PHP_EOL;
		$migration = '';

		$migration .= '$PageUtil = MigrationUtil::init(\'EvCore.Page\');' . PHP_EOL;
		$migration .= '$TemplateUtil = MigrationUtil::init(\'EvTemplates.Template\');' . PHP_EOL;
		$migration .= '$CustomFieldUtil = MigrationUtil::init(\'EvCustomFields.Field\');' . PHP_EOL;
		$migration .= '$ImageBlockUtil = MigrationUtil::init(\'EvTemplates.ImageBlock\');' . PHP_EOL;
		$migration .= '$DocumentBlockUtil = MigrationUtil::init(\'EvTemplates.DocumentBlock\');' . PHP_EOL;

		// Can't fetch it all at once because templates are added through a behaviour
		$page = $this->Model->find('first', [
			'conditions' => [
				'internal_title' => $internalTitle
			],
			'customFields' => true
		]);

		if (!empty($page['Page']['template_id'])) {
			// Fetch the template
			$Template = EvClassRegistry::init('EvTemplates.Template');
			$template = $Template->find('first', [
				'conditions' => [
					'Template.id' => $page['Page']['template_id']
				],
				'contain' => [
					'ImageBlock',
					'DocumentBlock'
				]
			]);
			$page['Template'] = $template;

			// Remove any ids
			unset(
				$template['Template']['id'],
				$template['Template']['created'],
				$template['Template']['modified']
			);

			// Build the template in the migration
			$migrationUp .= '$templateId = $TemplateUtil->createOnce([' . PHP_EOL;
			$migrationUp .= '	\'name\' => \'' . $template['Template']['name'] . '\',' . PHP_EOL;
			$migrationUp .= '	\'model\' => \'' . $template['Template']['model'] . '\',' . PHP_EOL;
			$migrationUp .= '], [' . PHP_EOL;
			foreach ($template['Template'] as $field => $value) {
				$migrationUp .= '	\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
			}
			$migrationUp .= ']);' . PHP_EOL;

			// Create the custom fields
			if (!empty($template['CustomFields'])) {
				$migrationUp .= '$customFieldIds = $CustomFieldUtil->createAllOnce([\'name\', \'model\', \'model_id\'], [' . PHP_EOL;
				foreach ($template['CustomFields'] as $customField) {
					$migrationUp .= '	' . $customField['Field']['id'] . ' => [' . PHP_EOL;

					unset(
						$customField['Field']['id'],
						$customField['Field']['model_id'],
						$customField['Field']['created'],
						$customField['Field']['modified']
					);

					foreach ($customField['Field'] as $field => $value) {
						$migrationUp .= '		\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
					}

					$migrationUp .= '		\'model_id\' => $templateId,' . PHP_EOL;
					$migrationUp .= '	],' . PHP_EOL;
				}
				$migrationUp .= ']);' . PHP_EOL;
			}

			// Copy any image blocks
			if (!empty($template['ImageBlock'])) {
				$migrationUp .= '$ImageBlockUtil->createAllOnce([\'template_id\', \'name\'], [' . PHP_EOL;
				foreach ($template['ImageBlock'] as $imageBlock) {

					$blockFields = json_encode($imageBlock['blockFields']);
					unset(
						$imageBlock['id'],
						$imageBlock['created'],
						$imageBlock['modified'],
						$imageBlock['template_id'],
						$imageBlock['dimensions'],
						$imageBlock['blockFields']
					);

					$migrationUp .= '	[' . PHP_EOL;
					foreach ($imageBlock as $field => $value) {
						$migrationUp .= '		\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
					}
					$migrationUp .= '		\'template_id\' => $templateId,' . PHP_EOL;
					$migrationUp .= '		\'blockFields\' => ' . $blockFields . ',' . PHP_EOL;
					$migrationUp .= '	],' . PHP_EOL;
				}
				$migrationUp .= ']);' . PHP_EOL;
			}

			// Copy any document blocks
			if (!empty($template['DocumentBlock'])) {
				$migrationUp .= '$DocumentBlockUtil->createAllOnce([\'template_id\', \'name\'], [' . PHP_EOL;
				foreach ($template['DocumentBlock'] as $documentBlock) {

					$blockFields = json_encode($imageBlock['blockFields']);
					unset(
						$documentBlock['id'],
						$documentBlock['created'],
						$documentBlock['modified'],
						$documentBlock['template_id'],
						$documentBlock['blockFields']
					);

					$migrationUp .= '	[' . PHP_EOL;
					foreach ($documentBlock as $field => $value) {
						$migrationUp .= '		\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
					}
					$migrationUp .= '		\'template_id\' => $templateId,' . PHP_EOL;
					$migrationUp .= '		\'blockFields\' => ' . $blockFields . ',' . PHP_EOL;
					$migrationUp .= '	],' . PHP_EOL;
				}
				$migrationUp .= ']);' . PHP_EOL;
			}

			// Don't add a migration down for the template or fields as we have no way of knowing if this migration was the one to actually create it.
			// Other pages may be using the existing template.
		}

		// Remove the ids from the data as they may not be the same in the next DB
		unset(
			$page['Page']['id'],
			$page['Page']['modified'],
			$page['Page']['created'],
			$page['Page']['template_id']
		);

		$migrationUp .= '$PageUtil->createOnce([' . PHP_EOL;
		$migrationUp .= '	\'internal_title\' => \'' . $page['Page']['internal_title'] . '\',' . PHP_EOL;
		$migrationUp .= '], [' . PHP_EOL;
		foreach ($page['Page'] as $field => $value) {
			$migrationUp .= '	\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
		}

		if (!empty($template)) {
			$migrationUp .= '	\'template_id\' => $templateId,' . PHP_EOL;
		}

		if (!empty($page['CustomFieldData'])) {
			$migrationUp .= '	\'CustomFieldData\' => [' . PHP_EOL;

			foreach ($page['CustomFieldData'] as $fieldId => $fieldData) {

				unset(
					$fieldData['FieldData']['id'],
					$fieldData['FieldData']['created'],
					$fieldData['FieldData']['modified'],
					$fieldData['FieldData']['model_id'],
					$fieldData['FieldData']['field_id']
				);

				$migrationUp .= '		$customFieldIds[' . $fieldId . '] => [' . PHP_EOL;
				$migrationUp .= '			\'FieldData\' => [' . PHP_EOL;
				foreach ($fieldData['FieldData'] as $field => $value) {
					$migrationUp .= '				\'' . $field . '\' => ' . $this->_migrationValue($value) . ',' . PHP_EOL;
				}
				$migrationUp .= '				\'field_id\' => $customFieldIds[' . $fieldId . '],' . PHP_EOL;
				$migrationUp .= '			],' . PHP_EOL;
				$migrationUp .= '		],' . PHP_EOL;
			}
			$migrationUp .= '	],' . PHP_EOL;
		}

		$migrationUp .= ']);' . PHP_EOL;

		$migrationDown .= '$pageId = $PageUtil->Model->field(\'Page.id\', [\'Page.internal_title\' => \'' . $internalTitle . '\']);' . PHP_EOL;
		$migrationDown .= '$CustomFieldUtil->Model->FieldData->deleteAll([\'FieldData.model\' => \'EvCore.Page\', \'FieldData.model_id\' => $pageId]);' . PHP_EOL;
		$migrationDown .= '$PageUtil->Model->delete($pageId);' . PHP_EOL;

		return [
			'common' => $migration,
			'up' => $migrationUp,
			'down' => $migrationDown
		];
	}

}
