<?php

class ArrayUtil {

/**
 * clear an array of any empty elements
 *
 * @param   array   Array to clear
 * @return  array   Clean array
 */
	public static function clearArray($array) {
		foreach ($array as $key => $value) {
			if (is_array($value)) {
				$array[$key] = self::clearArray($value);
			} else {
				if (empty($value)) {
					unset($array[$key]);
				}
			}
		}

		return $array;
	}

/**
 * take an array and split into the given number of arrays with equal number of elements
 * if an uneven number of elements one (or more) arrays may have more elements then the others
 *
 * @example http://snippi.com/s/9ls9sug
 *
 * @param array The array we want to split
 * @param int The number of sections we want
 * @return array The resulting split array
 */
	public static function splitArray($array, $sections) {
		if (count($array) < $sections) {
			$chunkSize = 1;
		} else {
			$chunkSize = ceil(count($array) / $sections);
		}

		return array_chunk($array, $chunkSize, true);
	}

/**
 * Add new elements to the given array after the element with the supplied key
 *
 * @example http://snippi.com/s/6trt9kq
 *
 * @param array The array we want to add to
 * @param string|int The key we wish to add our new elements after.
 * @param array The elements we wish to add
 * @return array The resulting array with new elements
 */
	public static function addAfter($array, $key, $newElements) {
		$offset = self::getOffsetByKey($array, $key);

		if ($offset >= 0) {
			// increment cause we want to actually splice in from the element AFTER the one we found
			$offset++;

			// get the slice, and insert the new elements and rebuild the array
			$arrayItems = array_splice($array, $offset);
			$newElements += $arrayItems;
			$array += $newElements;
		}

		return $array;
	}

/**
 * Add new elements to the given array before the element with the supplied key.
 *
 * @param array $array The array we want to add to
 * @param string|int $key The key we wish to add our new elements before.
 * @param array $newElements The elements we wish to add
 * @return array The resulting array with new elements
 */
	public static function addBefore($array, $key, $newElements) {
		$offset = self::getOffsetByKey($array, $key);

		if ($offset >= 0) {
			// get the slice, and insert the new elements and rebuild the array
			$arrayItems = array_splice($array, $offset);
			$newElements += $arrayItems;
			$array += $newElements;
		}

		return $array;
	}

/**
 * get the offset of an element within an array based on the key
 * useful for associative arrays
 *
 * @param array The containing array
 * @param string The key to search for
 * @return int|null The offset within an array | null if not found
 */
	public static function getOffsetByKey($array, $needle) {
		$offset = 0;

		foreach ($array as $key => $value) {
			if ($key === $needle) {
				return $offset;
			}

			$offset++;
		}

		return null;
	}

/**
 * get the offset of an element within an array based on the element value
 * useful for associative arrays
 *
 * @param array The containing array
 * @param string The value to search for
 * @return int|null The offset within an array | null if not found
 */
	public static function getOffsetByValue($array, $needle) {
		$offset = 0;

		foreach ($array as $key => $value) {
			if ($value === $needle) {
				return $offset;
			}

			$offset++;
		}

		return null;
	}

/**
 * Move Item
 *
 * Moves an existing array item to reposition it after another item.
 *
 * @param array The array we want to do the reordering in
 * @param string|int The element key we wish to move
 * @param array The element key that'll be before the one we're moving
 * @return array The resulting array with reordered elements
 */
	public static function moveItem($array, $key, $moveAfter) {
		$moveItem = array(
			$key => $array[$key]
		);

		unset($array[$key]);

		$result = self::addAfter($array, $moveAfter, $moveItem);

		return $result;
	}

/**
 * Produces the cartesian product of all the given arrays
 *
 * @param array Array of Arrays to combine e.g. array( array('a', 'b'), array('1', '2') )
 * @return array Array of products e.g. array( array('a', '1'), array('a', '2'), array('b', '1'), array('b', '2'))
 */
	public static function cartesianProduct($arrays) {
		$result = array();
		$arrays = array_values($arrays);
		$sizeIn = count($arrays);
		$size = $sizeIn > 0 ? 1 : 0;
		foreach ($arrays as $array) {

			$size = $size * count($array);
		}

		for ($i = 0; $i < $size; $i ++) {
			$result[$i] = array();

			for ($j = 0; $j < $sizeIn; $j ++) {
				array_push($result[$i], current($arrays[$j]));

			}

			for ($j = ($sizeIn - 1); $j >= 0; $j --) {
				if (next($arrays[$j])) {
					break;
				} elseif (isset($arrays[$j])) {
					reset($arrays[$j]);
				}
			}
		}

		return $result;
	}

/**
 * Find the first sub array in an array that has the supplied key with the supplied value.
 * If a returnkey is provided then the element from the sub array that matches the returnkey will be
 * returned if it is set.
 *
 * @param array      $array     The array to search for sub arrays.
 * @param string|int $key       The key to search for in the sub array.
 * @param mixed      $value     The value to match against in the sub array.
 * @param string|int $returnKey An optional key to return a sub element instead of the entire matched element.
 * @return bool|mixed False if no sub array matches they key value pair. An array if the sub array is
 *                          returned. Mixed if the $returnKey is provided and they key found.
 */
	public static function findFirst($array, $key, $value, $returnKey = null) {
		foreach ($array as $element) {
			if (isset($element[$key]) && $element[$key] === $value) {

				//Return a sub element if a return key has been provided.
				if ($returnKey !== null && isset($element[$returnKey])) {
					return $element[$returnKey];
				} elseif ($returnKey !== null) {
					//Sub element was requested but it isn't set
					return false;
				} else {
					return $element;
				}
			}
		}

		return false;
	}

/**
 * Checks whether an array is or is not associative
 *
 * @param array $array An array to check
 * @return bool
 */
	public function isAssociative($array) {
		if (empty($array)) {
			return false;
		}

		return array_keys($array) !== range(0, count($array) - 1);
	}

/**	
 * Helper function to extract out the values from sub arrays by name. Allows
 * you to pull 1 fixed key from a multidimensional array.
 *
 * @example http://snippi.com/s/apvy3ct
 *
 * @param  array $data The main array of data to extract from
 * @param  string $itemKey The name of the key to extract (e.g id)
 * @return array
 */
	public static function extractArrayItems(array $data, $itemKey) {
		$items = [];

		array_walk_recursive($data, function ($value, $key) use (&$items, $data, $itemKey) {
			if ($key === $itemKey) {
				$items[] = $value;
			}
		});

		return $items;
	}
}
