<?php
App::uses('CustomEmail', 'Lib');

/**
 * Handles User requests
 *
 * @package CoreCMS/User
 */
class UsersController extends AppController {

/**
 * Controller name
 *
 * @var string
 */
	public $name = 'Users';

/**
 * List of actions permitted by admin users.
 */
	public $adminActions = array(
		'admin_index',
		'admin_add',
		'admin_edit',
		'admin_account',
		'admin_delete',
		'admin_toggle',
		'admin_password_reset',
		'admin_password_reset_callback',
		'admin_login',
		'admin_logout'

	);

/**
 * redefine the constructor to add in our user manage component
 */
	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		$this->components['UserManager'] = array(
			'className' => 'EvCore.UserManager'
		);

		parent::__construct($Request, $Response);
	}

/**
 * Allow public access to some methods.
 */
	public function beforeFilter() {
		$this->Auth->allow(array(
			'admin_login',
			'admin_password_reset',
			'admin_password_reset_callback',
			'password_reset',
			'password_reset_email',
			'password_reset_callback',
			'password_reset_fail',
			'login',
			'register',
			'verify_email'
		));

		parent::beforeFilter();

		$notInPrefix = empty($this->request->params['prefix']);

		if (!Configure::read('SiteSetting.users.enable_registration') && $notInPrefix && $this->request->action != 'ajax_toggle') {
			throw new NotFoundException('User registration is not enabled for this site.');
			return;
		}
	}

/**
 * Checks that if the register showAfterFailedAttempts value is assigned that we also
 * have a Google reCAPTCHA secret and siteKey assigned
 *
 * @return bool
 */
	protected function _checkRegisterReCaptcha() {
		// check that if we are using login reCAPTCHA that the correct credentials are defined
		return ! (
			Configure::check('EvCore.userReCaptcha.showAfterFailedRegisterAttempts') &&
			Configure::read('EvCore.userReCaptcha.showAfterFailedRegisterAttempts') !== false &&
			(
				empty(Configure::read('EvCore.userReCaptcha.secret')) ||
				empty(Configure::read('EvCore.userReCaptcha.siteKey'))
			)
		);
	}

/**
 * Checks that if the login showAfterFailedAttempts value is assigned that we also
 * have a Google reCAPTCHA secret and siteKey assigned
 *
 * @return bool
 */
	protected function _checkLoginReCaptcha() {
		// check that if we are using login reCAPTCHA that the correct credentials are defined
		return ! (
			Configure::check('EvCore.userReCaptcha.showAfterFailedLoginAttempts') &&
			Configure::read('EvCore.userReCaptcha.showAfterFailedLoginAttempts') !== false &&
			(
				empty(Configure::read('EvCore.userReCaptcha.secret')) ||
				empty(Configure::read('EvCore.userReCaptcha.siteKey'))
			)
		);
	}

/**
 * Handles User login. Redirects back to the calling page on successful login.
 *
 * @return void
 */
	public function login() {
		$this->_login();

		$this->view = 'EvCore.Users/login';
	}

	/**
	 * Admin login.
	 *
	 * @return void
	 */
	public function admin_login() {
		$this->_login();

		$this->layout = "EvCore.admin_login";

		$this->view = 'EvCore.Users/admin_login';
	}

	protected function _login() {
		// disable cache for the action's response. prevents potentially sensitive user
		// information from being cached in the browser upon failed login attempts
		// * require here for page load
		$this->response->disableCache();

		// check that we have Google reCAPTCHA setup if we are using it
		if (! $this->_checkLoginReCaptcha()) {
			$this->Flash->warning('Missing Google reCAPTCHA credentials in EvCore config');
		}

		// If user is logged in redirect to the homepage.
		$user = $this->Auth->user();
		if (!empty($user)) {
			if (empty($this->params['prefix'])) {
				return $this->redirect('/');
			}
		}

		if ($this->request->is('post')) {
			$result = $this->UserManager->login(EvClassRegistry::init('EvCore.User'));

			if ($result['result'] === true) {
				$this->redirect(
					$result['data']['redirect']
				);
			} else {
				$this->Flash->fail(
					$result['data']['errors']
				);
			}
		}

		// determine whether we're showing the reCAPTCHA field on page load
		// i.e. after zero failed attempts
		if (
			Configure::read('EvCore.userReCaptcha.showAfterFailedLoginAttempts') === 0
		) {
			$this->set('showReCaptcha', true);
		}
	}

/**
 * handle a user registration on the front end
 *
 */
	public function register() {
		// disable cache for the action's response. prevents potentially sensitive user
		// information from being cached in the browser upon failed registration attempts
		// * require here for page load
		$this->response->disableCache();

		// check that we have Google reCAPTCHA setup if we are using it
		if (! $this->_checkRegisterReCaptcha()) {
			$this->Flash->warning('Missing Google reCAPTCHA credentials in EvCore config');
		}

		if (! empty($this->request->data) && $this->request->is('post')) {

			$result = $this->UserManager->register(EvClassRegistry::init('EvCore.User'));
			if ($result['result'] === true) {

				$this->Flash->success($result['data']['message']);

				return $this->redirect($result['data']['redirect']);
			} else {

				$this->Flash->fail($result['data']['errors']);
			}
		}

		// determine whether we're showing the reCAPTCHA field on page load
		// i.e. after zero failed attempts
		if (
			Configure::read('EvCore.userReCaptcha.showAfterFailedRegisterAttempts') === 0
		) {
			$this->set('showReCaptcha', true);
		}

		$this->view = 'EvCore.Users/register';
	}

/**
 * handle a user verification via email
 *
 */
	public function verify_email($code) {
		if (Configure::read('SiteSetting.users.activation') !== 'email') {
			throw new NotFoundException;
		}

		// try to find the user
		$user = $this->User->findByVerificationCode($code);

		if (empty($user)) {
			$this->redirect(array('action' => 'login'));
			return;
		}

		// code found, activate and delete code
		if ($this->User->verifyFromEmail($user['User']['id'], $user['User']['verification_code'])) {
			$this->Session->setFlash(
				'Your account has been verified, redirecting you to the login',
				'flash_success'
			);
		} else {
			$this->Session->setFlash(
				'The verification code was correct but there was a problem activating your account, please try again or contact us for assistance.',
				'flash_fail'
			);
		}

		return $this->redirect(array('action' => 'login'));
	}

/**
 * Handles User logout. Redirects to Auth::logoutRedirect (defaults as the homepage).
 *
 * @return void
 */
	public function logout() {
		return $this->redirect($this->Auth->logout());
	}

/**
 * Handles admin logout. Currently just wrapper for standard logout
 *
 * @return void
 */
	public function admin_logout() {
		return $this->logout();
	}

/**
 * Password reset.
 *
 * @return void
 */
	public function password_reset() {
		return $this->_passwordReset();
	}

/**
 * Password reset when in admin section
 *
 * @return void;
 */
	public function admin_password_reset() {
		$this->_passwordReset(true);

		$this->layout = 'EvCore.admin_login';

		return;
	}

/**
 * Initially shows the password reset form and then processes the password reset request
 * on form submission, displaying a generic template showing further instructions
 *
 * @param bool $inAdmin Defines whether the request has come from the app or admin theme
 * @return void
 */
	protected function _passwordReset($inAdmin = false) {
		$Model = $this->{$this->modelClass};

		$this->view = 'EvCore.Users/';

		if (! empty($this->request->data)) {

			// Use the login validation to check the email is a valid
			$Model->validate = $Model->validateLogin();
			$Model->set($this->request->data);
			if (!$Model->validates(['fieldList' => ['email']])) {
				$this->Flash->fail(['list' => $Model->validationErrors]);
				$this->view .= $inAdmin === true ? 'admin_password_reset' : 'password_reset';
				return;
			}
			$Model->clear();

			$user = $Model->find('first', [
				'conditions' => [
					$Model->escapeField('email') => $this->request->data[$Model->alias]['email'],
					$Model->escapeField('is_active') => true,
					$Model->escapeField('is_guest_user') => false,
				]
			]);

			// if a user is found to matched the requested email, send reset email
			if (! empty($user)) {
				$this->_passwordResetEmail($user[$Model->alias]['id']);
			}

			// assign admin / app specific view
			$this->view .= $inAdmin === true ? 'admin_password_reset_email' : 'password_reset_email';
		} else {
			// assign admin / app specific view
			$this->view .= $inAdmin === true ? 'admin_password_reset' : 'password_reset';
		}
	}

/**
 * Creates a reset code and emails it to the user
 *
 * @param integer $userId ID of user to reset password for
 * @return void
 */
	protected function _passwordResetEmail($userId) {
		$Model = $this->{$this->modelClass};

		if ($Model->resetPassword($userId) === false) {
			$this->Session->setFlash('User not found.', 'flash_fail');
			$this->redirect(
				array(
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
			return;

		}

		$this->set('resetUser', $Model->findById($userId));

		return;
	}

/**
 * Handles callback from password reset email. Allows a user to change their
 * password.
 *
 * @param string $code Contains to password reset code
 * @return void
 */
	public function password_reset_callback($code) {
		$this->_passwordResetCallback($code);

		$this->view = 'EvCore.Users/password_reset_callback';
	}

/**
 * Handles callback from password reset email
 * Allows user to change their password
 *
 * @param string $code Contains to password reset code
 * @return void
 */
	public function admin_password_reset_callback($code) {
		$this->_passwordResetCallback($code, true);

		$this->layout = 'EvCore.admin_login';
		$this->view = 'EvCore.Users/admin_password_reset_callback';
	}

/**
 * Handles callback from password reset email
 * Allows user to change their password
 *
 * @param string $code Contains the hashed string to query against
 * @return mixed Returns bool on fail or redirects to the login template on success
 */
	protected function _passwordResetCallback($code, $isAdmin = false) {
		$Model = $this->{$this->modelClass};

		$user = $Model->find('first', [
			'conditions' => [
				$Model->alias . '.password_reset_code' => $code,
				$Model->alias . '.password_reset_code_expires >' => gmdate('Y-m-d H:i:s')
			]
		]);

		// if the code has expired or invalid then notify user are redirect away
		if (empty($user[$Model->alias]['id'])) {
			$this->Flash->fail('Password reset code not found or has expired', 'flash_fail');

			$this->redirect(
				array(
					'admin' => $isAdmin,
					'action' => 'password_reset'
				)
			);
		}

		// handle form submission
		if ($this->request->is('post') && ! empty($this->request->data)) {
			// manually assign the password validation rules
			$Model->validate = $Model->validatePassword($user[$Model->alias]['id'], null, $user);

			// manually assign data to save in an attempt to
			// prevent data injection
			$dataToSave = [
				$Model->alias => [
					'id' => $user[$Model->alias]['id'],
					'password' => $this->request->data[$Model->alias]['password'],
					'confirm_password' => $this->request->data[$Model->alias]['confirm_password'],
					'password_reset_code' => '',
					'password_reset_code_expires' => ''
				]
			];

			if ($Model->save($dataToSave)) {
				$this->Flash->success('Password changed', 'flash_success');

				$this->redirect(array(
					'admin' => $isAdmin,
					'action' => 'login'
				));
			} else {
				$this->Flash->fail('Please correct the errors:', 'flash_fail');

				// clear the two password fields
				$this->request->data[$Model->alias]['password'] = '';
				$this->request->data[$Model->alias]['confirm_password'] = '';
			}
		}
	}

/**
 * Populates drop down lists for admin forms.
 *
 * @return void
 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$userGroupParams = array(
			'order' => array(
				'UserGroup.level' => 'ASC'
			)
		);

		// ensure that a user cannot create other users with higher permissions than themselves
		$authUser = $this->Auth->user();

		if (! empty($authUser['UserGroup']['level'])) {
			$userGroupParams['conditions']['UserGroup.level >='] = $authUser['UserGroup']['level'];
		}

		$userGroupParams = Hash::merge($userGroupParams, $this->_addExtraParamsToAdminPopulateLookupsUserGroups());

		$this->set('userGroups', $Model->UserGroup->find('list', $userGroupParams));

		return;
	}

	protected function _addExtraParamsToAdminPopulateLookupsUserGroups() {
		return [];
	}

/**
 * Defines the fields used in the admin filter form.
 *
 * @return array
 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};

		$this->_adminPopulateFilterLookups();

		$fields = parent::_adminFilterFields();

		$fields[$Model->alias . '.user_group_id'] = array(
			'label' => 'User Group',
			'type' => 'select',
			'compare' => array($Model->alias . '.user_group_id' => '%s')
		);

		unset($fields[$Model->alias . '.password']);
		unset($fields[$Model->alias . '.password_reset_code']);
		unset($fields[$Model->alias . '.password_reset_code_expires']);

		return $fields;
	}

/**
 * Defines the columns to display for admin_index().
 *
 * @return array
 */
	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};

		return array(
			$Model->alias . '.id' => array(
				'label' => 'ID',
				'type' => 'integer',
			),
			$Model->alias . '.email' => array(
				'label' => 'Email',
				'type' => 'string',
			),
			'UserGroup.name' => array(
				'label' => 'User Group',
				'type' => 'string'
			),
			$Model->alias . '.is_active' => array(
				'type' => 'boolean',
				'label' => 'Active'
			)
		);
	}

/**
 * Pagination settings for admin_index
 *
 * @return array
 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};

		$paginate = parent::_adminIndexPaginate();
		$paginate['contain'][] = 'UserGroup';

		if ($this->Auth->user($Model->alias . '.id') != 1) {

			$paginate['conditions'][$Model->alias . '.id !='] = 1;
		}

		return $paginate;
	}

/**
 * Defines the fields displayed in an admin_form for this model
 *
 * Defaults to all fields in the db table
 *
 * @return array
 */
	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$Model = $this->{$this->modelClass};

		$fields[$Model->alias . '.password']['type'] = 'password';
		$fields[$Model->alias . '.password']['autocomplete'] = 'new-password';

		// pull out the password fields to use in a separate tab on the form
		$passwordFields = [
			$Model->alias . '.password' => $fields[$Model->alias . '.password'],
			$Model->alias . '.confirm_password' => $fields[$Model->alias . '.password']
		];

		$this->set(compact('passwordFields'));

		// Prepend fake fields to prevent autofill - these get filled instead on FF. autocomplete=new-password works with Chrome.
		$fields = [
			$Model->alias . '.fake_email' => [
				'type' => 'input',
				'div' => 'hidden'
			],
			$Model->alias . '.fake_password' => [
				'type' => 'password',
				'div' => 'hidden'
			]
		] + $fields;

		unset(
			$fields[$Model->alias . '.is_guest_user'],
			$fields[$Model->alias . '.password_reset_code'],
			$fields[$Model->alias . '.password_reset_code_expires'],
			$fields[$Model->alias . '.verification_code'],
			$fields[$Model->alias . '.password'],
			$fields[$Model->alias . '.confirm_password']
		);

		return $fields;
	}

/**
 * Add or edit a user.
 *
 * @param  integer $id User ID
 * @return void
 */
	public function admin_edit($id = null) {
		$Model = $this->{$this->modelClass};

		// unset password fields if they're not needed
		if (!empty($this->request->data)) {

			if (empty($this->request->data[$Model->alias]['password'])) {

				unset($this->request->data[$Model->alias]['password'], $this->request->data[$Model->alias]['confirm_password']);
			}

		}

		// call the parent admin_edit to do the add / edit
		parent::admin_edit($id);

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->view = 'EvCore.Users/admin_edit';
	}

/**
 * Admin 'My Account'.
 *
 * @return void
 */
	public function admin_account() {
		$Model = $this->{$this->modelClass};

		$user = $this->Auth->user();
		$id = $user[$Model->alias]['id'];

		$Model->validate = $Model->validateAccount();

		if (!empty($this->request->data)) {
			// Ensure the user is updating their own record only.
			$this->request->data[$Model->alias]['id'] = $id;

			if ($this->request->data[$Model->alias]['password'] == "") {
				unset($this->request->data[$Model->alias]['password']);
				unset($this->request->data[$Model->alias]['conform_password']);
			} else {
				$Model->validate = array_merge_recursive($Model->validate, $Model->validatePassword($id));
			}

			if ($Model->saveAll($this->request->data)) {

				$this->Session->setFlash(array(
					'title' => "Account Updated",
					'description' => "Your account has been successfully updated!"
				), 'flash_success');

				return $this->redirect(array('action' => 'account'));

			} else {

				$this->Session->setFlash(array(
					'title' => 'Save failed',
					'description' => 'Failed to update your account'
				), 'flash_fail');

			}

		} else {

			$this->request->data = $Model->readForEdit($id);

		}

		$this->_adminPopulateLookups();

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'My Account');

		$this->view = 'EvCore.Users/admin_account';
	}

/**
 * Edit current user account.
 *
 * @return void
 */
	public function account() {
		$Model = $this->{$this->modelClass};

		$user = $this->Auth->user();

		$Model->validate = $Model->validateAccount();

		$id = $user[$Model->alias]['id'];

		if (!empty($this->request->data)) {
			$this->request->data[$Model->alias]['id'] = $id;

			if ($this->request->data[$Model->alias]['password'] == "") {
				unset($this->request->data[$Model->alias]['password']);
				unset($this->request->data[$Model->alias]['conform_password']);
			} else {
				$Model->validate = array_merge_recursive($Model->validate, $Model->validatePassword($id));
			}

			if ($Model->save($this->request->data)) {
				$this->Auth->updateUserSession();

				$this->Session->setFlash(__('Information updated'), 'flash_success');
				return $this->redirect(
					array(
						'action' => 'account'
					)
				);
			}

		} else {

			$this->request->data = $Model->findById($id);

		}

		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'Update Account Information | ' . Configure::read('SiteSetting.general.site_title'));

		$this->view = 'EvCore.Users/account';
	}
}
