<?php

App::uses('DocumentsController', 'EvCore.Controller');

/**
 * Handles image requests
 *
 * @package CoreCMS/Image
 */
class ImagesController extends DocumentsController {

/**
 * Called before the controller action. You can use this method to configure and customize components
 * or perform logic that needs to happen before each controller action.
 *
 * @return void
 * @link https://book.cakephp.org/2.0/en/controllers.html#request-life-cycle-callbacks
 */
	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(
			[
				'get_image',
				'generate_image',
				'img_cache',
			]
		);
	}

/**
 * Add a new image slot via AJAX.
 *
 * @param string $model      The model to link the image to e.g. Page
 * @param int    $key        Image slot number.
 * @param string $type       The image slot name.
 * @param string $prefix     Used when embedding images multiple levels deep e.g. Hotel.Room.0.Image
 * @param string $plugin     The name of the plugin the model is in.
 * @param string $itemFields Comma separated list of fields.
 * @return void.
 */
	public function ajax_add($model, $key = 0, $type = 'Image', $prefix = "", $plugin = "", $itemFields = "") {
		$slotType = str_replace("Image", "", $type);

		if ($prefix == "" || $prefix == "false") {
			$prefix = $slotType;
		}

		if ($slotType == "") {
			$slotType = 'main';
		}

		if (! empty($plugin) && $plugin !== 'false') {
			$this->loadModel($plugin . "." . $model);
		} else {
			$this->loadModel($model);
		}

		$fields = false;
		if (! empty($itemFields) && $itemFields !== "false") {

			$fields = explode(',', $itemFields);
		} else {
			if (
				isset($this->$model->imageSlots[Inflector::underscore($slotType)])
			) {
				if (
					is_array($this->$model->imageSlots[Inflector::underscore($slotType)])
					&& array_key_exists('fields', $this->$model->imageSlots[Inflector::underscore($slotType)])
				) {
					$fields = $this->$model->imageSlots[Inflector::underscore($slotType)]['fields'];
				} elseif (
					! is_array($this->$model->imageSlots[Inflector::underscore($slotType)])
					&& ! empty($this->$model->imageSlots[Inflector::underscore($slotType)])
				) {
					$fields = $this->$model->imageSlots[Inflector::underscore($slotType)];
				}
			}
		}

		if (isset($this->{$model}->imageSlots[Inflector::underscore($slotType)]['fields']) && is_array($fields)) {
			$availableFields = $this->{$model}->imageSlots[Inflector::underscore($slotType)]['fields'];
			// Check the model to see if any of the fields have custom field settings defined.
			foreach ($fields as $fieldKey => $field) {

				if (isset($availableFields[$field]) && is_array($availableFields[$field])) {
					unset($fields[$fieldKey]);
					$fields[$field] = $availableFields[$field];

				}
			}
		}

		$this->set(compact('model', 'prefix', 'key', 'type', 'fields'));
		$this->theme = "Admin";
		$this->render('ajax_add', 'ajax');
	}

/**
 * Returns a preview image for ajax calls.
 *
 * @param int $id ID of image to return
 * @param int $width Width of image to return
 * @param int $height Height of image to return
 * @return void.
 */
	public function ajax_preview($id, $width = 260, $height = 195) {
		$image = $this->Image->findById($id);

		$this->set(compact('image', 'width', 'height'));

		$this->theme = "Admin";

		$this->render('ajax_preview', 'ajax');
	}

/**
 * Change the sequence of a document.
 *
 * @param int    $id        file to be reordered
 * @param string $direction direction to reorder, up or down
 * @return void.
 */
	public function ajax_reorder($id, $direction = 'down') {
		// Model name
		$Model = $this->{$this->modelClass};

		$this->set('model', $Model->alias);

		// Fetch the currenct document
		$currentDocument = $Model->findById($id);

		if (!empty($currentDocument)) {

			// Find the relevant adjacent document
			if ($direction == 'down') {
				$order = "sequence ASC";
			} else {
				$order = "sequence DESC";
			}

			$direction = $direction == 'down' ? '>' : '<';

			$params = array(
				'conditions' => array(
					'id <>' => $id,
					'model' => $currentDocument[$Model->alias]['model'],
					'model_id' => $currentDocument[$Model->alias]['model_id'],
					'attachment_type' => $currentDocument[$Model->alias]['attachment_type'],
					"sequence $direction" => $currentDocument[$Model->alias]['sequence']
				),
				'order' => $order
			);

			$adjacentDocument = $Model->find('first', $params);

			if (! empty($adjacentDocument)) {
				// Swap the sequence numbers of the current and adjacent documents

				$data = array(
					array(
						'id' => $id,
						'sequence' => $adjacentDocument[$Model->alias]['sequence']
					),
					array(
						'id' => $adjacentDocument[$Model->alias]['id'],
						'sequence' => $currentDocument[$Model->alias]['sequence']
					)
				);

				$Model->saveMany($data);
			}

			$this->set(compact('currentDocument', 'adjacentDocument'));
		}

		$this->theme = "Admin";

		$this->render('ajax_reorder', 'ajax');
	}

/**
 * Returns an image URL in the View.
 *
 * @param int $imageId The id of the image to generate.
 * @param int $width   The width of the image.
 * @param int $height  The height of the image.
 * @return void
 */
	public function generate_image($imageId, $width = null, $height = null) {
		$image = $this->Image->findById($imageId);

		$width = !empty($width) ? $width : null;
		$height = !empty($height) ? $height : null;

		$this->set(compact('image', 'width', 'height'));

		return $this->render('image', 'ajax');
	}

/**
 * Returns an image from the database.
 *
 * @param int $imageId The id of the image to get.
 * @param int $width   The width of the image to render.
 * @param int $height  The height of the image to render.
 * @return void
 * @throws NotFoundException when the image couldn't be found.
 */
	public function get_image($imageId, $width = null, $height = null) {
		$Model = $this->{$this->modelClass};

		$image = $Model->readForViewOrFail($imageId);

		$this->set(compact('image', 'width', 'height'));
		$this->set('isAjaxRequest', true);
		$this->View = $this->_getViewObject();
		$imagePath = $this->View->render('image', 'ajax');

		$this->autoRender = false;
		$this->response->file(WWW_ROOT . $imagePath);
	}

/**
 * Urls that request a cached image file that could not be found will fall through to here. Check to see if we can
 * generated the request cached file so that future requests don't get to here. Return the cached file as if it was
 * cached in the first place.
 *
 * @param int    $id             The id of the cached image to get.
 * @param string $cachedFileName The file name of the cached image.
 * @return file.
 * @throws NotFoundException when image does not match or file name could not be parsed.
 */
	public function img_cache($id, $cachedFileName) {
		$Model = $this->{$this->modelClass};

		$image = $Model->readForViewOrFail($id);

		//File extension is passed through separately, append it back to the file name.
		if (!empty($this->request->params['ext'])) {
			$cachedFileName .= '.' . $this->request->params['ext'];
		}

		//Check the image is the one we are looking for.
		if (strpos($cachedFileName, $image[$Model->alias]['filename']) === false) {
			throw new NotFoundException;
		}

		/*
		 * Image cache filename structure {dimensions}_{filename}. Remove the file name and then split the
		 * remaining string into dimensions.
		 */
		$cacheDimensions = str_replace(
			'_' . $image[$Model->alias]['filename'],
			'',
			$cachedFileName
		);

		if (empty($cacheDimensions)) {
			throw new NotFoundException;
		}

		/*
		 * Map the image dimensions onto their corresponding dimension
		 * Note: The dimensions may not match their original values due to the string not being ordered but will
		 * generate the correct filepath.
		 */
		$cacheDimensions = explode('_', $cacheDimensions);
		$cacheOptions = [
			'width',
			'height',
			'crop',
			'fit',
			'blur',
		];

		$dimensions = [];
		foreach ($cacheOptions as $optionIndex => $option) {
			//False values need to be passed through.
			if (isset($cacheDimensions[$optionIndex])) {
				$dimensions[$option] = $cacheDimensions[$optionIndex];
			}
		}

		$this->set(compact('image', 'dimensions'));
		$this->set('isAjaxRequest', true);
		$this->View = $this->_getViewObject();
		$imagePath = $this->View->render('EvCore.Images/image', 'ajax');

		$this->autoRender = false;
		$this->response->file(WWW_ROOT . $imagePath);
	}
}
