<?php

use Sunspikes\Ratelimit\Cache\Adapter\DesarrollaCacheAdapter;
use Sunspikes\Ratelimit\Cache\Factory\DesarrollaCacheFactory;
use Sunspikes\Ratelimit\RateLimiter;
use Sunspikes\Ratelimit\Throttle\Factory\ThrottlerFactory;
use Sunspikes\Ratelimit\Throttle\Hydrator\HydratorFactory;

App::uses('AppComponent', 'Controller/Component');

class UserManagerComponent extends AppComponent {

/**
 * Settings for rate limiting
 * @var array
 */
	public $rateLimit = [
		'attempts' => 5,
		'timeout' => 600
	];

/**
 * Set component settings
 *
 * @see Component::initialize()
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		if (!empty($this->settings['rateLimit'])) {
			$this->rateLimit = array_merge($this->rateLimit, $this->settings['rateLimit']);
		}
	}

/**
 * register a user
 *
 * @param 	object 	$Model 		The model object to use for saving
 * @return 	array 	$result 	Two element array - result(bool) / data(array)
 */
	public function register($Model) {
		// disable cache for the action's response. prevents potentially sensitive user
		// information from being cached in the browser upon failed registration attempts
		// * used here so external sources (i.e. EvCheckout) benefit from disableCache
		$this->_controller->response->disableCache();

		// Setup the throttler, this stops someone continuously trying usernames from the same IP
		$registrationThrottler = $this->_getRegistrationThrottler();

		if (Configure::check('EvCore.userReCaptcha.showAfterFailedRegisterAttempts')) {
			// determine whether we're using login reCAPTCHA, cache allowed failed login
			// attempts total if so
			$showReCaptchaAfterAttempts = Configure::read('EvCore.userReCaptcha.showAfterFailedRegisterAttempts');
		}

		// if we're using the reCAPTCHA field, append to the validate
		if (
			isset($showReCaptchaAfterAttempts) &&
			$registrationThrottler->count() >= $showReCaptchaAfterAttempts
		) {
			$Model->validate = array_merge($Model->validate, $Model->validateReCaptcha());

			if (! empty($this->_controller->request->data['g-recaptcha-response'])) {
				// rebase the recaptcha request data to allow us to validate
				$this->_controller->request->data[$Model->alias]['g-recaptcha-response'] = $this->_controller->request->data['g-recaptcha-response'];
			} else {
				// assign an empty value if the recaptcha has been tampered with and unset
				$this->_controller->request->data[$Model->alias]['g-recaptcha-response'] = '';
			}
		}

		if (Configure::read('SiteSetting.users.enable_registration')) {
			// Check if the IP has been temporarily banned
			if (! $registrationThrottler->check() && ! isset($showReCaptchaAfterAttempts)) {
				// too many failed login attempts have been made
				$result = false;

				$returnData = array(
					'errors' => array(
						'title' => __('Login failed'),
						'description' => __(
							'You have failed to register %s times, You will not be able to register for the next %s minutes!',
							$registrationThrottler->getLimit(),
							$registrationThrottler->getTtl() / 60
						)
					)
				);
			} else {
				if ($Model->register($this->_controller->request->data)) {
					// clear the register throttler
					$registrationThrottler->clear();

					$activationType = Configure::read('SiteSetting.users.activation');

					if ($activationType == 'auto') {
						$message = Configure::read('EvCore.flashMessages.User.register.autoActivation');

						$fullUser = $Model->getUserForLogin($Model->id);
						$this->_controller->Auth->login($fullUser);
						$redirect = $this->_controller->Auth->redirect();

					} elseif ($activationType == 'email') {
						$message = Configure::read('EvCore.flashMessages.User.register.emailActivation');
					} else {
						$message = Configure::read('EvCore.flashMessages.User.register.manualActivation');
					}

					if (! isset($redirect)) {
						$redirect = array(
							'action' => 'login'
						);
					}

					$result = true;
					$returnData = array(
						'redirect' => $redirect,
						'message' => $message
					);

				} else {
					// Check register limits and write updated ones to session
					$registrationThrottler->hit();

					$errors = array(
						'description' => 'Please correct the following issues and try again',
						'list' => $Model->validationErrors
					);

					$result = false;
					$returnData = array(
						'errors' => $errors
					);
				}
			}
		} else {
			$result = false;
			$returnData = array(
				'errors' => 'Registration is disabled'
			);
		}

		// show the reCAPTCHA element after config defined failed attempts
		if (
			isset($showReCaptchaAfterAttempts) &&
			$registrationThrottler->count() >= $showReCaptchaAfterAttempts
		) {
			$this->_controller->set('showReCaptcha', true);
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}

/**
 * attempt to login a user
 *
 * @param 	object 	$Model 		The model object to use for saving
 * @return 	array 	$result 	Two element array - result(bool) / data(array)
 */
	public function login($Model) {
		// disable cache for the action's response. prevents potentially sensitive user
		// information from being cached in the browser upon failed login attempts
		// * used here so external sources (i.e. EvCheckout) benefit from disableCache
		$this->_controller->response->disableCache();

		// Setup the throttler, this stops someone continuously trying usernames from the same IP
		$loginThrottler = $this->_getLoginThrottler();

		if (Configure::check('EvCore.userReCaptcha.showAfterFailedLoginAttempts')) {
			// determine whether we're using login reCAPTCHA, cache allowed failed login
			// attempts total if so
			$showReCaptchaAfterAttempts = Configure::read('EvCore.userReCaptcha.showAfterFailedLoginAttempts');
		}

		$Model->validate = $Model->validateLogin();

		// if we're using the reCAPTCHA field, append to the validate
		if (
			isset($showReCaptchaAfterAttempts) &&
			$loginThrottler->count() >= $showReCaptchaAfterAttempts
		) {
			$Model->validate = array_merge($Model->validate, $Model->validateReCaptcha());

			if (! empty($this->_controller->request->data['g-recaptcha-response'])) {
				// rebase the recaptcha request data to allow us to validate
				$this->_controller->request->data[$Model->alias]['g-recaptcha-response'] = $this->_controller->request->data['g-recaptcha-response'];
			} else {
				// assign an empty value if the recaptcha has been tampered with and unset
				$this->_controller->request->data[$Model->alias]['g-recaptcha-response'] = '';
			}
		}

		$Model->set($this->_controller->request->data);

		// Check if the IP has been temporarily banned
		if (! $loginThrottler->check() && ! isset($showReCaptchaAfterAttempts)) {
			// too many failed login attempts have been made
			$result = false;

			$returnData = array(
				'errors' => array(
					'title' => __('Login failed'),
					'description' => __(
						'You have failed to login %s times, You will not be able to login for the next %s minutes!',
						$loginThrottler->getLimit(),
						$loginThrottler->getTtl() / 60
					)
				)
			);
		} else {
			// no login delay set, proceed as normal
			if ($Model->validates()) {
				if ($this->_controller->Auth->login()) {
					// successful login attempt

					// clear the login throttler
					$loginThrottler->clear();

					//Get a better version of the user model out in the format $data['User']
					$user = $this->_controller->Auth->user();

					$fullUser = $Model->getUserForLogin($user['id']);

					$this->_controller->Auth->login($fullUser);

					$result = true;

					$returnData = array(
						'redirect' => $this->_controller->Auth->redirect(),
						'message' => 'Login successful'
					);
				}
			}

			// login attempt failed
			if (! isset($result) || $result !== true) {

				// Check login limits and write updated ones to session
				$loginThrottler->hit();

				$result = false;

				$returnData = array(
					'errors' => array(
						'title' => __('Login failed'),
						'description' => __('The email and / or password entered were incorrect')
					)
				);

				unset($this->_controller->request->data[$Model->alias]['password']);
			}
		}

		// show the reCAPTCHA element after config defined failed attempts
		if (
			isset($showReCaptchaAfterAttempts) &&
			$loginThrottler->count() >= $showReCaptchaAfterAttempts
		) {
			$this->_controller->set('showReCaptcha', true);
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}

/**
 * Sets up the login throttler to stop users brute forcing from the same IP address.
 * @return mixed A rate throttler for the login using the client IP as the unique key
 */
	protected function _getLoginThrottler() {
		$config = [
			'driver' => 'file',
			'file' => [
				'cache_dir' => TMP . 'rate_limiting'
			]
		];
		$cacheAdapter = new DesarrollaCacheAdapter((new DesarrollaCacheFactory(null, $config))->make());

		$rateLimitAttempts = Configure::read('EvCore.userReCaptcha.showAfterFailedLoginAttempts') ? Configure::read('EvCore.userReCaptcha.showAfterFailedLoginAttempts') : $this->rateLimit['attempts'];

		$ratelimiter = new RateLimiter(
			new ThrottlerFactory(),
			new HydratorFactory(),
			$cacheAdapter,
			$rateLimitAttempts,
			$this->rateLimit['timeout']
		);

		// 2. Get a throttler for key login and the client IP
		$loginThrottler = $ratelimiter->get(['login', $this->_controller->request->clientIp()]);

		return $loginThrottler;
	}

/**
 * Sets up the register throttler to stop users brute forcing from the same IP address.
 * @return mixed A rate throttler for the registration using the client IP as the unique key
 */
	protected function _getRegistrationThrottler() {
		$config = [
			'driver' => 'file',
			'file' => [
				'cache_dir' => TMP . 'rate_limiting'
			]
		];
		$cacheAdapter = new DesarrollaCacheAdapter((new DesarrollaCacheFactory(null, $config))->make());

		$rateLimitAttempts = Configure::read('EvCore.userReCaptcha.showAfterFailedRegisterAttempts') ? Configure::read('EvCore.userReCaptcha.showAfterFailedRegisterAttempts') : $this->rateLimit['attempts'];

		$ratelimiter = new RateLimiter(
			new ThrottlerFactory(),
			new HydratorFactory(),
			$cacheAdapter,
			$rateLimitAttempts,
			$this->rateLimit['timeout']
		);

		// 2. Get a throttler for key register and the client IP
		$registrationThrottler = $ratelimiter->get(['register', $this->_controller->request->clientIp()]);

		return $registrationThrottler;
	}

}
