<?php

//Load our EvClassRegistry to allow us to load overriding models using PluginName.Modelname format
if (empty(App::location('EvClassRegistry'))) {
	App::uses('EvClassRegistry', 'EvCore.Lib');
}

if (!function_exists('ctype_digit')) {

/**
 * Replacement for php function ctype_digit as it doesn't exist in older versions of PHP.
 * To override this function, define it before loading EvCore.
 *
 * @param string $str The string to check.
 * @return bool True if $str is a digit.
 */
	function ctype_digit($str) {
		return (is_string($str) || is_int($str) || is_float($str)) &&
		preg_match('/^\d+\z/', $str);
	}

}

if (! function_exists('bytesFromFilesize')) {

/**
 * Convert a formatted filesize into bytes.
 *
 * @param string $filesize The formatted filesize. E.g. "1M" or "1MB"
 * @return int The fileseize in bytes.
 */
	function bytesFromFilesize($filesize) {
		$value = intval($filesize);
		$unit = preg_replace("/[\d ]/", '', $filesize);

		switch($unit)
		{
			case 'g':
			case 'G':
			case 'GB':
				$value *= 1024;
			case 'm':
			case 'M':
			case 'MB':
				$value *= 1024;
			case 'k':
			case 'K':
			case 'KB':
				$value *= 1024;
		}

		return $value;
	}

}

if (! function_exists('formatBytes')):

/**
 * Format a filesize into a string.
 *
 * @param int $bytes The value of bytes to format.
 * @return string The formatted bytes.
 */
	function formatBytes($bytes) {
		$unit = intval(log($bytes, 1024));
		$units = array('B', 'KB', 'MB', 'GB');

		if (array_key_exists($unit, $units) === true) {
			return sprintf('%d %s', $bytes / pow(1024, $unit), $units[$unit]);
		}
	}

endif;

if (!function_exists('filesizeFormatted')) {

/**
 * Returns a human readable filesize for the file at $path. To override this function, define it before loading
 * EvCore.
 *
 * @param string $path File system path to file.
 * @return string formatted filesize.
 */
	function filesizeFormatted($path) {
		$bytes = sprintf('%u', filesize($path));

		if ($bytes > 0) {
			return formatBytes($bytes);
		}

		return $bytes;
	}

}

if (!function_exists('getEnvironment')) {

/**
 * Gets the apps environment to set the correct DB config for the console. It will look for the existance
 * of a /etc/evoluted/production file and takes into account devXX.evoluted.net URLs.
 * To override this function, define it before loading EvCore.
 *
 * @return string either local, dev or production
 */
	function getEnvironment() {
		if (file_exists(DS . 'etc' . DS . 'evoluted' . DS . 'production')) {
			return preg_match('/\/production\//', getcwd()) ? 'production' : 'dev';
			// We check if the file exists first to speed up checking
		} elseif (file_exists(ROOT . DS . '.env') && getenv('ENVIRONMENT') == 'PREVIEW') {
			return 'preview';
		} elseif (preg_match('/\/staging\//', getcwd()) || preg_match('/\/html\//', getcwd())) {
			return 'dev';
		} elseif (preg_match('/\/uat\//', getcwd())) {
			return 'uat';
		} else {
			return 'local';
		}
	}

}

if (!function_exists('getCachePrefix')) {

/**
 * Get the cache prefix for the application. Each application cache on the same server should be prefixed
 * with a different string, to avoid Memcache and APC conflicts.
 * To override this function, define it before loading EvCore.
 *
 * @return string Application name for cache prefix.
 */
	function getCachePrefix() {
		if (isset($_SERVER['HTTP_HOST']) && $_SERVER['HTTP_HOST'] === 'cmd' && Configure::check('App.fullBaseUrl')) {
			$prefix = preg_replace('/[^\da-zA-Z]/i', '_', Configure::read('App.fullBaseUrl')) . '_';

			//Need to remove the protocol form the beginning of the string, the "://" will now be the fist "___".
			$searchString = '___';
			if (strpos($prefix, $searchString) !== false) {
				$prefix = substr($prefix, strpos($prefix, $searchString) + strlen($searchString));
			}
		} elseif (isset($_SERVER['HTTP_HOST'])) {
			$prefix = preg_replace('/[^\da-zA-Z]/i', '_', $_SERVER['HTTP_HOST']) . '_';
		} else {
			$prefix = 'myapp_';
		}

		return $prefix;
	}

}

if (!function_exists('loadAllPlugins')) {

/**
 * Load all the plugins in the project apart from EvCore as it is loaded separately before any other plugin.
 * To override this function, define it before loading EvCore.
 *
 * @return void.
 */
	function loadAllPlugins() {
		//Load all installed plugins. EvCore isn't loaded here as it has already been loaded.
		$allPlugins = App::objects('plugins');
		if ($allPlugins !== false) {
			$evCoreIndex = array_search('EvCore', $allPlugins);
			if ($evCoreIndex !== false) {
				unset($allPlugins[$evCoreIndex]);
			}
		}

		CakePlugin::load(
			$allPlugins,
			array(
				'routes' => true,
				'bootstrap' => true,
				'ignoreMissing' => true,
			)
		);
	}

}

if (!function_exists('checkRepository')) {

/**
 * Method to check the correct repository has been set in deploy.php
 * This is to only be called in DEVELOPMENT environments.
 *
 * @return void.
 */
	function checkRepository() {
		if (Configure::read('app.environment') === 'DEVELOPMENT') {
			$file = file_get_contents(ROOT . DS . '/deploy.php');
			if (strpos($file, 'git@git.evoluted.net:evoluted-core/core-cms.git') !== false) {
				die('Please update the repository set in <b>deploy.php</b>');
			}
		}
	}

}

$path = dirname(__FILE__);
$pluginDir = explode(DS, dirname($path . '../'));

$pluginName = array_pop($pluginDir);
$pluginConfig = Inflector::underscore($pluginName);
Configure::load($pluginName . '.config');

// Load overrides to settings if they exist
$configOverridePath = APP . 'Config' . DS;
$configOverride = 'Plugin' . DS . $pluginConfig . '.php';

if (file_exists($configOverridePath . $configOverride) && is_readable($configOverridePath . $configOverride)) {
	Configure::load($configOverride);
}

require_once (ROOT . DS . 'Plugin' . DS . 'EvCore' . DS . 'Config' . DS . 'events.php');

loadAllPlugins(); //Load all project plugins
