<?php

App::uses('AppHelper', 'View/Helper');
App::uses('BasicPermissions', 'EvCore.Lib');

class PermissionsHelper extends AppHelper {

	public $helpers = [
		'Form'
	];

/**
 * Constructor. Set up the permission library.
 *
 * @param View $View The View this helper is being attached to.
 * @param array $settings Configuration settings for the helper.
 */
	public function __construct(View $View, $settings = array()) {
		parent::__construct($View, $settings);

		$this->Permissions = new BasicPermissions(
			EvClassRegistry::init('EvCore.UserGroup')
		);
	}

/**
 * Magic method to check if the function exists in the lib file and call it if so
 * if not, it will try and return the string we were trying to Inflect
 * if it can't do that, it will just return false
 *
 */
	public function __call($name, $arguments) {
		if (method_exists($this->Permissions, $name)) {

			return call_user_func_array(array($this->Permissions, $name), $arguments);
		}

		// we can't find it, return false
		return false;
	}

/**
 * Inject a form onto an admin edit page that displays the available permissions available for a user group.
 *
 * @return void.
 */
	public function injectAdminForm() {
		if (!empty($this->_View->viewVars['permissions'])) {
			echo $this->element(
				'form_tab',
				array(
					'title' => 'Permissions',
					'tabContent' => $this->element('EvCore.UserGroups/permissions', compact('permissions')),
					'divOptions' => [
						'class' => 'user-group-permissions'
					]
				)
			);
		}
	}

/**
 * Display permissions in a collapsible hierarchial format.
 *
 * @param array  $permissions An array containing the permissions to display. Expected a threaded array.
 * @param string $modelName   The name of the model the permissions are being associated with.
 * @param int    $modelId     The id of the model the permissions are being assocaited with.
 * @param int    $depth       The current depth in the hierarchy.
 * @param string $prefix      The prefix to be added to child permissions so the correct permission node can
 *                            be identified.
 * @return string A html string to display.
 */
	public function adminDisplayPermissions($permissions, $modelName, $modelId, $depth = 0, $prefix = '') {
		$content = '';

		foreach ($permissions as $permission) {
			$permissionChildren = $permission['children'];

			if (!empty($permissionChildren) && $depth === 0) {
				//Top level category and has children
				$fieldsetClass = 'permission top-level-permission has-children collapsable-fieldset col-xs-12';
				$legendClass = 'col-xs-3';
			} elseif (empty($permissionChildren) && $depth === 0) {
				//Top level category and no children
				$fieldsetClass = 'permission top-level-permission no-children collapsable-fieldset col-xs-12';
				$legendClass = '';
			} elseif (!empty($permissionChildren)) {
				//category with children
				$fieldsetClass = 'permission has-children collapsable-fieldset col-xs-12';
				$legendClass = '';
			} else {
				//category without children
				$fieldsetClass = 'permission col-xs-12';
				$legendClass = '';
			}

			$content .= '<fieldset class="' . $fieldsetClass . '" data-depth="' . $depth . '"">';

			$content .= '<legend class="' . $legendClass . '">';

			$permissionPermissions = Hash::extract($permission['Aro'], '{n}[model=' . $modelName . '][foreign_key=' . $modelId . ']');
			$allowChecked = false;
			$denyChecked = false;
			if (!empty($permissionPermissions[0]['Permission'])) {
				if ($this->isPermissionAllowed($permissionPermissions[0]['Permission'])) {
					$allowChecked = true;
				} else {
					$denyChecked = true;
				}
			}

			$checkboxDivClass = (!empty($permissionChildren) || $depth === 0 ) ? 'col-xs-11' : 'col-xs-11';
			$content .= $this->_adminDisplayPermissionCheckboxes($permission['Aco']['alias'], $permission['Aco']['id'], $checkboxDivClass, $allowChecked, $denyChecked);

			$content .= '</legend>';

			if (!empty($permissionChildren)) {
				$content .= '<div>';
				$depth++;
				$content .= $this->adminDisplayPermissions($permissionChildren, $modelName, $modelId, $depth, $prefix . '/' . $permission['Aco']['alias']);
				$depth--;
				$content .= '</div>';
			}

			$content .= '</fieldset>';
		}

		return $content;
	}

/**
 * Display a set of checkboxes for a specific permission. A checkbox exists for allow and deny because the
 * permission can be set directly or inherited by it's parent. No selection will make a permission inherit.
 *
 * @param string $permissionName The name of the permission to display.
 * @param int    $permissionId   The id of the permission to edit.
 * @param string $divClass       The class to apply to the current checkbox group div.
 * @param bool   $allowChecked   True if the allow checkbox is to be checked. False if not.
 * @param bool   $denyChecked    True if the deny checkbox is to be checked. False if not.
 * @return string A html string containing the checkboxes to display for a permission.
 */
	protected function _adminDisplayPermissionCheckboxes($permissionName, $permissionId, $divClass = 'col-xs-11', $allowChecked = false, $denyChecked = false) {
		$content = '<div class="checkbox ' . $divClass . '" style="margin-left:25px;padding-top:0;">';

		$content .= $this->Form->checkbox('Permission.' . $permissionName . '.allow', array(
			'type' => 'checkbox',
			'value' => $permissionId,
			'hiddenField' => false,
			'checked' => $allowChecked,
			'class' => 'col-xs-1',
		));

		$content .= $this->Form->label(
			'Permission.' . $permissionName . '.allow',
			'Allow ' . $permissionName,
			[
				'class' => 'col-xs-4',
			]
		);

		$content .= $this->Form->checkbox('Permission.' . $permissionName . '.deny', array(
			'type' => 'checkbox',
			'value' => $permissionId,
			'hiddenField' => false,
			'checked' => $denyChecked,
			'class' => 'col-xs-1',
		));

		$content .= $this->Form->label(
			'Permission.' . $permissionName . '.deny',
			'Deny ' . $permissionName,
			[
				'class' => 'col-xs-4',
			]
		);

		$content .= '</div>';

		return $content;
	}

/**
 * Check if a permission is currently allowed or denied to be used when creating a hierarchy of permissions.
 * Loops through each permission action and checks if all the actions have been allowed.
 *
 * @param array $permission An array containing the permission actions.
 * @param int   $type       The action to check for.
 * @return bool
 */
	public function isPermissionAllowed($permission, $type = null) {
		if (!empty($type)) {
			if ($permission[$type] == 1) {
				return true;
			}
		} else {
			foreach (array_keys($permission) as $permissionType) {
				if (strpos($permissionType, '_') === 0) {
					if ($permission[$permissionType] != 1) {
						return false;
					}
				}
			}

			return true;
		}

		return null;
	}

/**
 * Check if a user group has permission to view page based on a url array.
 *
 * @param array $urlArray    A url array containing plugin, controller and action.
 * @param int   $userGroupId The id of the user group to check.
 * @return bool True if the user group has permission, false otherwise.
 */
	public function check($urlArray, $userGroupId) {
		$userGroupPath = [
			'model' => 'UserGroup',
			'foreign_key' => $userGroupId
		];

		$requestAclPath = 'controllers';
		if (!empty($urlArray['plugin'])) {
			$requestAclPath .= '/' . InflectorExt::camelize($urlArray['plugin']);
		}

		if (!empty($urlArray['controller'])) {
			$requestAclPath .= '/' . InflectorExt::camelize($urlArray['controller']);
		}

		if (!empty($urlArray['action'])) {
			$requestAclPath .= '/' . $urlArray['action'];
		}

		$result = EvClassRegistry::init('Permission')->check($userGroupPath, $requestAclPath);

		return $result;
	}
}
