<?php
/**
 * Copyable Behavior class file.
 *
 * Adds ability to copy a model record, including all hasMany and hasAndBelongsToMany
 * associations. Relies on Containable behavior, which this behavior will attach
 * on the fly as needed.
 *
 * HABTM relationships are just duplicated in the join table, while hasMany and hasOne
 * records are recursively copied as well.
 *
 * Usage is straightforward:
 * From model: $this->copy($id); // id = the id of the record to be copied
 * From controller: $this->MyModel->copy($id);
 *
 * @filesource
 * @author          Jamie Nay
 * @copyright       Jamie Nay
 * @license         http://www.opensource.org/licenses/mit-license.php The MIT License
 * @link            http://github.com/jamienay/copyable_behavior
 *
 * @author 			Mike Barlow
 * @link            https://github.com/snsmurf/copyable_behavior
 */
class CopyableBehavior extends ModelBehavior {

	/**
	 * Behavior settings
	 *
	 * @access public
	 * @var array
	 */
	public $settings = array();

	/**
	 * Array of contained models.
	 *
	 * @access public
	 * @var array
	 */
	public $contain = array();

	/**
	 * The full results of Model::find() that are modified and saved
	 * as a new copy.
	 *
	 * @access public
	 * @var array
	 */
	public $record = array();

	/**
	 * Default values for settings.
	 *
	 * - recursive: whether to copy hasMany and hasOne records
	 * - habtm: whether to copy hasAndBelongsToMany associations
	 * - stripFields: fields to strip during copy process
	 *
	 * @access protected
	 * @var array
	 */
	protected $defaults = array(
		'recursive' => true,
		'habtm' => true,
		'stripFields' => array('id', 'created', 'modified', 'lft', 'rght'),
		'recurse_level' => 1,
		'append_display_field' => 0
	);

	public $main_model = null;

	/**
	 * Configuration method.
	 *
	 * @param object $Model Model object
	 * @param array $config Config array
	 * @access public
	 * @return boolean
	 */
	public function setup(Model $Model, $config = array()) {
		$this->settings[$Model->alias] = array_merge($this->defaults, $config);
		return true;
	}

	/**
	 * Copy method.
	 *
	 * @param object $Model model object
	 * @param mixed $id String or integer model ID
	 * @access public
	 * @return boolean
	 */
	public function copy($Model, $id, $recurse_level = 1) {

		$this->main_model = $Model;

		$this->generateContain($Model, $recurse_level);

		$this->record = $Model->find('first', array(
			'conditions' => array($Model->alias.'.id' => $id),
			'contain' => $this->contain
		));


		if (empty($this->record)) {
			return false;
		}

		if (!$this->_convertData($Model)) {
			return false;
		}

		if ($recurse_level > 1) {
			$options = ['deep' => true];
		} else {
			$options = [];
		}

		return $this->_copyRecord($Model, $options);
	}

	/**
	 * Wrapper method that combines the results of _recursiveChildContain()
	 * with the models' HABTM associations.
	 *
	 * @param object $Model Model object
	 * @access public
	 * @return array;
	 */
	public function generateContain($Model, $recurse_level) {
		if (!$this->_verifyContainable($Model)) {
			return false;
		}

		$this->contain = array();

		if ($this->settings[$Model->alias]['habtm']) {
			$this->contain = array_keys($Model->hasAndBelongsToMany);
		}

		$this->contain = array_merge($this->_recursiveChildContain($Model, $recurse_level), $this->contain);
		return $this->contain;
	}

	/**
	 * Strips primary keys and other unwanted fields
	 * from hasOne and hasMany records.
	 *
	 * @param object $Model model object
	 * @param array $record
	 * @access protected
	 * @return array $record
	 */
	protected function _convertChildren($Model, $record) {
		$children = array_merge($Model->hasMany, $Model->hasOne);
		foreach ($children as $key => $val) {
			if (!isset($record[$key]) || empty($record[$key])) {
				continue;
			}
			if (isset($record[$key][0])) {
				foreach ($record[$key] as $innerKey => $innerVal) {
					$record[$key][$innerKey] = $this->_stripFields($Model->{$key}, $innerVal);

					if (array_key_exists($val['foreignKey'], $innerVal)) {
						unset($record[$key][$innerKey][$val['foreignKey']]);
					}

					$record[$key][$innerKey] = $this->_convertChildren($Model->{$key}, $record[$key][$innerKey]);
				}
			} else {
				$record[$key] = $this->_stripFields($Model->{$key}, $record[$key]);

				if (isset($record[$key][$val['foreignKey']])) {
					unset($record[$key][$val['foreignKey']]);
				}

				$record[$key] = $this->_convertChildren($Model->{$key}, $record[$key]);
			}
		}

		return $record;
	}

	/**
	 * Strips primary and parent foreign keys (where applicable)
	 * from $this->record in preparation for saving.
	 *
	 * @param object $Model Model object
	 * @access protected
	 * @return array $this->record
	 */
	protected function _convertData($Model) {
		$this->record[$Model->alias] = $this->_stripFields($Model, $this->record[$Model->alias]);
		$this->record = $this->_convertHabtm($Model, $this->record);
		$this->record = $this->_convertChildren($Model, $this->record);
		return $this->record;
	}

	/**
	 * Loops through any HABTM results in $this->record and plucks out
	 * the join table info, stripping out the join table primary
	 * key and the primary key of $Model. This is done instead of
	 * a simple collection of IDs of the associated records, since
	 * HABTM join tables may contain extra information (sorting
	 * order, etc).
	 *
	 * @param object $Model    Model object
	 * @access protected
	 * @return array modified $record
	 */
	protected function _convertHabtm($Model, $record) {

		if (!$this->settings[$Model->alias]['habtm']) {
			return $record;
		}

		foreach ($Model->hasAndBelongsToMany as $key => $val) {

			// check whether a custom relationship is used, if so then
			// overwrite the $val['className']
			if ($key != $val['className']) {

				$val['className'] = $key;

			}

			if (!isset($record[$val['className']]) || empty($record[$val['className']])) {
				continue;
			}

			$joinInfo = Hash::extract($record[$val['className']], '{n}.'.$val['with']);

			if (empty($joinInfo)) {
				continue;
			}

			foreach ($joinInfo as $joinKey => $joinVal) {
				$joinInfo[$joinKey] = $this->_stripFields($Model, $joinVal);

				if (array_key_exists($val['foreignKey'], $joinVal)) {
					unset($joinInfo[$joinKey][$val['foreignKey']]);
				}
			}

			$record[$val['className']] = $joinInfo;
		}

		return $record;
	}

	/**
	 * Performs the actual creation and save.
	 *
	 * @param object $Model Model object
	 * @access protected
	 * @return mixed
	 */
	protected function _copyRecord($Model, $options = []) {
		$Model->create();

		// Option to append a string to the end of the display field.
		if (! empty($this->settings[$Model->alias]['append_display_field'])) {
			$this->record[$Model->alias][$Model->displayField] = $this->record[$Model->alias][$Model->displayField] . $this->settings[$Model->alias]['append_display_field'];
		}

		$Model->set($this->record);
		return $Model->saveAll($Model->data, $options);
	}

	/**
	 * Generates a contain array for Containable behavior by
	 * recursively looping through $Model->hasMany and
	 * $Model->hasOne associations.
	 *
	 * @param object $Model Model object
	 * @access protected
	 * @return array
	 */
	protected function _recursiveChildContain($Model, $recurse_level = false) {
		$contain = array();
		if (isset($this->settings[$Model->alias]) && !$this->settings[$Model->alias]['recursive']) {
			return $contain;
		}

		if($recurse_level > 1)
		{
			$children = array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));
			  foreach ($children as $child) {
				  $contain[$child] = $this->_recursiveChildContain($Model->{$child}, $recurse_level - 1);
			  }

		}
		else {
			return array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));

		}

		  return $contain;
	}

	/**
	 * Strips unwanted fields from $record, taken from
	 * the 'stripFields' setting.
	 *
	 * @param object $Model Model object
	 * @param array $record
	 * @access protected
	 * @return array
	 */
	protected function _stripFields($Model, $record) { //debug($this->settings); debug($record);
		foreach ($this->settings[$this->main_model->alias]['stripFields'] as $field) {

			// check if we 're defining model.field for stripping fields
			if (strpos($field, '.') !== false) {

				$ex = explode('.', $field);
				$strip_model = $ex['0'];
				$strip_field = $ex['1'];

				// the models for the fields we are stripping don't match
				if ($strip_model != $Model->alias) {

					continue;
				}

			} else {

				$strip_field = $field;
			}

			// check we have it and strip
			if (array_key_exists($strip_field, $record)) {
				unset($record[$strip_field]);
			}
		}

		return $record;
	}

	/**
	 * Attaches Containable if it's not already attached.
	 *
	 * @param object $Model Model object
	 * @access protected
	 * @return boolean
	 */
	protected function _verifyContainable($Model) {
		if (!$Model->Behaviors->attached('Containable')) {
			return $Model->Behaviors->attach('Containable');
		}

		return true;
	}

}