<?php

App::uses('Folder', 'Utility');

abstract class Attachment extends AppModel {

	/**
	 * Number of document slots for the current model.
	 */
	public $documentSlots = 0;

	/**
	 * Defines number of image slots for the current model.
	 */
	public $imageSlots = 0;

	public $order = array(
		'sequence' => 'ASC'
	);

/**
 * Constructor
 * @param bool|int|string|array $id Set this ID for this model on startup, can also be an array of
 * options.
 * @param string $table Name of database table to use.
 * @param string $ds DataSource connection name.
 */
	public function __construct($id = false, $table = null, $ds = null) {
		if (Configure::check('EvCore.s3Upload.bucket')) {
			$this->actsAs['Upload.Upload']['filename']['handleUploadedFileCallback'] = 's3Upload';
		}
		parent::__construct($id, $table, $ds);
	}

	/**
	 * @see AppModel::beforeSave()
	 */
	public function beforeSave($options = array()) {
		if (!parent::beforeSave($options)) {
			return false;
		}

		// Insert sequence number if creating an attachment.
		if (empty($this->id) && empty($this->data[$this->alias]['sequence'])) {

			$this->data[$this->alias]['sequence'] = $this->getNextSequenceNo(array(
					'conditions' => array(
						'model' => $this->data[$this->alias]['model'],
						'model_id' => $this->data[$this->alias]['model_id'],
						'attachment_type' => $this->data[$this->alias]['attachment_type'],
					)
				)
			);

		}

		return true;
	}

/**
 * @see AppModel::beforeDelete()
 */
	public function beforeDelete($cascade = true) {
		// It's important we check the parent hasn't returned false before
		// proceeding.
		if (! parent::beforeDelete($cascade)) {
			return false;
		}

		$this->data = $this->findById($this->id);

		$success = true;

		// Remove file before we delete the attachment record.
		if (!empty($this->data[$this->alias]) && file_exists($this->data[$this->alias]['filepath'])) {

			$class = strtolower($this->name);

			$folder = new Folder('files/' . $class . '/' . $this->data[$this->alias]['dir']);

			$success = $folder->delete();
		}

		return $success;
	}

	/**
	 * callback method for uploading, allows to prefix filenames to prevent collisions
	 *
	 * @param string $field Name of field being modified
	 * @param string $currentName current filename
	 * @param array $data Array of data being manipulated in the current request
	 * @param array $options Array of options for the current rename
	 * @return string
	 */
	public function fixUploadFilename($field, $currentName, $data, $options) {
		$currentName = preg_replace('/[^A-Za-z0-9\-_.]/', '', $currentName);
		return time() . '_' . str_replace(' ', '_', $currentName);
	}

/**
 * Upload file to an S3 bucket.
 * @param string $field Field name
 * @param string $filename Temporary filename
 * @param string $destination Destination path
 * @return bool
 */
	public function s3Upload($field, $filename, $destination) {
		try {
			$s3Key = Configure::read('EvCore.s3Upload.key_prefix') . '/' . strtolower($this->name) . '/';
			$s3Key .= $this->id . '/' . basename($destination);
			$s3Client = new \Aws\S3\S3Client([
				'version' => 'latest',
				'region' => Configure::read('EvCore.s3Upload.region'),
				'credentials' => Configure::read('EvCore.s3Upload.credentials')
			]);
			$response = $s3Client->putObject([
				'Bucket' => Configure::read('EvCore.s3Upload.bucket'),
				'Key' => $s3Key,
				'Body' => fopen($filename, 'r'),
				'ACL' => 'public-read'
			]);
			// The file has been uploaded so the Upload behaviour has done its thing so needs
			// disabling so that we can modify the model's table without it expecting a file upload.
			$this->Behaviors->disable('Upload');
			$this->save([
				'id' => $this->id,
				$field => $response['ObjectURL']
			], ['callbacks' => false]);
		} catch (\Aws\S3\Exception\S3Exception $e) {
			$this->log('There was an error uploading the file to S3');
		}

		return ! empty($response['ObjectURL']);
	}
}
