<?php

class EvClassRegistry {

	/**
	 * function to mimic ClassRegistry::init(), but works with our overriding model workflow
	 * Pass PluginName.Model (normal models will work as normal) and it will check PluginNameModel first
	 * then standard PluginName.Model
	 *
	 * @param   String  Model to load
	 * @return  Object  Model object if found
	 */
	public static function init($modelClass, $strict = false) {
		list($plugin, $modelName) = pluginSplit($modelClass);

		// Evoluted Amends - Check for overriding model
		$override = self::findOverrideModel($modelClass);

		if ($override !== false) {
			$modelClass = $override;
		}

		return ClassRegistry::init(array(
			'class' => $modelClass, 'alias' => $modelName, 'id' => null
		), $strict);
	}

	/**
	 * check for overriding model
	 *
	 * @param   string          model to try and load
	 * @return  string|bool     model to load or false on fail
	 */
	public static function findOverrideModel($model) {
		if (is_array($model) && empty($model['class'])) {
			return false;
		}

		$modelClass = $model;
		if (is_array($model) && ! empty($model['class'])) {
			$modelClass = $model['class'];
		}

		if (strpos($modelClass, '.') !== false) {

			// first check if we can find an 'overriding model'
			$modelsPath = App::path('Model');
			$overwrittenFile = str_replace('.', '', $modelClass);
			$modelFile = $overwrittenFile . '.php';

			if (file_exists($modelsPath['0'] . $modelFile)) {
				if (is_array($model)) {
					$model['class'] = $overwrittenFile;
				} else {
					$model = $overwrittenFile;
				}

				return $model;
			}
		}

		return false;
	}

	/**
	 * given the type and the collection of items, check any plugin based ones for any
	 * overriding files
	 *
	 * @param   string      The type we are dealing with 'helper' / 'component'
	 * @param   array       Collection array $this->helpers / $this->components
	 * @return  array       The updated collection array
	 */
	public static function checkOverridingItems($type, $itemCollection) {
		$methods = array(
			'behaviors' => 'findOverrideBehavior',
			'components' => 'findOverrideComponent',
			'helpers' => 'findOverrideHelper'
		);

		if (isset($methods[$type]) && method_exists('EvClassRegistry', $methods[$type])) {
			$findMethod = $methods[$type];
		} else {
			return false;
		}

		$returnCollection = array();
		foreach ($itemCollection as $key => $settings) {

			// try and find the class name and then try and find an overriding method
			if (is_array($settings) && ! empty($settings['className'])) {
				$itemName = $settings['className'];
			} elseif (is_array($settings) && empty($settings['className'])) {
				$itemName = $key;
			} elseif (empty($settings)) {
				$itemName = $key;
			} else {
				$itemName = $settings;
			}

			// if no overriding file is found, original file will be returned
			$itemToLoad = EvClassRegistry::$findMethod($itemName);

			// rebuild the collection array
			if (is_array($settings)) {

				$newKey = self::getNewArrayKey($key);

				$returnCollection[$newKey] = $settings;
				$returnCollection[$newKey]['className'] = $itemToLoad;
			} else {

				if (empty($settings)) {
					$newKey = self::getNewArrayKey($key);
				} else {
					$newKey = self::getNewArrayKey($settings);
				}

				$returnCollection[$newKey] = array(
					'className' => $itemToLoad
				);
			}
		}

		return $returnCollection;
	}

	/**
	 * check to see if a behavior has been overridden
	 *
	 * @param   string      behavior to check
	 * @return  string      behavior to load
	 */
	public static function findOverrideBehavior($behavior) {
		if (strpos($behavior, '.') !== false) {

			$behaviorsPath = App::path('Model/Behavior');
			$overwrittenFile = str_replace('.', '', $behavior);
			$behaviorFile = $overwrittenFile . 'Behavior.php';

			if (file_exists($behaviorsPath['0'] . $behaviorFile)) {

				return $overwrittenFile;
			}
		}

		return $behavior;
	}

	/**
	 * check to see if a component has been overridden
	 *
	 * @param   string      Component to check
	 * @return  string      Component to load
	 */
	public static function findOverrideComponent($component) {
		if (strpos($component, '.') !== false) {

			$componentPath = App::path('Controller/Component');
			$overwrittenFile = str_replace('.', '', $component);
			$componentFile = $overwrittenFile . 'Component.php';

			if (file_exists($componentPath['0'] . $componentFile)) {

				return $overwrittenFile;
			}
		}

		return $component;
	}

	/**
	 * check to see if a helper has been overridden
	 *
	 * @param   string      Component to check
	 * @return  string      Component to load
	 */
	public static function findOverrideHelper($helper) {
		if (strpos($helper, '.') !== false) {

			$helperPath = App::path('View/Helper');
			$overwrittenFile = str_replace('.', '', $helper);
			$helperFile = $overwrittenFile . 'Helper.php';

			if (file_exists($helperPath['0'] . $helperFile)) {

				return $overwrittenFile;
			}
		}

		return $helper;
	}

	/**
	 * get new item key for the array item
	 * the key will generally get used by cake as the 'name' to use when setting to $this
	 * with the amends to the component / helper array setting className, means this cannot be plugin Notation
	 *
	 * @param   string      Full item to get key for
	 * @return  string      New key to use in array
	 */
	public static function getNewArrayKey($itemKey) {
		if (strpos($itemKey, '.') !== false) {
			list($plugin, $newKey) = explode('.', $itemKey, 2);
			return $newKey;
		}

		return $itemKey;
	}

	/**
	 * given the model object, try and work out the plugin.name version of the model
	 * incase any models have been extend, init method above can be used to load the model
	 *
	 * @param Object    Model Object
	 * @return string   Model name in the form of Plugin.Model (or just Model if none plugin)
	 */
	public static function getNameFromModel($Model) {
		if (! empty($Model->plugin)) {

			return $Model->plugin . '.' . $Model->name;

		} else {

			$className = get_class($Model);
			$pluginName = preg_replace('/' . $Model->name . '$/', '', $className);

			if (! empty($pluginName) && CakePlugin::loaded($pluginName)) {

				return $pluginName . '.' . $Model->name;
			}
		}

		return $Model->name;
	}
}
