<?php

App::uses('AppShell', 'Console/Command');

App::uses('AclShell', 'Console/Command');
App::uses('AclExtrasShell', 'AclExtras.Console/Command');

class PermissionsShell extends AppShell {

	protected $_Aro;

	protected $_Aco;

	protected $_Permission;

	public function getOptionParser() {
		$parser = parent::getOptionParser();
		$parser->addOption(
			'update-default-permissions',
			[
				'short' => 'u',
				'help' => 'Once synchronised, update the default permissions for the default user groups',
				'boolean' => true
			]
		);
		$parser->addOption(
			'skip-sync',
			[
				'short' => 's',
				'help' => 'Skip the synchronisation step.',
				'boolean' => true
			]
		);

		return $parser;
	}

	public function startup() {
		parent::startup();

		$this->_Aro = EvClassRegistry::init('Aro');
		$this->_Aco = EvClassRegistry::init('Aco');
		$this->_Permission = EvClassRegistry::init('EvCore.EvCorePermission');
	}

	public function sync() {
		if (empty($this->params['skip-sync'])) {
			//Pull in all the controller actions to be used as aco records.
			$aclExtrasShell = new AclExtrasShell();
			$aclExtrasShell->startup();
			$aclExtrasShell->aco_sync();
		}

		if (!empty($this->params['update-default-permissions'])) {
			$model = 'UserGroup';
			$foriegnKey = 1;

			$this->out("Updating default permissions for " . $model . ' ' . $foriegnKey);
			$this->_allowGlobalPermission($model, $foriegnKey);

			$model = 'UserGroup';
			$foriegnKey = 2;

			$this->out("Updating default permissions for " . $model . ' ' . $foriegnKey);
			$this->_allowGlobalPermission($model, $foriegnKey);

			$model = 'UserGroup';
			$foriegnKey = 3;

			$this->out("Updating default permissions for " . $model . ' ' . $foriegnKey);
			$this->_allowGlobalPermission($model, $foriegnKey);
			$this->_denyAdminPermission($model, $foriegnKey);
		}

		return null;
	}

/**
 * Give a user group permission to the global controller aco. This would allow a user access to any
 * controller action if it hasn't been given a specific permission.
 *
 * @param string $aroModel The model alias of the aro to attach the permission to.
 * @param int    $aroKey   The id of the aro to attach the permission to.
 * @return bool True if the permission is added successfully, False otherise.
 */
	protected function _allowGlobalPermission($aroModel, $aroKey) {
		$globalControllerAco = $this->_Aco->find(
			'first',
			[
				'conditions' => [
					'Aco.parent_id' => null,
					'Aco.alias' => 'controllers',
				]
			]
		);

		if (!empty($globalControllerAco)) {
			return $this->_allowPermissions([$globalControllerAco['Aco']['id']], $aroModel, $aroKey);
		}

		return false;
	}

/**
 * Give a user group permission to all admin methods on all controllers.
 *
 * @param string $aroModel The model alias of the aro to attach the permission to.
 * @param int    $aroKey   The id of the aro to attach the permission to.
 * @return bool True if the permission is added successfully, False otherise.
 */
	protected function _allowAdminPermission($aroModel, $aroKey) {
		$adminAcos = $this->_findControllerAcoNodesByAlias('admin_%');

		$acoIds = Hash::extract($adminAcos, '{n}.Aco.id');

		if (!empty($acoIds)) {
			return $this->_allowPermissions($acoIds, $aroModel, $aroKey);
		}

		return false;
	}

/**
 * Allow an aro node to have access to all EvTemplate ajax methods so that they can use the template tabs and
 * fields on other pages. This doesn't give access to the EvTemplate admin pages.
 *
 * @param string $aroModel The model of the aro node to deny.
 * @param int    $aroKey   The foreign key of the aro node to deny.
 * @return bool True if the ajax template controller actions were allowed, false otherwise.
 */
	protected function _allowAdminTemplateAjax($aroModel, $aroKey) {
		$templateAco = $this->_Aco->find(
			'first',
			[
				'conditions' => [
					'Aco.alias' => 'EvTemplates',
				]
			]
		);

		if (!empty($templateAco['Aco']['lft']) && !empty($templateAco['Aco']['rght'])) {
			$templateAjaxAcos = $this->_findControllerAcoNodesByAlias(
				'ajax_%',
				'LIKE',
				[
					'Aco.lft >' => $templateAco['Aco']['lft'],
					'Aco.rght <' => $templateAco['Aco']['rght'],
				]
			);

			$templateAjaxAcoIds = Hash::extract($templateAjaxAcos, '{n}.Aco.id');

			if (!empty($templateAjaxAcoIds)) {
				return $this->_allowPermissions($templateAjaxAcoIds, $aroModel, $aroKey);
			}
		}
	}

/**
 * Deny an aro node of all admin controller actions.
 *
 * @param string $aroModel The model of the aro node to deny.
 * @param int    $aroKey   The foreign key of the aro node to deny.
 * @return bool True if the admin controller actions were denied, false otherwise.
 */
	protected function _denyAdminPermission($aroModel, $aroKey) {
		$adminAcos = $this->_findControllerAcoNodesByAlias('admin_%');

		$acoIds = Hash::extract($adminAcos, '{n}.Aco.id');

		if (!empty($acoIds)) {
			return $this->_denyPermissions($acoIds, $aroModel, $aroKey);
		}

		return false;
	}

/**
 * Allow all aco nodes for a provided aro node. The aro node is found using the aro model and aro foreign key.
 * All actions for the permission are allowed.
 *
 * @param array  $acoIds   An array list of aco node ids to allow.
 * @param string $aroModel The model to find the aro node with.
 * @param int    $aroKey   The foreign key to find the aro node with.
 * @return bool|null True if all aco nodes were allowed, false otherwise. Null if aro node wasn't found.
 */
	protected function _allowPermissions($acoIds, $aroModel, $aroKey) {
		$currentAro = $this->_getAroFromModelKey($aroModel, $aroKey);

		if (!empty($currentAro['Aro']['id'])) {
			$permissions = [];
			foreach ($acoIds as $acoId) {
				$permissions[] = $this->_allow($currentAro['Aro']['id'], $acoId);
			}

			if (count($permissions) == count($acoIds)) {
				return true;
			} else {
				return false;
			}
		}

		return null;
	}

/**
 * Deny all aco nodes for a provided aro node. The aro node is found using the aro model and aro foreign key.
 * All actions for the permission are denied.
 *
 * @param array  $acoIds   An array list of aco node ids to deny.
 * @param string $aroModel The model to find the aro node with.
 * @param int    $aroKey   The foreign key to find the aro node with.
 * @return bool|null True if all aco nodes were denied, false otherwise. Null if aro node wasn't found.
 */
	protected function _denyPermissions($acoIds, $aroModel, $aroKey) {
		$currentAro = $this->_getAroFromModelKey($aroModel, $aroKey);

		if (!empty($currentAro['Aro']['id'])) {
			$permissions = [];
			foreach ($acoIds as $acoId) {
				$result = $this->_allow($currentAro['Aro']['id'], $acoId, '*', -1);
				$permissions[] = $result;
			}

			if (count($permissions) == count($acoIds)) {
				return true;
			} else {
				return false;
			}
		}

		return null;
	}

/**
 * Get an aro node based on the model and foreign key.
 *
 * @param string $aroModel The model to search with.
 * @param int    $aroKey   The foreign key to search with.
 * @return array The found aro node.
 */
	protected function _getAroFromModelKey($aroModel, $aroKey) {
		return $this->_Aro->find(
			'first',
			[
				'conditions' => [
					'Aro.model' => $aroModel,
					'Aro.foreign_key' => $aroKey,
				]
			]
		);
	}

/**
 * Save/update a permission. Creates a link between an aro node and an aco node. All actions can be updated
 * or individual actions can be updated.
 *
 * @param int          $aroId      The id of the aro to link.
 * @param int          $acoId      The id of the aco to link.
 * @param string|array $actions    An array of actions to update. "*" updates all actions, name of action to update or
 *                                 an array of actions to update.
 * @param int          $permission Whether to allow or deny the actions. 1 to allow, -1 to deny.
 * @return bool True if permission was updated sucessfully, false otherwise.
 */
	protected function _allow($aroId, $acoId, $actions = '*', $permission = 1) {
		$savePermission = [
			'aro_id' => $aroId,
			'aco_id' => $acoId,
		];

		//See if there is a current permission to update.
		$currentPermission = $this->_Permission->find(
			'first',
			[
				'conditions' => [
					'Permission.aro_id' => $aroId,
					'Permission.aco_id' => $acoId,
				]
			]
		);

		if (!empty($currentPermission)) {
			$savePermission['id'] = $currentPermission['Permission']['id'];
		}

		//Get the crud action types.
		$permissionActions = $this->_Permission->getAcoKeys($this->_Permission->schema());

		foreach ($permissionActions as $permissionAction) {
			if (!empty($currentPermission) && isset($currentPermission['Permission'][$permissionAction])) {
				$savePermission[$permissionAction] = $currentPermission['Permission'][$permissionAction];
			}

			if (is_array($actions)) {
				foreach ($actions as $action) {
					if (in_array($action, $permissionActions)) {
						$savePermission[$action] = $permission;
					}
				}
			} elseif (is_string($actions) && $actions != '*') {
				if (in_array($actions, $permissionActions)) {
					$savePermission[$actions] = $permission;
				}
			} elseif ($actions == '*') {
				foreach ($permissionActions as $permissionAction) {
					$savePermission[$permissionAction] = $permission;
				}
			}
		}

		$this->_Permission->create();
		return $this->_Permission->save($savePermission);
	}

/**
 * Find all the aco nodes by their alias. Defaults to compare by LIKE but can be changed by providing the
 * comparator parameter. Extra conditions can be provided and are merged with the defaults. To avoid
 * returning permissions that aren't controller methods, the aco nodes are limited between the gloabl
 * controller nodes left and right values.
 *
 * @param string $alias           The alias to search by.
 * @param string $comparator      The comparator to use with the alias. Defautls to like.
 * @param array  $extraConditions An array to provide extra conditions to the query, these are merged with the defaults.
 * @return array The aco nodes that are found.
 */
	protected function _findControllerAcoNodesByAlias($alias, $comparator = 'LIKE', $extraConditions = []) {
		if (empty($this->_globalControllerNode)) {
			//If there are other permissions that aren't controller based then we want to limit the nods to
			//controller actions.
			$this->_globalControllerNode = $this->_Aco->find(
				'first',
				[
					'conditions' => [
						'Aco.parent_id' => null,
						'Aco.alias' => 'controllers',
					]
				]
			);
		}

		//Default aco node conditions.
		$acoConditions = [
			'Aco.alias ' . $comparator => $alias,
		];

		if (!empty($this->_globalControllerNode)) {
			$acoConditions['Aco.lft >'] = $this->_globalControllerNode['Aco']['lft'];
			$acoConditions['Aco.rght <'] = $this->_globalControllerNode['Aco']['rght'];
		}

		$acoConditions = Hash::merge($acoConditions, $extraConditions);

		return $this->_Aco->find(
			'all',
			[
				'conditions' => $acoConditions,
			]
		);
	}
}
