<?php

App::uses('CakeTime', 'Utility');
App::uses('CakeEmail', 'Network/Email');
App::uses('AuthComponent', 'Controller/Component');
App::uses('SessionComponent', 'Controller/Component');

/**
 * Business logic for user accounts
 */
class User extends AppModel {

	public $displayField = "email";

	public $order = array('email' => 'ASC');

	public $belongsTo = array(
		'UserGroup'
	);

	public $validate = array(
		'user_group_id' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'User group cannot be blank'
			),
		),
		'email' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Email cannot be left blank'
			),
			'email' => array(
				'rule' => 'email',
				'message' => 'You must enter a valid email address'
			),
			'customUnique' => array(
				'rule' => 'customUnique',
				'message' => 'A user with that email address already exists'
			)
		),
		'password' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'Password cannot be left blank',
				'on' => 'create'
			)
		),
		'confirm_password' => array(
			'required' => array(
				'rule' => 'notBlank',
				'message' => 'You must confirm your password',
				'on' => 'create'
			),
			'matches' => array(
				'rule' => array('checkMatches', 'password'),
				'message' => 'The passwords do not match'
			)
		)
	);

	public $virtualFields = array(
		'name' => 'CONCAT(IFNULL(first_name, \'\'), " ", IFNULL(last_name, \'\'))'
	);

	/**
	 * Ensure the password is encrypted if present before it's saved to the DB
	 *
	 * @param array $options beforeSave options
	 * @return void
	 */
	public function beforeSave($options = array()) {
		if (isset($this->data[$this->alias]['password']) && ! empty($this->data[$this->alias]['password'])) {
			$this->data[$this->alias]['password'] = AuthComponent::password($this->data[$this->alias]['password']);
		} else {

			unset($this->data[$this->alias]['password']);
		}

		return;
	}

	/**
	 * Validate password rules
	 */
	public function validatePassword($id = null, $minCharacters = 15) {
		if (! empty($id)) {
			$existingPassword = $this->fieldOnly('password', array('id' => $id));
		}

		$rules = array(
			'password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Password cannot be left blank',
					'on' => 'create'
				),
				'minlength' => array(
					'rule' => array('minLength', $minCharacters),
					'message' => 'Your password must be a minimum of ' . $minCharacters . ' characters long'
				),
				'alphaNumeric' => array(
					'rule' => 'customAlphaNumericSpecial',
					'message' => 'Your password must contain at least one alphabetic, one numeric character and one special character (!@#$&*)'
				),
				'case' => array(
					'rule' => 'customLowercaseAndUppercase',
					'message' => 'Your password must contain at least one lowercase and one uppercase character'
				),
				'dictionary' => array(
					'rule' => 'customPasswordDictionary',
					'message' => 'Your password cannot contain any dictionary words'
				),
			),
			'confirm_password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'You must confirm your password'
				),
				'matches' => array(
					'rule' => array('checkMatches', 'password'),
					'message' => 'The passwords do not match'
				)
			)
		);

		if (! empty($existingPassword)) {
			$rules['password']['existing'] = array(
				'rule' => array('customNotMatchExisting', $existingPassword),
				'message' => 'The new password must be different to the previous password used'
			);
		}

		return $rules;
	}

/**
 * Validation rules called from admin_account and account actions
 */
	public function validateAccount() {
		$rules = $this->validate;

		unset($rules['password']);
		unset($rules['confirm_password']);

		return $rules;
	}

/**
 * Validation rules called from the admin_edit action
 */
	public function validateEdit() {
		$rules = $this->validate;

		// add in a permissions rule to check that the user being edited cannot be
		// granted higher permissions than the current auth user
		$rulesUserGroupPermissions = array(
			'user_group_id' => array(
				'permissionLevel' => array(
					'rule' => 'validatePermissionLevel',
					'message' => 'The user group selection was invalid'
				)
			)
		);

		return array_merge_recursive($rules, $rulesUserGroupPermissions);
	}

	/**
	 * Validation rules for a login form submission
	 *
	 * @return array
	 */
	public function validateLogin() {
		return array(
			'email' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Email cannot be left blank.'
				),
				'email' => array(
					'rule' => 'email',
					'message' => 'You must enter a valid email address'
				)
			),
			'password' => array(
				'required' => array(
					'rule' => 'notBlank',
					'message' => 'Password cannot be left blank'
				)
			)
		);
	}

/**
 * Validation rules to check the supplied reCAPTCHA submission
 *
 * @return array
 */
	public function validateReCaptcha() {
		return array(
			'g-recaptcha-response' => array(
				'recaptcha' => array(
					'rule' => array('customValidateReCaptcha'),
					'message' => 'Please complete the reCAPTCHA field'
				)
			)
		);
	}

	/**
	 * custom unique check, need to check to see if user is guest user first
	 *
	 * @param array $check email field to check
	 * @return 	bool
	 */
	public function customUnique($check) {
		$conditions = array(
			$this->alias . '.email' => $check['email'],
			$this->alias . '.is_guest_user' => 0
		);

		if ($this->id > 0) {
			$conditions[] = $this->alias . '.id != ' . $this->id;
		}

		$count = $this->find('count', array('conditions' => $conditions));

		return ($count >= 1 ? false : true);
	}

	/**
	 * Returns true if an email already exists
	 *
	 * @param array $check email field to check
	 * @return true if email doesn't exists, otherwise false
	 */
	public function checkIsRegistered($check) {
		App::uses('User', 'Model');

		$value = array_pop($check);

		$user = $this->findByEmail($value);

		return empty($user);
	}

	/**
	 * Gets the user and all associated data out that is needed for login
	 * @param  int $id  The ID of the user
	 * @return array 	User array
	 */
	public function getUserForLogin($id) {
		$fullUser = $this->find('first', array(
				'conditions' => array(
					$this->alias . '.id' => $id
				),
				'contain' => array(
					'UserGroup'
				)
			)
		);

		return $fullUser;
	}

	/**
	 * Sets a password reset code for a user.
	 *
	 * @param integer $userId
	 * @return boolean - returns false if the user cannot be found
	 */
	public function resetPassword($userId) {
		$user = $this->findById($userId);

		if (!empty($user)) {
			$code = md5(uniqid());
			$this->id = $userId;
			$this->save([
				'password_reset_code' => $code,
				'password_reset_code_expires' => gmdate('Y-m-d H:i:s', strtotime('+30 minutes'))
			]);

			App::uses('EvUserPasswordResetListener', 'EvCore.Lib/Event');
			$this->getEventManager()->attach(new EvUserPasswordResetListener());
			$event = new CakeEvent('Model.User.passwordReset', $this, array(
					'Model' => $this->alias,
					'User' => $user,
					'code' => $code,
					'url' => Router::url(
						array(
							'plugin' => false,
							'controller' => 'ev_core_users',
							'action' => 'password_reset_callback',
							$code
						),
						true
					)
				)
			);
			$this->getEventManager()->dispatch($event);

			return true;
		}

		return false;
	}

	/**
	 * register a new user
	 *
	 * @param 	array 	data from request->data
	 * @return  bool
	 */
	public function register($data) {
		$activationType = Configure::read('SiteSetting.users.activation');

		$data[$this->alias]['is_active'] = 0;
		if (
			$activationType === 'auto' ||
			(isset($data[$this->alias]['is_guest_user']) && (bool)$data[$this->alias]['is_guest_user'] === true)
		) {
			$data[$this->alias]['is_active'] = 1;
		}

		if (
			isset($data[$this->alias]['is_guest_user']) &&
			(bool)$data[$this->alias]['is_guest_user'] === true
		) {
			$data[$this->alias]['password'] = $data[$this->alias]['confirm_password'] = 'gu3st-u5Er!';
		}

		// if a user group id has been requested then check that it is allowed
		$allowedUserGroupIds = Configure::read('EvCore.allowed_user_group_registration_ids');
		if (
			! empty($data[$this->alias]['user_group_id']) &&
			in_array($data[$this->alias]['user_group_id'], $allowedUserGroupIds)
		) {
			$userGroupId = $data[$this->alias]['user_group_id'];
		} else {
			// no user_group_id was requested, or the requested value is not allowed
			$userGroupId = Configure::read('EvCore.default_user_group_registration_id');
		}

		$data[$this->alias]['user_group_id'] = $userGroupId;

		$return = $this->saveAll($data);

		if ($return) {
			// dispatch new user event
			$this->getEventManager()->dispatch(
				new CakeEvent('Model.User.newUser', $this, array(
					'Model' => $this->alias,
					'user' => $data
				))
			);

			if (! isset($data[$this->alias]['is_guest_user']) || intval($data[$this->alias]['is_guest_user']) === 0) {
				if ($activationType == 'auto') {

					// dispatch auto activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.autoActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data
						))
					);

				} elseif ($activationType == 'email') {

					$code = hash('sha1', $this->id . '-' . $data[$this->alias]['email'] . '-' . time());
					$this->saveField('verification_code', $code);

					// dispatch email activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.emailActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data,
							'url' => Router::url(
								array(
									'plugin' => false,
									'controller' => 'ev_core_users',
									'action' => 'verify_email',
									$code
								),
								true
							)
						))
					);
				} elseif ($activationType == 'manual') {

					// dispatch email activate event
					$this->getEventManager()->dispatch(
						new CakeEvent('Model.User.manualActivate', $this, array(
							'Model' => $this->alias,
							'User' => $data,
							'url' => Router::url(
								array(
									'admin' => true,
									'plugin' => false,
									'controller' => 'ev_core_users',
									'action' => 'edit',
									$this->id
								),
								true
							)
						))
					);
				}
			}
		}

		return $return;
	}

	/**
	 * Verify a user account from the code given in email
	 *
	 * @param 	int 	User ID
	 * @param 	string 	Verification code
	 * @return 	bool
	 */
	public function verifyFromEmail($id, $code) {
		$return = $this->updateAll(
			array(
				'User.is_active' => 1,
				'User.verification_code' => null,
				'modified' => '"' . CakeTime::format('Y-m-d H:i:s', time()) . '"'
			),
			array(
				'User.id' => $id,
				'User.verification_code' => $code
			)
		);

		// dispatch email activate event
		$this->getEventManager()->dispatch(
			new CakeEvent('Model.User.emailVerified', $this, array(
				'Model' => $this->alias,
				'User' => $this->readForView($id)
			))
		);

		return $return;
	}

	/**
	 * Checks whether the required level of user group is less than or equal to the current auth user
	 *
	 * @param array $check containing user group to check against
	 * @return bool based on whether the items validate correctly
	 */
	public function validatePermissionLevel($check) {
		if (! empty($check['user_group_id'])) {
			$requiredLevel = $this->UserGroup->fieldOnly(
				'level',
				array('UserGroup.id' => $check['user_group_id'])
			);

			$authUserLevel = SessionComponent::read('Auth.User.UserGroup.level');

			return (! empty($authUserLevel) && $authUserLevel <= $requiredLevel);
		}

		return false;
	}

/**
 * Validates the given reCAPTCHA field against the Google reCAPTCHA API
 *
 * @param array $check Contains the field to validate against
 * @return bool
 */
	public function customValidateReCaptcha($check) {
		$userReCaptchaSecret = Configure::read('EvCore.userReCaptcha.secret');

		$recaptcha = new \ReCaptcha\ReCaptcha($userReCaptchaSecret);
		$resp = $recaptcha->verify(current($check), $_SERVER['REMOTE_ADDR']);

		return $resp->isSuccess();
	}

/**
 * Custom validation rule to check for at least one occurance of an alphabetic character, at least one
 * occurance of a numeric character and at least one special character
 */
	public function customAlphaNumericSpecial($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		return preg_match('/(?=.*\d)(?=.*[a-zA-Z])(?=.*[!@#$&*])/', $value);
	}

/**
 * Custom validation rule to check that the requested password doesn't include
 * and values found in the password dictionary
 *
 * @param array $check The field to validate against
 * @return bool
 */
	public function customPasswordDictionary($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		// query the database to check that the password does not contain
		// and dictionary words
		return ! (bool)EvClassRegistry::init('EvCore.PasswordDictionary')->fieldOnly('id', [
			'LOCATE(password, \'' . $value . '\')'
		]);
	}

/**
 * Custom validation rule to check for at least one occurance of an lowercase character and at least one
 * occurance of a uppercase character
 */
	public function customLowercaseAndUppercase($check) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		return preg_match('/(?=.*[a-z])(?=.*[A-Z])/', $value);
	}

/**
 *	Custom validation rule to check that the newly supplied password is different to the previously stored password
 */
	public function customNotMatchExisting($check, $existing) {
		// $data array is passed using the form field name as the key
		// have to extract the value to make the function generic
		$value = array_values($check);
		$value = $value[0];

		// has the requested password value - mirroring the functionality used by the Auth component
		$value = Security::hash($value, 'sha1', true);

		return (empty($existing) || $value != $existing);
	}
}
