<?php

App::uses('Controller', 'Controller');
App::uses('InflectorExt', 'Lib');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('AuthExtComponent', 'EvCore.Controller/Component');
App::uses('CakeTime', 'Utility');

/**
 * Evoluted Core Application Controller
 *
 * This is the main controller for the Evoluted CMS.
 * AppController extends from this controller to allow all methods and functionality to be accessible
 *
 * @package       app.Controller
 * @link http://book.cakephp.org/2.0/en/controllers.html#the-app-controller
 */
class EvCoreController extends Controller {

	/**
	 * override the View class so we can check helpers when loading
	 * for overridden files
	 */
	public $viewClass = 'EvCore.EvCustom';

	/**
	 * Default components
	 */
	public $components = array(
		'Auth' => array(
			'className' => 'EvCore.AuthExt',
			'loginAction' => array(
				'plugin' => false,
				'controller' => 'ev_core_users',
				'action' => 'login'
			),
			'logoutRedirect' => '/',
			'authenticate' => array(
				'Form' => array(
					'fields' => array('username' => 'email'),
					'scope' => array('User.is_active' => true)
				)
			),
			'authorize' => 'Controller' // override isAuthorized() to customise
		),
		'Session',
		'Flash',
		'EvSiteSettings.SiteSettings',
		'MetaData.Meta',
		'Routable.Routable',
		'DebugKit.Toolbar',
		'EvTemplates.Tpl',
		'EvCore.Permissions',
		'EvCore.Tidy' => array('minify' => true, 'js' => false, 'css' => false),
		'Paginator',
		'EvCore.MailQueue'
	);

	/**
	 * Default helpers
	 */
	public $helpers = array(
		'Html' => array(
			'className' => 'HtmlExt'
		),
		'Form' => array(
			'className' => 'EvForm.FormExt'
		),
		'Session',
		'Flash',
		'Number' => array(
			'className' => 'EvCore.NumberExt'
		),
		'InflectorExt',
		'EvImg.Img',
		'Listing',
		'EvNavigation.Navigation',
		'Routable.Route',
		'EvCustomFields.CustomFields',
		'EvCore.Permissions',
		'EvCore.Page'
	);

	/**
	 * List of admin actions the controller supports
	 * Used when you want to stop any administrator from performing an action
	 * e.g. If you don't want an admin to delete something, override this in the controller and remove 'admin_delete'
	 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_toggle',
		'admin_copy',
		'admin_view',
		'admin_reorder'
	);

	public $reordering = true;

	/**
	 * Paginator defaults for admin.
	 *
	 * @var array
	 */
	public $adminPaginate = array(
		//'limit' => 25
	);

	/**
	 * store the pluginDot data for this controller
	 * so we can use later when checking the main model for any behavior form injections
	 * @var string|null
	 */
	public $pluginDot = null;

	/**
	 * all the components / helpers we want to inject into form
	 * @var array
	 */
	public $toInject = array(
		'components' => array(),
		'helpers' => array()
	);

	/**
	 * Array of options to pass into the save method in `admin_edit()`. This is useful if you need
	 * to alter the callbacks.
	 * @var array
	 */
	public $adminSaveOptions = [
		'deep' => true
	];

	/**
	 * change the way cake loads models.
	 * controllers can work out if it's model has been overwritten
	 *
	 * Merge components, helpers, and uses vars from
	 * Controller::$_mergeParent and PluginAppController.
	 *
	 * @return void
	 */
	protected function _mergeControllerVars() {
		$pluginController = $pluginDot = null;
		$mergeParent = is_subclass_of($this, $this->_mergeParent);
		$pluginVars = array();
		$appVars = array();

		if (!empty($this->plugin)) {
			$pluginController = $this->plugin . 'AppController';
			if (!is_subclass_of($this, $pluginController)) {
				$pluginController = null;
			}
			// added the pluginDot to $this->pluginDot for use later
			$this->pluginDot = $pluginDot = $this->plugin . '.';
		}

		// Added: extra checks to see if this controller actually extends a plugin class
		if ($pluginDot === null) {

			$className = get_class($this);
			$pluginName = str_replace($this->name . 'Controller', '', $className);

			if (! empty($pluginName) && CakePlugin::loaded($pluginName)) {

				$this->pluginDot = $pluginDot = $pluginName . '.';
			}
		}

		if ($pluginController) {
			$merge = array('components', 'helpers');
			$this->_mergeVars($merge, $pluginController);
		}

		if ($mergeParent || !empty($pluginController)) {
			$appVars = get_class_vars($this->_mergeParent);
			$merge = array('components', 'helpers');
			$this->_mergeVars($merge, $this->_mergeParent, true);
		}

		if ($this->uses === null) {
			$this->uses = false;
		}
		if ($this->uses === true) {
			$this->uses = array($pluginDot . $this->modelClass);
		}
		if (isset($appVars['uses']) && $appVars['uses'] === $this->uses) {
			array_unshift($this->uses, $pluginDot . $this->modelClass);
		}
		if ($pluginController) {
			$pluginVars = get_class_vars($pluginController);
		}
		if ($this->uses !== false) {
			$this->_mergeUses($pluginVars);
			$this->_mergeUses($appVars);
		} else {
			$this->uses = array();
			$this->modelClass = '';
		}
	}

	/**
	 * override in order to calculate if any overriding components exist
	 *
	 * Loads Model classes based on the uses property
	 * see Controller::loadModel(); for more info.
	 * Loads Components and prepares them for initialization.
	 *
	 * @return mixed true if models found and instance created.
	 * @see Controller::loadModel()
	 * @link http://book.cakephp.org/2.0/en/controllers.html#Controller::constructClasses
	 * @throws MissingModelException
	 */
	public function constructClasses() {
		$this->_mergeControllerVars();
		if ($this->uses) {
			$this->uses = (array)$this->uses;
			list(, $this->modelClass) = pluginSplit(reset($this->uses));
		}

		// Evoluted Amend, get the main model and check the behaviors incase it needs to inject into form
		if (isset($this->request->params['admin']) && $this->request->params['admin']) {

			$this->loadModel($this->pluginDot . $this->modelClass);

			if (! empty($this->{$this->modelClass}->actsAs)) {

				foreach ($this->{$this->modelClass}->actsAs as $behavior => $config) {

					if (isset($config['formInject']) && $config['formInject']) {

						list($plugin, $class) = pluginSplit($config['className']);

						if ($this->hasInjectFile('component', $config)) {

							$this->toInject['components'][$class] = $config['className'];
							$this->components[] = $config['className'];
						}

						if ($this->hasInjectFile('helper', $config)) {

							$this->toInject['helpers'][$class] = $config['className'];
							$this->helpers[] = $config['className'];
						}
					}
				}
			}
		}

		// Evoluted Amend. Loop and check for any plugins in components
		// Check these haven't been overwritten
		if (! empty($this->components)) {

			$this->components = EvClassRegistry::checkOverridingItems('components', $this->components);
		}

		$this->Components->init($this);
		return true;
	}

	/**
	 * model Loader
	 * Works by receiving PluginName.Model (or just Model) and checking for overriding
	 * Model before falling back to load Plugin Model
	 *
	 * @param string $modelClass Name of model class to load
	 * @param int|string $id Initial ID the instanced model class should have
	 * @return bool True if the model was found
	 * @throws MissingModelException if the model class cannot be found.
	 */
	public function loadModel($modelClass = null, $id = null) {
		if ($modelClass === null) {
			$modelClass = $this->modelClass;
		}
		list($plugin, $modelName) = pluginSplit($modelClass);

		// Evoluted Amends - Check for overriding model
		$override = EvClassRegistry::findOverrideModel($modelClass);

		if ($override !== false) {
			$modelClass = $override;
		}

		$this->{$modelName} = EvClassRegistry::init(array(
			'class' => $modelClass, 'alias' => $modelName, 'id' => $id
		));

		return true;
	}

	/**
	 * component loader
	 * Custom wrapper for loading components to check for any overridden ones before loading
	 *
	 * @param string $component Component name to load
	 * @param array $settings Settings for the component.
	 * @return Component A component object, Either the existing loaded component or a new one.
	 * @throws MissingComponentException when the component could not be found
	 */
	public function loadComponent($component, $settings = array()) {
		// check for a plugin / overriding file
		$origComponent = $component;

		if (! empty($settings['className'])) {
			$component = $settings['className'];
		}

		$component = EvClassRegistry::findOverrideComponent($component);
		$settings['className'] = $component;

		// as we have set the className, we need to chop up the original
		// helper call if it was a plugin to set the alias correctly.
		$component = EvClassRegistry::getNewArrayKey($origComponent);

		$loadedComponent = $this->Components->load($component, $settings);
		$loadedComponent->initialize($this);

		return $loadedComponent;
	}

	/**
	 * Ensures only administrators are allowed to access /admin areas
	 *
	 * Override in your controller to customise this behaviour
	 *
	 * @param mixed $user   User attempting to access the controller/action
	 * @return boolean
	 */
	public function isAuthorized($user) {
		if (!isset($user['User'])) {
			$user['User'] = $user;
		}

		if (isset($this->request->params['admin']) && $this->request->params['admin'] == true) {
			// Administrators and Super Users can access admin section, everyone else cannot.
			if ($this->Permissions->hasAdminPermission($user)) {

				return true;

			} else {

				$this->Session->setFlash('You are not authorized to access that location', 'flash_fail');

				$this->redirect(
					array(
						'plugin' => false,
						'admin' => true,
						'controller' => 'ev_core_users',
						'action' => 'login'
					)
				);

				return false;
			}

		} else {

			return true;
		}
	}

	/**
	 * Handles Theme selection and admin access
	 *
	 * @see Controller::beforeFilter()
	 */
	public function beforeFilter() {
		// See if we need to redirect to an alias.
		$this->checkAliasRouting();

		// Set the theme based on prefix routing
		$this->theme = isset($this->params['admin']) ? 'Admin' : Configure::read('theme');

		if (isset($this->params['admin'])) {

			// if accessing the admin section, use the admin login screens
			$this->Auth->loginAction = array(
				'plugin' => false,
				'admin' => true,
				'controller' => 'ev_core_users',
				'action' => 'login'
			);
			$this->Auth->loginRedirect = "/admin";

			// check the current admin action is permitted for this controller
			if (!in_array($this->action, $this->adminActions)) {

				$Model = $this->{$this->modelClass};
				$this->Session->setFlash('Action ' . h($this->action) . ' not supported for ' . InflectorExt::camelToHumanize($Model->displayName), 'flash_fail');
				return $this->redirect('/admin');
			}
		}

		// Load the Navigation menu model ready for building up site
		// navigation.
		$this->loadModel('EvNavigation.Menu');

		return;
	}

	/**
	 * Passes useful variables to the view
	 *
	 * @see Controller::beforeRender()
	 */
	public function beforeRender() {
		// Model name
		$Model = $this->{$this->modelClass};

		if (empty($this->viewVars['primaryModel'])) {
			$this->set('primaryModel', $Model->alias);
		}

		if (empty($this->viewVars['primaryModelPlugin'])) {
			$this->set('primaryModelPlugin', $Model->plugin);
		}

		$this->set('displayName', $Model->displayName);

		// Model primaryKey (check for existance as error pages don't have models)
		// Model displayField
		if (isset($Model->primaryKey)) {

			$this->set('primaryKey', $Model->primaryKey);
			$this->set('displayField', $Model->displayField);

		}

		// Logged in User
		AuthComponent::$sessionKey = 'Auth.User';
		$user = $this->Auth->user();
		$this->set('authUser', $user);

		// Set the primary navigation.
		if ($this->theme == 'Admin' && !empty($this->Menu)) {
			$this->set('primary_nav', $this->Menu->getMenu(1));
		} elseif (!empty($this->Menu)) {
			$this->set('primary_nav', $this->Menu->getMenu(8));
		} else {
			$this->set('primary_nav', array());
		}

		if (isset($this->request->params['admin']) && $this->request->params['admin']) {
			$this->set('toInjectData', $this->toInject);
		}
	}

	/**
	 * Processes multiedit form data.
	 *
	 * @return void
	 */
	protected function _processMultiEdit() {
		$Model = $this->{$this->modelClass};

		if ($this->checkProcessForm('multiedit') === true) {
			foreach ($this->request->data[$Model->alias][$Model->primaryKey] as $key => $value) {
				if (empty($value)) {
					unset($this->request->data[$Model->alias][$Model->primaryKey][$key]);
				}
			}

			if (method_exists($this, '_processMultiEdit' . $this->request->data[$Model->alias]['multiedit'])) {
				$method = '_processMultiEdit' . $this->request->data[$Model->alias]['multiedit'];
				$this->$method(array_keys($this->request->data[$Model->alias][$Model->primaryKey]));
			}

			$this->redirect($this->here);

		}

		return;
	}

	/**
	 * Processes a limit passed for number of results to page by.
	 *
	 * @return void
	 */
	protected function _processLimit() {
		$Model = $this->{$this->modelClass};

		$url = $this->request->params;

		if (! empty($this->request->data[$Model->alias]['results_per_page'])) {

			$url['named']['limit'] = $this->request->data[$Model->alias]['results_per_page'];
			foreach ($url['named'] as $key => $value) {
				$url[$key] = $value;
			}
			unset($url['named']);

			return $this->redirect($url);

		}

		if (! empty($this->request->params['named']['limit'])) {
			$this->request->data[$Model->alias]['results_per_page'] = $this->request->params['named']['limit'];

			if (Configure::read('EvCore.remember_last_admin_results_per_page')) {
				CakeSession::write('ev_core_admin_results_per_page', $this->request->params['named']['limit']);
			}
		} else {
			if (Configure::read('EvCore.remember_last_admin_results_per_page')) {
				$perPage = CakeSession::read('ev_core_admin_results_per_page');

				if ($perPage <= 0) {
					$perPage = 20;
				}
				$this->request->data[$Model->alias]['results_per_page'] = $perPage;
				$this->adminPaginate['limit'] = $perPage;
			} else {
				$this->request->data[$Model->alias]['results_per_page'] = 20;
			}
		}

		return;
	}

	/**
	 * Converts a filter form POST submission into a bookmarkable GET request
	 * Converts a GET request into a conditions array suitable for filtering paginate
	 *
	 * @return array
	 */
	protected function _processFilter($adminFilter = true) {
		if (isset($this->request->data['Filter'])) {

			// convert to get
			$url['action'] = $this->request->action;

			foreach ($this->request->data['Filter'] as $model => $fields) {
				foreach ($fields as $field => $value) {
					if ($value !== "") {
						$url['Filter'][$model][$field] = str_replace('/', '_s_', $value);
					}
				}
			}

			if (Configure::read('EvCore.remember_last_admin_filter')) {
				$filters = CakeSession::read('ev_core_last_admin_filter');

				if (isset($this->request->data['Filter'][$this->modelClass])) {
					$filters[$this->modelClass] = $this->request->data['Filter'][$this->modelClass];

					CakeSession::write('ev_core_last_admin_filter', $filters);
				}
			}

			if (Configure::read('EvCore.remember_last_admin_sort_field')) {

				$lastSort = CakeSession::read('ev_core_last_admin_field');

				if (isset($lastSort[$this->modelClass])) {
					$url['sort'] = $lastSort[$this->modelClass]['sort'];
					$url['direction'] = $lastSort[$this->modelClass]['direction'];
				}
			}

			if (isset($url['Filter'])) {

				$pageSession = CakeSession::read('ev_core_admin_last_page');

				if (isset($pageSession[$model])) {
					unset($pageSession[$model]);
					CakeSession::write('ev_core_admin_last_page', $pageSession);
				}

				// redirect to bookmarkable get
				$this->redirect($url);
			}
		}

		// build filter conditions
		$conditions = array();

		// only these fields will be concidered as part of the filter
		if ($adminFilter) {
			$filterFields = $this->_adminFilterFields();
		} else {
			$filterFields = $this->_frontFilterFields();
		}

		// Set any defined default filters.
		foreach ($filterFields as $field => $filter) {
			if (isset($filter['default']) && !empty($filter['default'])) {
				$conditions[$field] = $filter['default'];
			}
		}

		if (isset($this->request->params['named']['Filter'])) {
			foreach ($this->request->params['named']['Filter'] as $model => $fields) {
				foreach ($fields as $field => $value) {

					if (is_array($value)) {
						$value = $value['year'] . "-" . $value['month'] . "-" . $value['day'];
					} else {
						$value = str_replace('_s_', '/', $value);
					}

					if (isset($filterFields[$model . "." . $field])) {
						$key = $filterFields[$model . "." . $field]['compare'];
						$isDate = $filterFields[$model . "." . $field]['type'] == 'date';
						$isNull = (isset($filterFields[$model . "." . $field]['is_null']) && $filterFields[$model . "." . $field]['isnull'] = true) ? true : false;

						//If there are multiple fields to compare join them together with ORs
						if (count($key) > 1) {
							$orArray = array();
							foreach ($key as $fieldi => $condition) {
								$orArray[$fieldi] = sprintf($condition, $value);
							}
							$conditions[]['OR'] = $orArray;
						} else {
							if ($isDate) {
								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column . " >="] = CakeTime::toServer($value . ' 00:00:00', Configure::read('Config.timezone'), 'Y-m-d H:i:s');
								$conditions[$column . " <="] = CakeTime::toServer($value . ' 24:00:00', Configure::read('Config.timezone'), 'Y-m-d H:i:s');
							} elseif ($isNull) {
								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								if ($value == 1) {
									$conditions[] = $column . " IS NULL";
								} else {
									$conditions[] = $column . " IS NOT NULL";
								}

							} elseif ($filterFields[$model . "." . $field]['type'] == 'daterange_from') {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column] = CakeTime::toServer($value . ' 00:00:00', Configure::read('Config.timezone'), 'Y-m-d H:i:s');
							} elseif ($filterFields[$model . "." . $field]['type'] == 'daterange_to') {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column] = CakeTime::toServer($value . ' 24:00:00', Configure::read('Config.timezone'), 'Y-m-d H:i:s');
							} else {

								$conditions[key($key)] = sprintf(array_pop($key), $value);
							}
						}

						$this->request->data['Filter'][$model][$field] = $value;
					}
				}
			}
		}

		// If the model has an is_removed column then we want to ensure removed
		// rows are filtered out of the admin_index listings
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if ($Model->hasField('is_removed')) {
			$conditions[$modelAlias . '.is_removed'] = false;
		}

		return $conditions;
	}

	/**
	 * Does a recursive str_replace on the $compare array, replacing instances of %s with $value
	 * in both the keys and values
	 *
	 * @param array $compare
	 * @param string $value
	 * @return array
	 * @deprecated
	 */
	protected function _buildFilterCompare($compare, $value) {
		$key = key($compare);

		if (in_array(strtolower($key), array('or', 'and'))) {

			$compare[$key] = $this->_buildFilterCompare($compare[$key], $value);
		} else {

			$result = array();

			foreach ($compare as $compareKey => $compareValue) {

				$compareKey = str_replace('%s', $value, $compareKey);
				$compareValue = str_replace('%s', $value, $compareValue);

				$result[$compareKey] = $compareValue;
			}

			$compare = $result;
		}

		return $compare;
	}

	/**
	 * Defines the toolbar buttons displayed in admin_index
	 *
	 * @return array
	 */
	protected function _adminIndexToolbar() {
		$toolbar = array(
			'Add New' => array(
				'url' => array('action' => 'add')
			)
		);

		if ($this->reordering) {
			$showReordering = false;
			$modelColumns = $this->{$this->modelClass}->getColumnTypes();

			if (
				array_key_exists('lft', $modelColumns) ||
				array_key_exists('orderno', $modelColumns) ||
				array_key_exists('sequence', $modelColumns)
			) {

				$showReordering = true;
			}

			if ($showReordering) {
				$toolbar['Reorder'] = array(
					'url' => array('action' => 'reorder'),
					'class' => 'toolbar__btn--reorder'
				);
			}
		}

		return $toolbar;
	}

	/**
	 * Pagination settings for admin_index
	 *
	 * Override in your own controller to customise
	 *
	 * @return array
	 */
	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions
		);

		// Include defaults.
		$paginate = array_merge_recursive($this->adminPaginate, $paginate);

		$modelColumns = $Model->getColumnTypes();

		if (array_key_exists('orderno', $modelColumns)) {

			$paginate['order'] = $Model->alias . '.orderno ASC';

		} elseif (array_key_exists('sequence', $modelColumns)) {

			$paginate['order'] = $Model->alias . '.sequence ASC';

		} elseif (array_key_exists('lft', $modelColumns)) {

			$paginate['order'] = $Model->alias . '.lft ASC';

		}

		// Group the results by the primary key to prevent duplicate results.
		$paginate['group'] = $Model->alias . '.' . $Model->primaryKey;

		return $paginate;
	}

	/**
	 * Defines which actions are available to each row of the admin_index table results
	 *
	 * Override in your controller to customise
	 * Customise the handlers for these actions in /view/Admin/Elements/index_results.ctp
	 *
	 * @return array
	 */
	protected function _adminIndexActions() {
		$actions = array(
			'Edit' => array(
				'cell' => array(
					'class' => 'action'
				),
				'link' => array(
					'url' => array(
						'action' => 'edit'
					),
					'text' => '<i class="s s--edit"></i>',
					'options' => array(
						'escape' => false
					)
				)
			),
			'Delete' => array(
				'cell' => array(
					'class' => 'action'
				),
				'link' => array(
					'url' => array(
						'action' => 'delete'
					),
					'text' => '<i class="s s--delete"></i>',
					'options' => array(
						'escape' => false
					)
				),
				'helper' => 'deleteAction',
				'protected' => '<i class="s s--delete-disabled" title="Disabled"></i>',
			)
		);

		if (method_exists($this, 'view')) {

			$actions['View'] = array(
				'cell' => array(
					'class' => 'action'
				),
				'link' => array(
					'url' => array(
						'admin' => false,
						'action' => 'view'
					),
					'text' => '<i class="s s--view"></i>',
					'options' => array(
						'escape' => false,
						'target' => '_blank'
					)
				)
			);
		}

		return $actions;
	}

	/**
	 * Define an array of actions that can be performed on multiple items on the
	 * admin listing page.
	 *
	 * @return array
	 */
	protected function _adminMultiEditActions() {
		return array();
	}

	/**
	 * Defines the fields displayed in a filter form.
	 *
	 * Defaults to id, display name, is_active, created and modified fields.
	 *
	 * Override in your controller to customise.
	 *
	 * Format:
	 *
	 * 'id' => array(
	 *         'label' => label text,
	 *         'type' => data type of control
	 *         'compare' => SQL query. e.g. Model.field='%s' (where %s is replace by the value of the field)
	 *         'default' => value you want the data filtering by by default
	 * )
	 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = array(
			"$modelAlias.id" => array(
				'label' => 'ID',
				'type' => 'integer',
				'compare' => array("$modelAlias.id" => "%s")
			)
		);

		if ($Model->displayField) {
			$fields[$modelAlias . '.' . $Model->displayField] = array(
				'label' => InflectorExt::camelToHumanize($Model->displayField),
				'type' => 'string',
				// We use double '%' in the comparison as we don't want them
				// treating as literal %.
				'compare' => array($modelAlias . '.' . $Model->displayField . ' LIKE' => "%%%s%%")
			);
		}

		if ($Model->hasField('is_active')) {
			$fields[$modelAlias . '.is_active'] = array(
				'label' => 'Active',
				'type' => 'boolean',
				'compare' => array("$modelAlias.is_active" => "%s")
			);
		}

		if ($Model->hasField('created')) {
			$fields[$modelAlias . '.created'] = array(
				'label' => 'Created',
				'type' => 'date',
				'compare' => array("$modelAlias.created" => "%s")
			);
		}

		if ($Model->hasField('modified')) {
			$fields[$modelAlias . '.modified'] = array(
				'label' => 'Modified',
				'type' => 'date',
				'compare' => array("$modelAlias.modified" => "%s")
			);
		}

		return $fields;
	}

	/**
	 * Stub function
	 * Used to populate form drop down selects
	 *
	 * Override in your controller to customise
	 */
	protected function _adminPopulateLookups() {
	}

	/**
	 * Stub function
	 * Used to populate filter form drop down selects
	 *
	 * Override in your controller to customise
	 */
	protected function _adminPopulateFilterLookups() {
		$this->_adminPopulateLookups();
	}

	/**
	 * Defines the columns displayed in the admin_index results table
	 * using a columns whitelist.
	 *
	 * Override in your controller to customise
	 */
	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = array();

		$columnsWhitelist = $this->_adminIndexColumnsWhitelist();

		foreach ($Model->schema() as $field => $attr) {

			if (in_array("$modelAlias.$field", $columnsWhitelist)) {

				$attr['label'] = InflectorExt::humanize($field);
				$attr['label'] = str_replace('Is ', '', $attr['label']);
				$columns[$modelAlias . "." . $field] = $attr;
			}
		}

		return $columns;
	}

	/**
	 * Defines a whitelist of columns to be displayed on the admin index.
	 * @return array whitelisted columns
	 */
	protected function _adminIndexColumnsWhitelist() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = array();

		$columns[] = "$modelAlias.id";

		if ($Model->displayField) {
			$columns[] = "$modelAlias.{$Model->displayField}";
		}

		if ($Model->hasField('is_active')) {
			$columns[] = "$modelAlias.is_active";
		}
		if ($Model->hasField('created')) {
			$columns[] = "$modelAlias.created";
		}
		if ($Model->hasField('modified')) {
			$columns[] = "$modelAlias.modified";
		}

		return $columns;
	}

/**
 * Discards values from the formFields that now appear in the tabs
 *
 * @param array $formFields The array of form fields and tabs
 * @return array Containing the form fields with the tabbed items removed
 */
	protected function _discardTabbedFormFields($formFields = []) {
		if (! empty($formFields['tabs'])) {
			// loops around each tab instance are removes items
			// found in the parent formFields array
			foreach ($formFields['tabs'] as $tab) {
				$tabKeys = array_keys($tab);
				$formFields = array_diff_key($formFields, array_combine($tabKeys, $tabKeys));
			}
		}
		return $formFields;
	}

/**
 * Defines the fields displayed in an admin_form for this model
 *
 * Defaults to all fields in the db table
 *
 * Override in your controller to customise
 */
	protected function _adminFormFields() {
		$Model = $this->{$this->modelClass};

		$data = $Model->schema();

		$fields = array();

		foreach ($data as $key => $value) {
			$fields["{$Model->alias}.$key"] = $value;
		}

		// If the model is routable we want to include the route fields
		if ($Model->hasBehaviour('Routable.Routable')) {

			if (isset($fields["{$Model->alias}.name"])) {

				$insertRoutesAfter = "{$Model->alias}.name";

			} elseif (isset($fields["{$Model->alias}.title"])) {

				$insertRoutesAfter = "{$Model->alias}.title";

			}

			$routeFields = array(
				'Route.id' => array(
					'type' => 'hidden'
				),
				'Route.alias' => array(
					'type' => 'route-alias',
					'length' => 128,
					'url_prefix' => $Model->getUrlPrefix($Model)
				)
			);

			if (isset($insertRoutesAfter)) {

				$fields = ArrayUtil::addAfter($fields, $insertRoutesAfter, $routeFields);

			} else {

				$fields += $routeFields;
			}
		}

		// only show is_protected if main admin
		if ($Model->hasField('is_protected')) {
			if (!$this->Permissions->hasAdminPermission($this->Auth->user())) {
				// Remove to all but superusers.
				unset($fields[$Model->alias . '.is_protected']);
			}
		}

		// is_removed columns shouldn't be exposed to the user
		unset($fields[$Model->alias . '.is_removed']);

		unset($fields[$Model->alias . '.lft']);
		unset($fields[$Model->alias . '.rght']);
		unset($fields[$Model->alias . '.sequence']);

		return $this->_discardTabbedFormFields($fields);
	}

	/**
	 * Defines the meta data fields in an admin_form for this model
	 * when it has the Meta behaviour.
	 */
	protected function _adminFormMetaFields() {
		$Model = $this->{$this->modelClass};

		$fields = array();

		if ($Model->hasBehaviour('MetaData.Meta')) {

			$fields['MetaData.id'] = array(
				'type' => 'hidden'
			);

			$fields['MetaData.title'] = array(
				'type' => 'string',
				'length' => 155
			);

			$fields['MetaData.description'] = array(
				'type' => 'string',
				'length' => 155
			);
		}

		return $fields;
	}

	/**
	 * Defines the menu fields in an admin_form for this model
	 * when it has the Navigatable behaviour.
	 *
	 * @param  integer $id ID of the current menu to exclude
	 * @return array
	 */
	protected function _adminFormMenuFields($id) {
		$Model = $this->{$this->modelClass};

		$fields = array();

		if ($Model->hasBehaviour('EvNavigation.Navigatable')) {

			$menu = $this->Menu->findTreeBranches($Model->parentMenu, $id);

			$fields['Menu.is_active'] = array(
				'type' => 'boolean',
				'label' => 'Add to menu'
			);

			$fields['Menu.id'] = array(
				'type' => 'hidden'
			);

			$fields['Menu.name'] = array(
				'type' => 'string',
				'length' => 45
			);

			$fields['Menu.parent_id'] = array(
				'type' => 'integer',
				'options' => $menu
			);
		}

		return $fields;
	}

	/**
	 * Defines the buttons in the toolbar displayed on an admin_form
	 *
	 * Override in your controller to customise
	 *
	 * @param integer $id
	 * @return array
	 */
	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = array(
			'Add New' => array(
				'url' => array('action' => 'add')
			)
		);

		$data = null;

		if ($id) {
			$data = $this->request->data;
		}

		if (!isset($data[$modelAlias]['is_protected']) || !$data[$modelAlias]['is_protected']) {

			// Check to make sure we're editing an item before displaying the copy button.
			if ($id && $Model->hasBehaviour('EvCore.CustomCopyable')) {

				$actions['Copy'] = array(
					'url' => array('action' => 'copy', $id),
					'class' => 'toolbar__btn--copy'
				);

			}
			// Only show the delete button for existing records.
			if ($id > 0) {
				$actions['Delete'] = array(
					'url' => array('action' => 'delete', $id),
					'class' => 'toolbar__btn--delete'
				);
			}

		}

		return $actions;
	}

	/**
	 * Defines the fields displayed in an admin_form for this model
	 *
	 * Defaults to all fields in the db table
	 *
	 * Override in your controller to customise
	 *
	 * @return array
	 */
	protected function _adminViewFields() {
		return $this->_adminFormFields();
	}

	/**
	 * Defines the buttons in the toolbar displayed on an admin_form
	 *
	 * Override in your controller to customise
	 *
	 * @param integer $id
	 * @return array
	 */
	protected function _adminViewToolbar($id = null) {
		return $this->_adminFormToolbar($id);
	}

	/**
	 * Defines the toolbar buttons displayed in admin_reorder
	 *
	 * @return array
	 */
	protected function _adminReorderToolbar() {
		$toolbar = array(
			'Listing' => array(
				'url' => array(
					'action' => 'index'
				),
				'class' => 'toolbar__btn--listing'
			),
			'Add New' => array(
				'url' => array(
					'action' => 'add'
				)
			)
		);

		return $toolbar;
	}

	/**
	 * Manages a default admin index view of this model
	 * Includes a filter form and paginated results
	 *
	 * Override in your controller to customise
	 *
	 * @return void
	 */
	public function admin_index() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (Configure::read('EvCore.remember_last_admin_sort_field')) {
			$lastSort = CakeSession::read('ev_core_last_admin_field');

			if (isset($this->request->params['named']['sort'])) {
				$lastSort[$this->modelClass] = array(
					'sort' => $this->request->params['named']['sort'],
					'direction' => $this->request->params['named']['direction']
				);
				CakeSession::write('ev_core_last_admin_field', $lastSort);
			} else {

				if (isset($lastSort[$this->modelClass])) {

					$url['action'] = $this->request->action;
					$url['sort'] = $lastSort[$this->modelClass]['sort'];
					$url['direction'] = $lastSort[$this->modelClass]['direction'];

					return $this->redirect($url);
				}
			}
		}

		if (Configure::read('EvCore.remember_last_admin_filter')) {
			$lastFilter = CakeSession::read('ev_core_last_admin_filter');
			if (! isset($this->request->params['named']['Filter'][$this->modelClass])) {

				if (isset($lastFilter[$this->modelClass])) {
					$this->request->data['Filter'][$this->modelClass] = $lastFilter[$this->modelClass];
				}

				$this->_processFilter(true);
			}
		}

		if (Configure::read('EvCore.remember_last_admin_page_number')) {
			$lastPage = CakeSession::read('ev_core_admin_last_page');

			if (isset($this->request->params['named']['page'])) {
				$lastPage[$this->modelClass] = $this->request->params['named']['page'];
				CakeSession::write('ev_core_admin_last_page', $lastPage);
			} elseif (isset($lastPage[$this->modelClass]) && $lastPage[$this->modelClass] > 1) {
				$url = $this->referer(null, true);
				$referer = Router::parse($url);

				if (isset($referer['controller']) && $referer['controller'] == $this->request->params['controller']
					&& isset($referer['named']['page'])) {
					unset($lastPage[$this->modelClass]);
					CakeSession::write('ev_core_admin_last_page', $lastPage);
				} else {
					$this->redirect($this->here . '/index/page:' . $lastPage[$this->modelClass]);
				}
			}
		}

		$this->_processMultiEdit();
		$this->_processLimit();
		$this->_adminPopulateFilterLookups();
		$this->set('filter', $this->_adminFilterFields());
		$this->set('columns', $this->_adminIndexColumns());
		$this->set('toolbar', $this->_adminIndexToolbar());
		$this->set('actions', $this->_adminIndexActions());
		$this->set('multiEditActions', $this->_adminMultiEditActions());

		$this->Paginator->settings = $this->_adminIndexPaginate();
		$this->set('data', $this->Paginator->paginate());

		$this->set('title_for_layout', InflectorExt::camelToPluralize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_index';

		return;
	}

	/**
	 * Manages an admin edit form for this model
	 *
	 * @param integer $id ID of row to edit
	 * @return void
	 */
	public function admin_edit($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (
			! empty($this->request->data) &&
			(
				$this->request->is('post') ||
				$this->request->is('put')
			)
		) {

			// If its in use, reset the stored page session
			if (Configure::read('EvCore.remember_last_admin_page_number')) {
				$lastPage = CakeSession::read('ev_core_admin_last_page');
				unset($lastPage[$modelAlias]);
				CakeSession::write('ev_core_admin_last_page', $lastPage);
			}

			$this->request->data = $this->_formatMultiImages($this->request->data);
			$this->request->data = $this->_processImages($this->request->data);

			if ($Model->saveAssociated($this->request->data, $this->adminSaveOptions)) {

				$action = ($id === null) ? 'created' : 'updated';
				$title = empty($this->data[$modelAlias][$Model->displayField]) ? InflectorExt::camelToHumanize($Model->displayName) : '&lsquo;' . InflectorExt::camelToHumanize($this->data[$modelAlias][$Model->displayField]) . '&rsquo;';

				// check if there were any stored errors from afterSave callbacks
				// if not proceed with success message
				if (empty($Model->afterSaveValidationErrors)) {
					$this->Session->setFlash(array(
						'title' => InflectorExt::camelToHumanize($Model->displayName) . " $action",
						'description' => "$title has been successfully $action!"
					), 'flash_success');

					if (isset($this->request->data['return']) && $this->request->data['return'] == 1) {

						$this->Session->write("Admin.$modelAlias.return", true);

						$return = array('action' => 'edit', $Model->id);

					} elseif (isset($this->adminRedirect) && ! empty($this->adminRedirect)) {

						$return = $this->adminRedirect;

					} else {

						$this->Session->delete("Admin.$modelAlias.return");

						$redirect = $this->Session->read("Admin.$modelAlias.filter");
						if (strpos($redirect, 'Filter') !== false) {
							$return = $redirect;
						} else {
							$return = array('action' => 'index');
						}
					}

					return $this->redirect($return);
				}

				// if we get here, there were some afterSave errors
				// show warning messsage
				$this->Session->setFlash(array(
					'title' => InflectorExt::camelToHumanize($Model->displayName) . " $action",
					'description' => "$title has been successfully $action! There was however some issues saving extra data.",
					'list' => $Model->afterSaveValidationErrors
				), 'flash_warning');

			} else {
				$this->Session->setFlash(array(
					'title' => 'Save failed',
					'description' => 'Failed to save ' . InflectorExt::camelToHumanize($Model->displayName) . ', see below for details',
					'list' => $Model->validationErrors
				), 'flash_fail');

			}

		} elseif ($id !== null) {

			$this->request->data = $Model->readForEdit($id);

			if (!$this->request->data) {

				throw new NotFoundException();

			}
			$this->Session->write("Admin.$modelAlias.filter", $this->referer());
		}

		$actionLabel = ($id === null) ? 'Add' : 'Edit';

		$this->set(
			array(
				'actionLabel' => $actionLabel,
				'title_for_layout' => $actionLabel . ' ' . InflectorExt::camelToHumanize($Model->displayName),
				'fields' => $this->_adminFormFields(),
				'toolbar' => $this->_adminFormToolbar($id),
				'imageSlots' => $Model->imageSlots,
				'documentSlots' => $Model->documentSlots,
				'menuFields' => $this->_adminFormMenuFields((isset($this->request->data['Menu']['id']) ? $this->request->data['Menu']['id'] : null)),
				'metaData' => $this->_adminFormMetaFields(),
				'actualModel' => get_class($Model),
				'plugins' => App::objects('plugin'),
				'activeItem' => array(
					'model' => EvClassRegistry::getNameFromModel($Model),
					'id' => $id
				)
			)
		);
		$this->_adminPopulateLookups();

		$this->set('return', $this->Session->read("Admin.$modelAlias.return"));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_form';

		// check for any injection components to run
		if (! empty($this->toInject['components'])) {

			foreach ($this->toInject['components'] as $class => $component) {

				if (! is_object($this->{$class})) {
					$this->{$class} = $this->loadComponent($component);
					$this->{$class}->initialize($this);
				}

				$this->{$class}->injectAdminForm(
					$this->request->data,
					$Model,
					$id
				);
			}
		}
	}

	/**
	 * View data for this model
	 *
	 * @param integer $id ID of row to edit
	 * @return void
	 */
	public function admin_view($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$this->request->data = $Model->readForAdminView($id);

		if (!$this->request->data) {

			throw new NotFoundException();
		}

		$actionLabel = 'View';

		$this->set('title_for_layout', "$actionLabel " . InflectorExt::camelToHumanize($Model->displayName));
		$this->set('fields', $this->_adminViewFields());
		$this->set('toolbar', $this->_adminViewToolbar($id));
		$this->set('imageSlots', $Model->imageSlots);
		$this->set('documentSlots', $Model->documentSlots);
		$this->_adminPopulateLookups();

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_view';

		return;
	}

	/**
	 * Manages an admin delete for this model
	 *
	 * Override in your controller to customise
	 *
	 * @todo: check $Model->cascade - if true we don't need to check for
	 * dependants as they will be automatically deleted when the parent is
	 * deleted
	 *
	 * @param integer $id ID of row to delete
	 * @return void
	 */
	public function admin_delete($id) {
		// Fallback incase someone comes to the delete url without an id
		if (!$id) {
			return $this->redirect($this->referer());
		}
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {

			$Model->delete($id);

			// If its in use, reset the stored page session
			if (Configure::read('EvCore.remember_last_admin_page_number')) {
				$lastPage = CakeSession::read('ev_core_admin_last_page');
				unset($lastPage[$modelAlias]);
				CakeSession::write('ev_core_admin_last_page', $lastPage);
			}

			$this->Session->setFlash(array(
				'title' => InflectorExt::camelToHumanize($Model->displayName) . ' deleted',
				'description' => InflectorExt::camelToHumanize($Model->displayName) . ' has been successfully deleted!'
			), 'flash_success');
			if (isset($this->adminRedirect) && ! empty($this->adminRedirect)) {

				$return = $this->adminRedirect;

			} else {

				$return = array('action' => 'index');

			}
			return $this->redirect($return);

		} else {

			$this->Session->write('delete_referrer', $this->request->referrer);
		}

		$this->request->data = $Model->findById($id);

		$this->set('dependants', $Model->findDependents($id));

		$this->set('title_for_layout', 'Delete ' . InflectorExt::camelToHumanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_delete';

		return;
	}

	/**
	 * Copies an item
	 *
	 * @param integer $id id of item to copy
	 * @return void
	 */
	public function admin_copy($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {

			if ($Model->copy($id)) {

				$newId = $Model->getInsertID();

				$this->Session->setFlash(array(
					'title' => InflectorExt::camelToHumanize($Model->displayName) . " copied",
					'description' => 'Item has been successfully copied!'
				), 'flash_success');

				return $this->redirect(array('action' => 'edit', $newId));

			} else {

				$this->Session->setFlash(array(
					'title' => 'Copy failed',
					'description' => 'Failed to copy ' . InflectorExt::camelToHumanize($Model->displayName)
				), 'flash_fail');

				return $this->redirect(array('action' => 'edit', $id));

			}

		}

		$this->request->data = $Model->findById($id);

		if (isset($this->request->data[$modelAlias]['is_protected']) && $this->request->data[$modelAlias]['is_protected']) {
			if (isset($this->adminRedirect) && ! empty($this->adminRedirect)) {

				$return = $this->adminRedirect;

			} else {

				$return = array('action' => 'index');

			}
			return $this->redirect($return);

		}

		$this->set('title_for_layout', 'Copy ' . InflectorExt::camelToHumanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_copy';

		return;
	}

	/**
	 * Toggles a field
	 *
	 * @param string $field Field to toggle
	 * @param integer $id id of row to toggle
	 * @return void
	 */
	public function admin_toggle($field, $id) {
		$Model = $this->{$this->modelClass};

		// Cleanse the data before updating the database
		$id = intval($id);

		$Model->toggleField($field, $id);

		return $this->redirect($this->referer());
	}

	/**
	 * Admin reordering by Ollie ┗(＾0＾)┓
	 * Put public max_depth in the controller to control how many levels it goes down
	 * 0 = Unlimited
	 *
	 * @return void
	 */
	public function admin_reorder() {
		$Model = $this->{$this->modelClass};

		// Determine the ordering column.
		if ($Model->hasField('orderno')) {
			$orderKey = 'orderno';
		} elseif ($Model->hasField('sequence')) {
			$orderKey = 'sequence';
		} elseif ($Model->hasField('lft')) {
			$orderKey = 'lft';
		} else {
			// Fail I need an ordering column
			$this->Flash->fail([
				'title' => 'Reordering Failed',
				'description' => 'Unable to find an ordering column on the table'
			]);
			return $this->redirect('/admin');
		}

		// See if it's a multi-level dealy
		$hasMultiLevel = false;
		if ($Model->hasField('parent_id')) {
			$hasMultiLevel = true;
		} else {
			$this->max_depth = 1;
		}

		// Save data if post
		if ($this->request->data) {
			if ($Model->hasBehaviour('Routable.Routable')) {
				$Model->Behaviors->unload('Routable.Routable');
			}

			unset($this->request->data['log']);
			$Model->validate = array();

			foreach ($this->request->data['reorder'] as &$item) {
				if (empty($item['parent_id'])) {
					$item['parent_id'] = null;
				}
			}

			$Model->removeBehaviour('Tree');
			if ($Model->saveMany($this->request->data['reorder'], ['callbacks' => false, 'validate' => false])) {
				$this->Flash->success([
					'title' => 'Order Saved',
					'description' => 'Reordering has been saved successfully!'
				]);
			} else {
				$this->Flash->fail([
					'title' => 'Reordering Failed',
					'description' => 'There was an error saving the data'
				]);
			}
			$Model->addBehaviour('Tree');
			if (isset($this->adminRedirect) && ! empty($this->adminRedirect)) {

				$return = $this->adminRedirect;

			} else {

				$return = array('action' => 'index');

			}
			return $this->redirect($return);
		}

		//Use only ID and the display field
		$columns = [
			$Model->alias . '.id' => '',
			$Model->alias . '.' . $Model->displayField => ''
		];
		$this->set('columns', $columns);

		if ($Model->hasField('is_reorderable')) {
			$params = [
				'conditions' => [
					$Model->alias . '.is_reorderable' => 1
				]
			];
		} else {
			$params = [];
		}

		// Get the data ready for re-ordering.
		$data = $Model->readForReorder($params);

		//Lets javascript know whats going on too
		$this->set('order_key', $orderKey);
		$this->set('max_depth', $this->max_depth);
		$this->set('toolbar', $this->_adminReorderToolbar());
		$this->set('data', $data);

		$this->view = '../Scaffolds/admin_reorder';

		return;
	}

	/**
	 * Recursive Finding of children - Takes array by reference
	 * @param data array by reference
	 * @param model name string
	 * @param order key string
	 * @return void
	 * @deprecated v2.3.3.0 Moved to the model
	 */
	protected function _getChildren(&$data, $Model, $orderKey) {
		foreach ($data as &$d) {

			$children = $this->$Model->find('all', array(
				'conditions' => array(
					'parent_id' => $d[$Model]['id']
				),
				'order' => $orderKey
			));
			$this->_getChildren($children, $Model, $orderKey);

			$d[$Model]['children'] = $children;
		}

		return;
	}

	/**
	 * Toggles a field via an AJAX call
	 *
	 * @param string $field Field to toggle
	 * @param integer $id id of row to toggle
	 * @return boolean
	 */
	public function ajax_toggle($field, $id) {
		$user = $this->Auth->user();

		if ($this->Permissions->hasAdminPermission($user)) {
			$Model = $this->{$this->modelClass};

			// Because we don't have CSRF token for AJAX we want to check the request has come from our domain
			if (strpos($this->request->referer(), Configure::read('App.fullBaseUrl') . '/admin') !== 0) {
				throw new NotFoundException();
			}

			// Cleanse the data before updating the database
			$id = intval($id);

			$this->render('/Themed/Admin/Scaffolds/ajax_toggle', 'ajax');

			return ($id) ? $Model->toggleField($field, $id) : false;
		} else {
			throw new NotFoundException();
		}
	}

	/**
	 * Parse Multi Image uploads to match required Attachments format
	 * @params array data - data from form submission
	 * @return array data - data as after processing of multi-images
	 */
	protected function _formatMultiImages($data) {
		$Model = $this->{$this->modelClass};

		$imageSlots = $Model->imageSlots;

		if (!is_array($imageSlots)) {
			$imageSlots = array(
				'main' => array(
					'slots' => $imageSlots
				)
			);
		}

		// Add any template images to the image slots array
		if (!empty($this->data[$this->modelClass]['template_id'])) {
			$this->loadModel('EvTemplates.ImageBlock');
			$templateSlots = $this->ImageBlock->find('list', [
				'conditions' => [
					'template_id' => $this->data[$this->modelClass]['template_id']
				],
				'fields' => [
					'ImageBlock.name',
					'ImageBlock.slots'
				],
				'callbacks' => false
			]);

			foreach ($templateSlots as $templateSlotName => $templateSlotCount) {
				$imageSlots[$templateSlotName] = [
					'slots' => (int)$templateSlotCount
				];
			}
		}

		foreach ($imageSlots as $imageSlotType => $imageSlot) {
			$imageSlotIndex = 'Image';
			$maxImages = $imageSlot['slots'];

			if ($imageSlotType != 'main') {
				$imageSlotIndex = ucfirst($imageSlotType) . $imageSlotIndex;
			}

			if (!empty($data[$imageSlotIndex]['multifile'])) {
				foreach ($data[$imageSlotIndex]['multifile'] as $uploadedImage) {
					if ($uploadedImage['error'] == '0') {
						$imageArray = array(
							'model' => $Model->alias,
							'attachment_type' => $imageSlotIndex,
							'filename' => $uploadedImage
						);

						$data[$imageSlotIndex][] = $imageArray;
					}
				}

				unset($data[$imageSlotIndex]['multifile']);

				if ($maxImages > 0) {
					$data[$imageSlotIndex] = array_slice($data[$imageSlotIndex], 0, $maxImages);
				}
			}
		}

		return $data;
	}

	/**
	 * Remove any image slots that are not been used in an admin form
	 *
	 * @param $this->request->data
	 * @return $this->request->data with surplus image slots removed
	 */
	protected function _processImages($data) {
		$rtnArray = $data;

		foreach ($data as $key => $value) {
			if ($key == 'Image' && isset($data['Image'])) {
				foreach ($data['Image'] as $keyi => $image) {
					if (empty($image['id']) && (empty($image['Image']['filename']['name']) && empty($image['filename']['name']))) {
						unset($rtnArray['Image'][$keyi]);
					}
				}
			} else {
				if (is_array($value)) {
					$rtnArray[$key] = $this->_processImages($value);
				} else {
					$rtnArray[$key] = $value;
				}
			}
		}

		return $rtnArray;
	}

	/**
	 * Returns true if a form submission contains a defined Form.process
	 *
	 * Used to identify which form to process on pages with multiple forms (for
	 * example a product page with an add to basket and review forms).
	 *
	 * @param string $process
	 * @return boolean
	 */
	public function checkProcessForm($process) {
		return ! empty($this->request->data['Form']['process']) && $this->request->data['Form']['process'] === $process;
	}

	/**
	 * add page data to a page
	 * Used for none page screens that need page titles / meta functionality
	 * such as module listing pages
	 *
	 * @param   string|int      page constant to load or page ID
	 * @return  bool|array 		Bool false or data array on success
	 */
	public function assignPage($page, $tplVar = 'data') {
		$this->loadModel('EvCore.Page');

		if (! is_int($page) && defined(get_class($this->Page) . '::' . $page)) {
			$page = constant(get_class($this->Page) . '::' . $page);
		} elseif (! is_int($page)) {
			return false;
		}

		$data = $this->Page->readForView($page);

		if (! $data) {
			return false;
		}

		$this->set($tplVar, $data);
		$this->Meta->set($data, 'Page');

		return $data;
	}

	/**
	 * check to see if the given behaviors has an associated component with an inject method
	 *
	 * @param   string  Type of file to check for injection
	 * @param   array   Behavior Config Array
	 * @return  bool
	 */
	public function hasInjectFile($type, $behaviorConfig) {
		$loadPaths = array(
			'component' => 'Controller/Component',
			'helper' => 'View/Helper'
		);

		if (isset($loadPaths[$type])) {
			$loadPath = $loadPaths[$type];
		} else {
			return false;
		}

		if (empty($behaviorConfig['className'])) {
			return false;
		}

		// split the class name for plugin and set dot after plugin so we can include in paths
		list($plugin, $class) = pluginSplit($behaviorConfig['className']);
		$path = App::path($loadPath, $plugin);
		$type = ucfirst($type);

		if (isset($path['0']) && file_exists($path['0'] . $class . $type . '.php')) {

			require_once ($path['0'] . $class . $type . '.php');
			return method_exists($class . $type, 'injectAdminForm');
		}

		return false;
	}

	/**
	 * inject component / helper for admin_edit
	 *
	 * @param 	string 	$type 	The type of file, either 'components', 'helpers'
	 * @param 	string 	$file 	The file to load in Plugin.File syntax
	 * @return 	bool
	 */
	public function toInject($type, $file) {
		if (strpos($file, '.') !== false) {
			list($plugin, $key) = pluginSplit($file);
		} else {
			$key = $file;
		}

		$this->toInject[$type][$key] = $file;
		$this->{$type}[] = $file;

		return true;
	}

	/**
	 * Instantiates the correct view class, hands it its data, and uses it to render the view output.
	 *
	 * @param string $view View to use for rendering
	 * @param string $layout Layout to use
	 * @return CakeResponse A response object containing the rendered view.
	 * @triggers Controller.beforeRender $this
	 * @link http://book.cakephp.org/2.0/en/controllers.html#Controller::render
	 */
	public function render($view = null, $layout = null) {
		if ($layout == 'ajax') {
			$this->set('isAjaxRequest', true);
		}

		return parent::render($view, $layout);
	}

	/**
	 * checks to see if an alias exists for the loaded route if it has been
	 * accessed directly.
	 *
	 * @return void
	 */
	public function checkAliasRouting() {
		if (Configure::read('EvCore.force_route_alias_redirects')) {
			// Check the current route to check if its a default controller route.
			$route = Router::currentRoute();
			if (! empty($route->options['defaultRoute'])) {

				// The route is an 'actual' route. Check to see if an alias is
				// available, and if so, redirect to it.
				$currentPath = ltrim($this->here, "/");

				$this->loadModel('Routable.Route');
				$routeRecord = $this->Route->find('first', [
					'conditions' => [
						'actual LIKE' => '%' . $currentPath
					],
					'callbacks' => false
				]);

				if (! empty($routeRecord)) {
					// Found an alias, do a 301 redirect to it as the user shouldn't
					// be accessing this direct url.
					$alias = $routeRecord['Route']['alias'];
					$this->redirect('/' . $alias, 301);
				}
			}
		}
	}
}
