<?php

App::uses('Model', 'Model');
App::uses('Document', 'EvCore.Model');
App::uses('Image', 'EvCore.Model');

/**
 * EvCore model for Cake.
 *
 * All logic which is default to the Evoluted CMS is in here
 *
 * @package       app.Model
 */
class EvCoreAppModel extends Model {

	// Switch off automatic recursize queries app wide.
	public $recursive = -1;

	public $displayField = "name";

	// Name used in the admin theme for the model (override in your models or
	// leave as null to default to the model name).
	public $displayName = null;

	/**
	 * Define the number of images associated with the model.
	 *
	 * @var  integer|array
	 *
	 * It is now possible to have unlimited image slots
	 * Image Slots
	 * 		-1 = 	Unlimited Image Slots
	 *		0  = 	No Image Slots
	 *		X  = 	X Amount of Image Slots
	 *
	 * This can be done in 2 ways
	 * 		1. If the model only has one set of images
	 *			"public $imageSlots = 0;"
	 *
	 *		2. If there are multiple image areas to a page
	 *			public $imageSlots = array(
	 *				'main' => 4,
	 *				'listing' => 1,
	 *			);
	 *
	 *		public $imageSlots = array(
	 *			'main' => array(
	 *				'fields' => array(
	 *					'caption' => array(
	 *						'label' => 'Alt', //This array is passed through to FormHelper::input so can be used to  add class etc.
	 *					),
	 *					'button_text'
	 *				),
	 *				'slots' => -1
	 *			)
	 *		);
	 *
	 *
	 */
	public $imageSlots = 0;

	/**
	 * Define the number of files/documents associated with the model.
	 *
	 * @see  AppModel::imageSlots
	 * @var  integer|array
	 */
	public $documentSlots = 0;

	// Define the menu content in this model belong in when using the
	// Navigatable behaviour.
	public $parentMenu = null;

	/**
	 * store when the behaviours have been checked so it doesn't happen again
	 * if bindModel is called
	 *
	 */
	public $behaviorsChecked = false;

	/**
	 * Returns a list of all events that will fire in the model during it's lifecycle.
	 *
	 * @return array
	 */
	public function implementedEvents() {
		$events = parent::implementedEvents();
		$events['Model.beforeBeforeSave'] = array('callable' => 'beforeBeforeSave', 'passParams' => true);
		return $events;
	}

	/**
	 * redefine and call parent so we can append our behavior checking code
	 * allows us to check for overriding behaviors without needing to overriding any actual code
	 */
	/**
	 * Create a set of associations.
	 *
	 * @return void
	 */
	protected function _createLinks() {
		parent::_createLinks();

		// Evoluted Amend. Loop and check for any plugins in behaviours
		// Check these haven't been overwritten
		if (! empty($this->actsAs) && ! $this->behaviorsChecked) {
			$this->actsAs = EvClassRegistry::checkOverridingItems('behaviors', $this->actsAs);
			$this->behaviorsChecked = true;
		}
	}

	/**
	 * Setup the model
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		if ($this->displayName == null) {
			$this->displayName = $this->alias;
		}

		$this->initialiseAssociateModels();

		return;
	}

	/**
	 * Initialises the associated models such as Image, Document and Block
	 */
	public function initialiseAssociateModels() {
		// Set up model associations for images automagically based on the
		// image slots.
		$this->_associateModels('EvCore.Image', $this->imageSlots);

		// Set up model associations for documents automagically based on the
		// document slots.
		$this->_associateModels('EvCore.Document', $this->documentSlots);
	}

	/**
	 * Automagically associate attachments and blocks with the model based on the
	 * number of slots defined. This method is called from the constructor.
	 *
	 * @param  string $modelAlias either 'Document', 'Image' or 'Block'
	 * @param  mixed  $modelSlots  array or integer defining the number of slots required
	 */
	protected function _associateModels($modelAlias, $modelSlots, $type = 'attachment_type') {
		list($plugin, $modelName) = pluginSplit($modelAlias);

		if (is_array($modelSlots)) {
			foreach ($modelSlots as $section => $slots) {
				if ($slots != 0) {
					$model = ($section == 'main') ? $modelName : Inflector::camelize($section) . $modelName;

					$conditions = array(
						'model' => $this->name,
						$type => $model
					);

					// check for custom conditions
					if (! empty($slots['conditions'])) {
						$conditions = array_merge(
							$conditions,
							$slots['conditions']
						);
					}

					$this->hasMany[$model] = array(
						'className' => $modelAlias,
						'foreignKey' => 'model_id',
						'conditions' => $conditions,
						'dependent' => true,
						'cascade' => true
					);
				}
			}

		} elseif ($modelSlots != 0) {

			$this->hasMany[$modelName] = array(
				'className' => $modelAlias,
				'foreignKey' => 'model_id',
				'conditions' => array(
					'model' => $this->name,
					$type => $modelName
				),
				'dependent' => true,
				'cascade' => true
			);

		}

		return;
	}

	/**
	 * Override in your on Model to customise
	 */
	public function validate() {
		return $this->validate;
	}

	/**
	 * define a generate paginateCount to get round cake doing a count on a model
	 * and that models behaviours / callbacks being triggered. Was causing slow downs with this like
	 * MetaData plugin and bringing out every single meta data record in the system!
	 * Simply do the count and set the conditions but disable callbacks
	 *
	 * @param array $conditions
	 * @param boolean $recursive
	 * @param array $extra
	 * @return array
	 */
	public function paginateCount($conditions = null, $recursive = 0, $extra = array()) {
		$params = array_merge(
			array(
				'conditions' => $conditions,
				'callbacks' => false
			),
			$extra
		);
		unset($params['contain']);

		return $this->find(
			'count',
			$params
		);
	}

	/**
	 * Model pre-save logic
	 */
	public function beforeSave($options = array()) {
		// remove trailing empty paragraph tag from content
		if (isset($this->data[$this->alias]['content']) && !empty($this->data[$this->alias]['content'])) {

			$this->data[$this->alias]['content'] = preg_replace("/(<p[^>]*>[\s|&nbsp;]*<\/p>\n?)*$/", '', $this->data[$this->alias]['content']);
		}

		return true;
	}

	/**
	 * redefine saveAll to add in beforeBeforeSave callback
	 * allows related data to be modified
	 *
	 */
	public function saveAll($data = array(), $options = array()) {
		if (empty($data)) {
			$data = $this->data;
		}

		/*
		 * manual callbacks for beforeBeforeSave for models and behaviors
		 * cannot be properly implemented by using events system
		 */
		if (method_exists($this, 'beforeBeforeSave')) {
			$data = $this->beforeBeforeSave($data, $options);

			if ($data === false) {
				return false;
			}
		}

		// loop and trigger any behaviors with the method
		$loadedBehaviors = $this->Behaviors->loaded();
		foreach ($loadedBehaviors as $behavior) {
			if (method_exists($this->Behaviors->{$behavior}, 'beforeBeforeSave')) {
				$data = $this->Behaviors->{$behavior}->beforeBeforeSave($this, $data, $options);
				if ($data === false) {
					return false;
				}
			}
		}

		return parent::saveAll($data, $options);
	}

	/**
	 * get the list of brands for drop down menu
	 *
	 * @param 	array 	array of query params
	 * @return 	array
	 */
	public function getForDropDown($query = array()) {
		if ($this->hasField('is_active')) {
			$query['condition'][$this->alias . '.is_active'] = 1;
		}

		if ($this->hasField('sequence')) {
			$query['order'] = $this->alias . '.sequence ASC';
		}
		$query['callbacks'] = false;

		return $this->find(
			'list',
			$query
		);
	}

	/**
	 * convience wrapper replacement for field()
	 * field() still passes to callbacks (behaviors as well) - in most instances
	 * this is undesired. Emulate this without callbacks
	 *
	 * @param 	string 		The field we wish to retrieve
	 * @param 	array 		Array of conditions to search on.
	 * @param 	string 		Sql Order Condition
	 * @return 	string 		Value from the first returned result
	 */
	public function fieldOnly($field, $conditions = array(), $order = '') {
		$data = $this->find(
			'first',
			array(
				'fields' => $field,
				'conditions' => $conditions,
				'order' => $order,
				'callbacks' => false
			)
		);
		return Hash::get($data, $this->alias . '.' . $field); // will return null if no $data found
	}

	/**
	 * Base code for retrieving a single record for edit or view
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	protected function _getItem($id, $query = array()) {
		$query = array_merge(
			array(
				'getRoutes' => true,
				'getMenuItem' => true,
				'customFields' => true,
				'customFieldsSyntaxFormat' => false
			),
			$query
		);

		$alias = $this->alias;

		$query['conditions'][$alias . '.id'] = $id;

		if ($this->hasField('is_removed')) {
			$query['conditions']['is_removed'] = false;
		}

		if ($this->hasBehavior('EvTemplates.Template')) {
			$templateId = intval($this->fieldOnly('template_id', array($alias . '.id' => $id)));
			if ($templateId > 0) {
				$this->setupAttachmentRelationships($templateId);
			}
		}

		$query = array_merge_recursive($query, $this->_containAssociateModel('Image', $this->imageSlots));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Document', $this->documentSlots));

		$data = $this->find('first', $query);

		// We only want to get the route if it has Routable and is also on Edit pages
		if ($query['getRoutes'] && $this->hasBehavior('Routable.Routable')) {

			$data['Route'] = $this->getRoute($data); // method on Routable Behaviour
		}

		// We only want to get the menu item if it has Navigatable and is also on Edit pages
		if ($query['getMenuItem'] && $this->hasBehavior('EvNavigation.Navigatable')) {

			$data['Menu'] = $this->getMenuItem($data); // method on Navigatable Behaviour
		}

		return $data;
	}

	protected function _containAssociateModel($modelAlias, $modelSlots) {
		$query = array();

		if (is_array($modelSlots)) {
			foreach ($modelSlots as $section => $slots) {
				if ($slots != 0) {
					$model = ($section == "main") ? $modelAlias : Inflector::camelize($section) . $modelAlias;
					$query['contain'][$model] = array(
						'order' => 'sequence'
					);
				}
			}
		} elseif (is_numeric($modelSlots) && $modelSlots != 0) {
			$query['contain'][$modelAlias] = array(
				'order' => 'sequence'
			);
		}

		return $query;
	}

	/**
	 * Wrapper for tge containAssociateModel method, is protected so provide array of alias => slots
	 * to be able to call method
	 *
	 * @param 	array
	 * @return 	array
	 */
	public function containSlotAssociations($slots) {
		$query = array();
		if (! empty($slots)) {
			foreach ($slots as $alias => $slots) {
				$query = array_merge_recursive($query, $this->_containAssociateModel($alias, $slots));
			}
		}

		return $query;
	}

	/**
	 * Query used to retrieve a record ready for edit
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $params = array()) {
		$params['readForEdit'] = true;

		return $this->_getItem($id, $params);
	}

	/**
	 * Query used to retrieve a record ready for view.
	 *
	 * By default calls readForEdit but can be extended to bring out more
	 * complicated data.
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForView($id, $params = array()) {
		if ($this->hasField('is_active')) {
			$params['conditions'][$this->escapeField('is_active')] = true;
		}

		$params['getRoutes'] = false;
		$params['getMenuItem'] = false;
		$params['customFieldsSyntaxFormat'] = true;
		$params['readForView'] = true;

		return $this->_getItem($id, $params);
	}

	/**
	 * Query used to retrieve a record ready for admin view.
	 *
	 * By default calls readForEdit but can be extended to bring out more
	 * complicated data.
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForAdminView($id, $params = array()) {
		return $this->readForEdit($id, $params);
	}

	/**
	 * Query used to retrieve a listing of records ready for template
	 *
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForIndex($query = array()) {
		$alias = $this->alias;

		if ($this->hasField('is_active')) {
			$query['conditions']['is_active'] = true;
		}

		if ($this->hasField('is_removed')) {
			$query['conditions']['is_removed'] = false;
		}

		$query = array_merge_recursive($query, $this->_containAssociateModel('Image', $this->imageSlots));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Document', $this->documentSlots));

		return $this->find('all', $query);
	}

	/**
	 * Checks if the record is protected
	 * Removes associated images and documents
	 *
	 * @param boolean $cascade true if the delete is to cascade to children
	 * @return boolean true if the opperation was successful, otherwise false
	 * @see Model::beforeDelete()
	 */
	public function beforeDelete($cascade = true) {
		if ($this->hasField('is_protected') && $this->field('is_protected')) {
			return false;
		}

		return parent::beforeDelete($cascade);
	}

	/**
	 * Returns a list of dependant records (usually displayed on the delete dialog)
	 *
	 * @param integer $id ID of the row we are checking dependencies for
	 */
	public function findDependents($id) {
		$dependents = array();

		$models = array_merge_recursive($this->hasOne, $this->hasMany);

		foreach ($models as $model => $attr) {
			//If model is set as not dependent then dont need to look in it for dependents
			if ((isset($attr['dependent']) && !$attr['dependent']) || (isset($attr['cascade']) && $attr['cascade'])) {
				continue;
			}

			$params = array();

			//If default conditions are set on the relationship we need to listen to them
			if (isset($attr['conditions'])) {
				$params['conditions'] = $attr['conditions'];
			}

			$params['conditions'][$model . "." . $attr['foreignKey']] = $id;

			$found = $this->$model->find('all', $params);

			foreach ($found as $item) {
				$value = $item[$model][$this->$model->displayField];
				$dependents[$model][] = $value;
			}
		}

		return $dependents;
	}

	/**
	 * Custom validate rule, checks if two fields are identical
	 * (useful for comparing password and confirm_password)
	 *
	 * @param array $check Field to trigger the check
	 * @param array $otherField Other field to compare with
	 */
	public function checkMatches($check, $otherField) {
		$value = array_pop($check);

		return $value == $this->data[$this->alias][$otherField];
	}

	/**
	 * Checks if a combination of field values are unique in the DB
	 *
	 * @param $check array field the check is triggered by
	 * @param $fields array list of fields that form part of the unique key
	 *
	 * @return boolean true if combination doesn't exists in the database, otherwise false
	 */
	public function checkCombinationUnique($check, $fields) {
		$fieldToTest = key($check);
		$value = array_pop($check);

		// build up the list of fields and their values into a condition array
		$conditions = array(
			$this->alias . "." . $fieldToTest => $value
		);

		foreach ($fields as $field) {
			$conditions[$this->alias . "." . $field] = $this->data[$this->alias][$field];
		}

		// don't include the current row in the duplicate check
		if (isset($this->data[$this->alias][$this->primaryKey]) && !empty($this->data[$this->alias][$this->primaryKey])) {
			$conditions[$this->alias . "." . $this->primaryKey . " <>"] = $this->data[$this->alias][$this->primaryKey];
		}

		// if we find a match, the combination already exists
		$found = $this->find('first', array(
			'conditions' => $conditions
		));

		return empty($found);
	}

	/**
	 * Toggles a field on or off.
	 *
	 * @param string $field field to toggle
	 * @param integer $id ID of row
	 *
	 * @return result of updateAll()
	 */
	public function toggleField($field, $id = null) {
		$id = empty($id) ? $this->id : $id;

		$field = $this->escapeField($field);

		$fields = array($field => "NOT $field");
		$conditions = array($this->escapeField() => $id);

		return $this->updateAll($fields, $conditions);
	}

	/**
	 * Custom delete method that checks whether the current model has an is_removed flag
	 * to mark as true rather than delete before actually deleting content
	 * @param  int  $id
	 * @param  boolean $cascade
	 * @return boolean
	 */
	public function delete($id = null, $cascade = true) {
		if ($this->hasField('is_removed')) {

			$this->id = $id;
			$this->saveField('is_removed', true);

			if ($this->hasField('is_active')) {
				$this->saveField('is_active', false);
			}

			$result = true;

		} else {
			$result = parent::delete($id, $cascade);
		}

		return $result;
	}

	/**
	 * Gets the next order number in a sequence.
	 *
	 * @param array $params
	 * @return integer
	 */
	public function getNextSequenceNo($params = array()) {
		$defaults = array(
			'order' => 'sequence DESC',
			'limit' => 1
		);

		$params = array_merge($params, $defaults);

		$result = $this->find('first', $params);

		return $result ? $result[$this->alias]['sequence'] + 1 : 0;
	}

	public function getLastQuery() {
		$log = $this->getDataSource()->getLog(false, false);
		// return the first element of the last array (i.e. the last query)
		return $log['log'][(count($log['log']) - 1)];
	}

	/**
	 * § Behaviours
	 */

	/**
	 * Helper function to check if the model has a particular behaviour
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function hasBehavior($behavior) {
		if (empty($this->actsAs)) {
			return false;
		}

		$setup = $this->setupBehavior($behavior);
		extract($setup);

		$behaviorKeys = array_keys($this->actsAs);
		$behaviorValues = array_values($this->actsAs);
		$behaviorClassNames = Hash::extract($this->actsAs, '{s}.className');

		return in_array($behavior, array_merge($behaviorKeys, $behaviorValues, $behaviorClassNames), true);
	}

	/**
	 * Helper function to remove a behaviour from a model
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function removeBehavior($behavior) {
		$setup = $this->setupBehavior($behavior);
		extract($setup);

		if ($this->hasBehaviour($behavior)) {
			unset($this->actsAs[$behavior]);

			$this->Behaviors->unload($behavior);
		}

		return true;
	}

	/**
	 * Helper function to add a behaviour to a model
	 *
	 * @param string $behaviour name of the behaviour
	 * @param array $config (optional) behaviour's config
	 * @param boolean $overrideBehaviour (optional) pass as true to reload the behaviour if already loaded
	 * @return boolean
	 */
	public function addBehavior($behavior, $config = array(), $overrideBehavior = false) {
		$setup = $this->setupBehavior($behavior, $config);
		extract($setup);

		if ($overrideBehavior === true || ! $this->hasBehaviour($behavior)) {
			$this->actsAs[$behavior] = $config;
			$this->Behaviors->load($behavior, $config);
		}

		return true;
	}

	/**
	 * backwards compatibility methods
	 */
	public function hasBehaviour($behavior) {
		return $this->hasBehavior($behavior);
	}

	public function removeBehaviour($behavior) {
		return $this->removeBehavior($behavior);
	}

	public function addBehaviour($behavior, $config = array(), $overrideBehavior = false) {
		return $this->addBehavior($behavior, $config, $overrideBehavior);
	}

	/**
	 * setup a behavior
	 *
	 * @param 	string 	Behavior to load
	 * @param 	array 	(optionval) Behaviors config
	 * @return  array  	array containing behavior / array config
	 */
	public function setupBehavior($behavior, $config = array()) {
		// check for a plugin / overriding file
		$origBehavior = $behavior;

		if (! empty($config['className'])) {
			$behavior = $config['className'];
		}

		$behavior = EvClassRegistry::findOverrideBehavior($behavior);
		$config['className'] = $behavior;

		// as we have set the className, we need to chop up the original
		// helper call if it was a plugin to set the alias correctly.
		$behavior = EvClassRegistry::getNewArrayKey($origBehavior);

		return array(
			'behavior' => $behavior,
			'config' => $config
		);
	}

	/**
	 * get a constant from the object
	 *
	 * @param 	string 	$constant 	The constant to retrieve
	 * @return 	mixed
	 */
	public function getConstant($constant) {
		return constant(get_class($this) . '::' . $constant);
	}
}
