<?php
App::uses('CustomEmail', 'Lib');

/**
 * Handles User requests
 *
 * @package CoreCMS/User
 */
class UsersController extends AppController {

	/**
	 * Controller name
	 *
	 * @var string
	 */
	public $name = 'Users';

	/**
	 * List of actions permitted by admin users.
	 */
	public $adminActions = array(
		'admin_index',
		'admin_add',
		'admin_edit',
		'admin_account',
		'admin_delete',
		'admin_toggle',
		'admin_password_reset',
		'admin_password_reset_callback',
		'admin_login',
		'admin_logout'

	);

	/**
	 * redefine the constructor to add in our user manage component
	 */
	public function __construct(CakeRequest $Request, CakeResponse $Response) {
		$this->components['UserManager'] = array(
			'className' => 'EvCore.UserManager'
		);

		parent::__construct($Request, $Response);
	}

	/**
	 * Allow public access to some methods.
	 */
	public function beforeFilter() {
		$this->Auth->allow(array(
			'admin_login',
			'admin_password_reset',
			'admin_password_reset_callback',
			'password_reset',
			'password_reset_email',
			'password_reset_callback',
			'password_reset_fail',
			'register',
			'verify_email'
		));

		parent::beforeFilter();

		$notInAdmin = (! isset($this->request->params['admin']) || $this->request->params['admin'] === false);
		if (! Configure::read('SiteSetting.users.enable_registration') && $notInAdmin) {
			throw new NotFoundException();
			return;
		}
	}

	/**
	 * Handles User login. Redirects back to the calling page on successful login.
	 *
	 * @return void
	 */
	public function login() {
		// If user is logged in redirect to the homepage.
		$user = $this->Auth->user();

		if (! empty($user)) {
			return $this->redirect('/');
		}

		if ($this->request->is('post')) {
			$result = $this->UserManager->login(EvClassRegistry::init('EvCore.User'));

			if ($result['result'] === true) {
				$this->redirect(
					$result['data']['redirect']
				);
			} else {
				$this->Flash->fail(
					$result['data']['errors']
				);
			}
		}

		$this->view = 'EvCore.Users/login';
	}

	/**
	 * Admin login.
	 *
	 * @return void
	 */
	public function admin_login() {
		if ($this->request->is('post')) {
			$result = $this->UserManager->login(EvClassRegistry::init('EvCore.User'));

			if ($result['result'] === true) {
				$this->redirect(
					$result['data']['redirect']
				);
			} else {
				$this->Flash->fail(
					$result['data']['errors']
				);
			}
		}

		$this->layout = "EvCore.admin_login";

		$this->view = 'EvCore.Users/admin_login';
	}

	/**
	 * handle a user registration on the front end
	 *
	 */
	public function register() {
		if (! empty($this->request->data) && $this->request->is('post')) {

			$result = $this->UserManager->register(EvClassRegistry::init('EvCore.User'));
			if ($result['result'] === true) {

				$this->Flash->success($result['data']['message']);

				return $this->redirect($result['data']['redirect']);
			} else {

				$this->Flash->fail($result['data']['errors']);
			}
		}

		$this->view = 'EvCore.Users/register';
	}

	/**
	 * handle a user verification via email
	 *
	 */
	public function verify_email($code) {
		if (Configure::read('SiteSetting.users.activation') !== 'email') {
			throw new NotFoundException;
		}

		// try to find the user
		$user = $this->User->findByVerificationCode($code);

		if (empty($user)) {
			$this->redirect(array('action' => 'login'));
			return;
		}

		// code found, activate and delete code
		if ($this->User->verifyFromEmail($user['User']['id'], $user['User']['verification_code'])) {
			$this->Session->setFlash(
				'Your account has been verified, redirecting you to the login',
				'flash_success'
			);
		} else {
			$this->Session->setFlash(
				'The verification code was correct but there was a problem activating your account, please try again or contact us for assistance.',
				'flash_fail'
			);
		}

		return $this->redirect(array('action' => 'login'));
	}

	/**
	 * Handles User logout. Redirects to Auth::logoutRedirect (defaults as the homepage).
	 *
	 * @return void
	 */
	public function logout() {
		return $this->redirect($this->Auth->logout());
	}

	/**
	 * Handles admin logout. Currently just wrapper for standard logout
	 *
	 * @return void
	 */
	public function admin_logout() {
		return $this->logout();
	}

	/**
	 * Password reset.
	 *
	 * @return void
	 */
	public function password_reset() {
		return $this->_passwordReset();
	}

	/**
	 * Password reset when in admin section
	 *
	 * @return void;
	 */
	public function admin_password_reset() {
		$this->_passwordReset(true);

		$this->layout = 'EvCore.admin_login';

		return;
	}

	/**
	 * User password reset form.
	 *
	 * @return void
	 */
	protected function _passwordReset($inAdmin = false) {
		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$user = $Model->findByEmail($this->request->data[$Model->alias]['email']);

			if (!$user) {

				$this->Session->setFlash('Email not found', 'flash_fail');

			} else {
				$this->_passwordResetEmail($user[$Model->alias]['id']);

				$this->view = 'EvCore.Users/';
				if ($inAdmin) {

					$this->view .= 'admin_password_reset_email';
				} else {

					$this->view .= 'password_reset_email';
				}

				return;
			}
		}

		$this->view = 'EvCore.Users/';
		if ($inAdmin) {

			$this->view .= 'admin_password_reset';
		} else {

			$this->view .= 'password_reset';
		}

		return;
	}

	/**
	 * Creates a reset code and emails it to the user
	 *
	 * @param integer $userId ID of user to reset password for
	 * @return void
	 */
	protected function _passwordResetEmail($userId) {
		$Model = $this->{$this->modelClass};

		if ($Model->resetPassword($userId) === false) {
			$this->Session->setFlash('User not found.', 'flash_fail');
			$this->redirect(
				array(
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
			return;

		}

		$this->set('resetUser', $Model->findById($userId));

		return;
	}

	/**
	 * Handles callback from password reset email. Allows a user to change their
	 * password.
	 *
	 * @param string $code
	 * @return void
	 */
	public function password_reset_callback($code) {
		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validatePassword();

		$user = $Model->findByPasswordResetCode($code);

		if (!$user) {

			$this->Session->setFlash('Password reset code not found', 'flash_fail');
			$this->redirect(
				array(
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
		}

		$this->_passwordResetCallback($code, $user);

		$this->view = 'EvCore.Users/password_reset_callback';
	}

	/**
	 * Handles callback from password reset email
	 * Allows user to change their password
	 *
	 * @param unknown_type $code
	 * @return void
	 */
	public function admin_password_reset_callback($code) {
		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validatePassword();

		$user = $Model->findByPasswordResetCode($code);

		if (!$user) {

			$this->Session->setFlash('Password reset code not found', 'flash_fail');
			$this->redirect(
				array(
					'admin' => true,
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
			return;
		}

		$this->_passwordResetCallback($code, $user);

		$this->layout = 'EvCore.admin_login';
		$this->view = 'EvCore.Users/admin_password_reset_callback';
	}

	/**
	 * Handles callback from password reset email
	 * Allows user to change their password
	 *
	 * @param unknown_type $code
	 * @param array user array from findByPasswordResetCode
	 * @return void
	 */
	protected function _passwordResetCallback($code, $user) {
		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];
			$this->request->data[$Model->alias]['password_reset_code'] = "";

			if ($Model->save($this->request->data)) {

				$user = $Model->findByPasswordResetCode($code);
				$this->Session->setFlash('Password changed', 'flash_success');

				$this->redirect(array(
					'action' => 'login'
				));
				return;

			} else {

				$this->Session->setFlash('Please correct the errors:', 'flash_fail');

			}
		}

		return;
	}

	/**
	 * Populates drop down lists for admin forms.
	 *
	 * @return void
	 */
	protected function _adminPopulateLookups() {
		$Model = $this->{$this->modelClass};

		$this->set('userGroups', $Model->UserGroup->find('list'));

		return;
	}

	/**
	 * Defines the fields used in the admin filter form.
	 *
	 * @return array
	 */
	protected function _adminFilterFields() {
		$Model = $this->{$this->modelClass};

		$this->_adminPopulateFilterLookups();

		$fields = parent::_adminFilterFields();

		$fields[$Model->alias . '.user_group_id'] = array(
			'label' => 'User Group',
			'type' => 'select',
			'compare' => array($Model->alias . '.user_group_id' => '%s')
		);

		unset($fields[$Model->alias . '.password']);
		unset($fields[$Model->alias . '.password_reset_code']);
		unset($fields[$Model->alias . '.password_reset_code_expires']);

		return $fields;
	}

	/**
	 * Defines the columns to display for admin_index().
	 *
	 * @return array
	 */
	protected function _adminIndexColumns() {
		$Model = $this->{$this->modelClass};

		return array(
			$Model->alias . '.id' => array(
				'label' => 'ID',
				'type' => 'integer',
			),
			$Model->alias . '.email' => array(
				'label' => 'Email',
				'type' => 'string',
			),
			'UserGroup.name' => array(
				'label' => 'User Group',
				'type' => 'string'
			),
			$Model->alias . '.is_active' => array(
				'type' => 'boolean',
				'label' => 'Active'
			)
		);
	}

	protected function _adminIndexPaginate() {
		$Model = $this->{$this->modelClass};

		$paginate = parent::_adminIndexPaginate();
		$paginate['contain'][] = 'UserGroup';

		if ($this->Auth->user($Model->alias . '.id') != 1) {

			$paginate['conditions'][$Model->alias . '.id !='] = 1;
		}

		return $paginate;
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$Model = $this->{$this->modelClass};

		$fields[$Model->alias . '.password']['div'] = 'input text required';
		$fields[$Model->alias . '.password']['type'] = 'password';

		unset(
			$fields[$Model->alias . '.is_guest_user'],
			$fields[$Model->alias . '.password_reset_code'],
			$fields[$Model->alias . '.password_reset_code_expires'],
			$fields[$Model->alias . '.verification_code']
		);

		return $fields;
	}

	/**
	 * Add or edit a user.
	 *
	 * @param  integer $id User ID
	 * @return void
	 */
	public function admin_edit($id = null) {
		$Model = $this->{$this->modelClass};

		// unset password fields if they're not needed
		if (!empty($this->request->data)) {

			if (empty($this->request->data[$Model->alias]['password'])) {

				unset($this->request->data[$Model->alias]['password'], $this->request->data[$Model->alias]['confirm_password']);
			}

		}

		// call the parent admin_edit to do the add / edit
		parent::admin_edit($id);

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		return;
	}

	/**
	 * Admin 'My Account'.
	 *
	 * @return void
	 */
	public function admin_account() {
		$user = $this->Auth->user();
		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validateEdit();

		if (!empty($this->request->data)) {

			// Ensure the user is updating their own record only.
			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];

			if ($Model->saveAll($this->request->data)) {

				$this->Session->setFlash(array(
					'title' => "Account Updated",
					'description' => "Your account has been successfully updated!"
				), 'flash_success');

				return $this->redirect(array('action' => 'account'));

			} else {

				$this->Session->setFlash(array(
					'title' => 'Save failed',
					'description' => 'Failed to update your account'
				), 'flash_fail');

			}

		} else {

			$this->request->data = $Model->readForEdit($user[$Model->alias]['id']);

		}

		$this->_adminPopulateLookups();

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'My Account');

		$this->view = 'EvCore.Users/admin_account';
	}

	/**
	 * Edit current user account.
	 *
	 * @return void
	 */
	public function account() {
		$Model = $this->{$this->modelClass};

		$user = $this->Auth->user();

		$Model->validate = $Model->validateEdit();

		if (!empty($this->request->data)) {

			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];

			if ($this->request->data[$Model->alias]['password'] == "") {

				unset($this->request->data[$Model->alias]['password']);
				unset($this->request->data[$Model->alias]['conform_password']);

			} else {

				$Model->validate = array_merge_recursive($Model->validate, $Model->validatePassword());

			}

			if ($Model->save($this->request->data)) {

				$this->Session->setFlash(__('Information updated'), 'flash_success');
				return $this->redirect(
					array(
						'action' => 'account'
					)
				);
			}

		} else {

			$this->request->data = $Model->findById($user[$Model->alias]['id']);

		}

		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'Update Account Information | ' . Configure::read('SiteSetting.general.site_title'));

		$this->view = 'EvCore.Users/account';
	}

}
