<?php

App::uses('AppComponent', 'Controller/Component');

class UserManagerComponent extends AppComponent {

	/**
	 * register a user
	 *
	 * @param 	object 	$Model 		The model object to use for saving
	 * @return 	array 	$result 	Two element array - result(bool) / data(array)
	 */
	public function register($Model) {
		if (Configure::read('SiteSetting.users.enable_registration')) {
			if ($Model->register($this->_controller->request->data)) {

				$activationType = Configure::read('SiteSetting.users.activation');

				if ($activationType == 'auto') {
					$message = 'Your account has been created, logging in...';

					$fullUser = $Model->getUserForLogin($Model->id);
					$this->_controller->Auth->login($fullUser);
					$redirect = $this->_controller->Auth->redirect();

				} elseif ($activationType == 'email') {
					$message = 'Your account has been created, however you will be unable to login until you have click the link in the verifcation email you have been sent.';
				} else {
					$message = 'Your account has been created but needs manually verifying by an adminitrator. You will be emailed once your account is active.';
				}

				if (! isset($redirect)) {
					$redirect = array(
						'action' => 'login'
					);
				}

				$result = true;
				$returnData = array(
					'redirect' => $redirect,
					'message' => $message
				);

			} else {

				$validationErrors = array();
				foreach ($Model->validationErrors as $key => $errors) {
					$validationErrors[] = $errors['0'];
				}

				$errors = array(
					'description' => 'Please correct the following issues and try again',
					'list' => $validationErrors
				);

				$result = false;
				$returnData = array(
					'errors' => $errors
				);
			}
		} else {
			$result = false;
			$returnData = array(
				'errors' => 'Registration is disabled'
			);
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}

	/**
	 * attempt to login a user
	 *
	 * @param 	object 	$Model 		The model object to use for saving
	 * @return 	array 	$result 	Two element array - result(bool) / data(array)
	 */
	public function login($Model) {
		$Model->validate = $Model->validateLogin();

		$Model->set($this->_controller->request->data);

		// Check if a delay is being applied to the login
		$loginDelay = $this->_controller->Session->read('Login.delay');

		if ($loginDelay && $loginDelay > time()) {
			// there is a delay preventing login

			$minutes = intval(($loginDelay - time()) / 60) + 1;

			$result = false;
			$returnData = array(
				'errors' => array(
					'title' => __('Login failed'),
					'description' => __('You will not be able to login for the next %s minute(s)!', $minutes)
				)
			);
		} else {
			// no login delay set
			// proceed as normal

			if ($Model->validates()) {
				if ($this->_controller->Auth->login()) {
					// Unset the login attempt limits

					$this->_controller->Session->write('Login.attempt', 0);
					$this->_controller->Session->write('Login.delay', 0);

					//Get a better version of the user model out in the format $data['User']
					$user = $this->_controller->Auth->user();

					$fullUser = $Model->getUserForLogin($user['id']);

					$this->_controller->Auth->login($fullUser);

					$result = true;
					$returnData = array(
						'redirect' => $this->_controller->Auth->redirect(),
						'message' => 'Login successful'
					);
				} else {
					// Define the time delays for limiting login attempts
					$loginDelays = array(0, 1, 5, 30, 60, 90);

					// Check login limits and write updated ones to session
					$loginAttempt = $this->_controller->Session->read('Login.attempt') ? $this->_controller->Session->read('Login.attempt') + 1 : 1;
					$loginDelay = isset($loginDelays[intval($loginAttempt / 5)]) ? $loginDelays[intval($loginAttempt / 5)] : 90;
					$this->_controller->Session->write('Login.attempt', $loginAttempt);
					$this->_controller->Session->write('Login.delay', time() + $loginDelay * 60);

					$result = false;

					if ($loginDelay > 0) {

						$returnData = array(
							'errors' => array(
								'title' => __('Login failed'),
								'description' => __('You will not be able to login for the next %s minute(s)!', $loginDelay)
							)
						);
					} else {
						$returnData = array(
							'errors' => array(
								'title' => __('Login failed'),
								'description' => __('Username or password is incorrect')
							)
						);
					}
				}

			} else {
				// validation failed

				$validationErrors = array();
				foreach ($Model->validationErrors as $key => $errors) {

					$validationErrors[] = $errors['0'];
				}

				$errors = array(
					'description' => 'The email and / or password was incorrect',
					'list' => $validationErrors
				);

				$result = false;
				$returnData = array(
					'errors' => $errors
				);

				unset($this->_controller->request->data[$Model->alias]['password']);
			}
		}

		return array(
			'result' => $result,
			'data' => $returnData
		);
	}
}
