<?php
/**
 * Copyable Behavior class file.
 *
 * Adds ability to copy a model record, including all hasMany and hasAndBelongsToMany
 * associations. Relies on Containable behavior, which this behavior will attach
 * on the fly as needed.
 *
 * HABTM relationships are just duplicated in the join table, while hasMany and hasOne
 * records are recursively copied as well.
 *
 * Usage is straightforward:
 * From model: $this->copy($id); // id = the id of the record to be copied
 * From controller: $this->MyModel->copy($id);
 *
 * @filesource
 * @author            Jamie Nay
 * @copyright       Jamie Nay
 * @license            http://www.opensource.org/licenses/mit-license.php The MIT License
 * @link            http://github.com/jamienay/copyable_behavior
 */
class CopyableBehavior extends ModelBehavior {
	/**
	 * Behavior settings
	 *
	 * @access public
	 * @var array
	 */
	public $settings = array();

	/**
	 * Array of contained models.
	 *
	 * @access public
	 * @var array
	 */
	public $contain = array();

	/**
	 * The full results of Model::find() that are modified and saved
	 * as a new copy.
	 *
	 * @access public
	 * @var array
	 */
	public $record = array();

	/**
	 * Default values for settings.
	 *
	 * - recursive: whether to copy hasMany and hasOne records
	 * - habtm: whether to copy hasAndBelongsToMany associations
	 * - stripFields: fields to strip during copy process
	 *
	 * @access private
	 * @var array
	 */
	private $defaults = array(
		'recursive' => true,
		'habtm' => true,
		'stripFields' => array('id', 'created', 'modified', 'lft', 'rght'),
		'append_display_field' => 0
	);

	/**
	 * Configuration method.
	 *
	 * @param object $Model Model object
	 * @param array $config Config array
	 * @access public
	 * @return boolean
	 */
	public function setup(Model $Model, $config = array()) {
		$this->settings[$Model->alias] = array_merge($this->defaults, $config);
		return true;
	}

	/**
	 * Copy method.
	 *
	 * @param object $Model model object
	 * @param mixed $id String or integer model ID
	 * @access public
	 * @return boolean
	 */
	public function copy($Model, $id, $recurse_level = 1) {
		$this->generateContain($Model, $recurse_level);

		$this->record = $Model->find('first', array(
			'conditions' => array($Model->alias.'.id' => $id),
			'contain' => $this->contain
		));


		if (empty($this->record)) {
			return false;
		}

		if (!$this->__convertData($Model)) {
			return false;
		}

		return $this->__copyRecord($Model);
	}

	/**
	 * Wrapper method that combines the results of __recursiveChildContain()
	 * with the models' HABTM associations.
	 *
	 * @param object $Model Model object
	 * @access public
	 * @return array;
	 */
	public function generateContain($Model, $recurse_level) {
		if (!$this->__verifyContainable($Model)) {
			return false;
		}

		$this->contain = array_merge($this->__recursiveChildContain($Model, $recurse_level), array_keys($Model->hasAndBelongsToMany));
		return $this->contain;
	}

	/**
	 * Strips primary keys and other unwanted fields
	 * from hasOne and hasMany records.
	 *
	 * @param object $Model model object
	 * @param array $record
	 * @access private
	 * @return array $record
	 */
	private function __convertChildren($Model, $record) {
		$children = array_merge($Model->hasMany, $Model->hasOne);
		foreach ($children as $key => $val) {
			if (!isset($record[$key]) || empty($record[$key])) {
				continue;
			}
			if (isset($record[$key][0])) {
				foreach ($record[$key] as $innerKey => $innerVal) {
					$record[$key][$innerKey] = $this->__stripFields($Model, $innerVal);

					if (array_key_exists($val['foreignKey'], $innerVal)) {
						unset($record[$key][$innerKey][$val['foreignKey']]);
					}

					$record[$key][$innerKey] = $this->__convertChildren($Model->{$key}, $record[$key][$innerKey]);
				}
			} else {
				$record[$key] = $this->__stripFields($Model, $record[$key]);

				if (isset($record[$key][$val['foreignKey']])) {
					unset($record[$key][$val['foreignKey']]);
				}

				$record[$key] = $this->__convertChildren($Model->{$key}, $record[$key]);
			}
		}

		return $record;
	}

	/**
	 * Strips primary and parent foreign keys (where applicable)
	 * from $this->record in preparation for saving.
	 *
	 * @param object $Model Model object
	 * @access private
	 * @return array $this->record
	 */
	private function __convertData($Model) {
		$this->record[$Model->alias] = $this->__stripFields($Model, $this->record[$Model->alias]);
		$this->record = $this->__convertHabtm($Model, $this->record);
		$this->record = $this->__convertChildren($Model, $this->record);
		return $this->record;
	}

	/**
	 * Loops through any HABTM results in $this->record and plucks out
	 * the join table info, stripping out the join table primary
	 * key and the primary key of $Model. This is done instead of
	 * a simple collection of IDs of the associated records, since
	 * HABTM join tables may contain extra information (sorting
	 * order, etc).
	 *
	 * @param object $Model    Model object
	 * @access public
	 * @return array modified $record
	 */
	private function __convertHabtm($Model, $record) {
		if (!$this->settings[$Model->alias]['habtm']) {
			return $record;
		}
		foreach ($Model->hasAndBelongsToMany as $key => $val) {
			if (!isset($record[$val['className']]) || empty($record[$val['className']])) {
				continue;
			}

			$joinInfo = Set::extract($record[$val['className']], '{n}.'.$val['with']);
			if (empty($joinInfo)) {
				continue;
			}

			foreach ($joinInfo as $joinKey => $joinVal) {
				$joinInfo[$joinKey] = $this->__stripFields($Model, $joinVal);

				if (array_key_exists($val['foreignKey'], $joinVal)) {
					unset($joinInfo[$joinKey][$val['foreignKey']]);
				}
			}

			$record[$val['className']] = $joinInfo;
		}

		return $record;
	}

	/**
	 * Performs the actual creation and save.
	 *
	 * @param object $Model Model object
	 * @access private
	 * @return mixed
	 */
	private function __copyRecord($Model) {
		$Model->create();

		// Option to append a string to the end of the display field.
        if (! empty($this->settings[$Model->alias]['append_display_field'])) {
            $this->record[$Model->alias][$Model->displayField] = $this->record[$Model->alias][$Model->displayField] . $this->settings[$Model->alias]['append_display_field'];
        }

		$Model->set($this->record);
		return $Model->saveAll($Model->data);
	}

	/**
	 * Generates a contain array for Containable behavior by
	 * recursively looping through $Model->hasMany and
	 * $Model->hasOne associations.
	 *
	 * @param object $Model Model object
	 * @access private
	 * @return array
	 */
	private function __recursiveChildContain($Model, $recurse_level = false) {
		$contain = array();
		if (isset($this->settings[$Model->alias]) && !$this->settings[$Model->alias]['recursive']) {
			return $contain;
		}

		if($recurse_level > 1)
		{
			$children = array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));
			  foreach ($children as $child) {
				  $contain[$child] = $this->__recursiveChildContain($Model->{$child}, $recurse_level - 1);
			  }

		}
		else {
			return array_merge(array_keys($Model->hasMany), array_keys($Model->hasOne));

		}

		  return $contain;
	}

	/**
	 * Strips unwanted fields from $record, taken from
	 * the 'stripFields' setting.
	 *
	 * @param object $Model Model object
	 * @param array $record
	 * @access private
	 * @return array
	 */
	private function __stripFields($Model, $record) {
		foreach ($this->settings[$Model->alias]['stripFields'] as $field) {
			if (array_key_exists($field, $record)) {
				unset($record[$field]);
			}
		}

		return $record;
	}

	/**
	 * Attaches Containable if it's not already attached.
	 *
	 * @param object $Model Model object
	 * @access private
	 * @return boolean
	 */
	private function __verifyContainable($Model) {
		if (!$Model->Behaviors->attached('Containable')) {
			return $Model->Behaviors->attach('Containable');
		}

		return true;
	}

}