<?php
App::uses('CustomEmail', 'Lib');

/**
 * Handles User requests
 *
 * @package CoreCMS/User
 */
class EvUsersController extends AppController {

/**
 * Controller name
 *
 * @var string
 */
	public $name = 'Users';

/**
 * List of actions permitted by admin users.
 */
	public $adminActions = array(
		'admin_index',
		'admin_add',
		'admin_edit',
		'admin_account',
		'admin_delete',
		'admin_toggle',
		'admin_password_reset',
		'admin_password_reset_email',
		'admin_password_reset_callback',
		'admin_login'

	);


/**
 * Allow public access to some methods.
 */
	public function beforeFilter() {

		parent::beforeFilter();

		$this->Auth->allow(array(
			'admin_login',
			'admin_password_reset',
			'admin_password_reset_email',
			'admin_password_reset_callback',
			'password_reset',
			'password_reset_email',
			'password_reset_callback',
			'password_reset_fail'
		));

	}


/**
 * Handles User login. Redirects back to the calling page on successful login.
 *
 * @return void
 */
	public function login() {

		// If user is logged in redirect to the homepage.
		$user = $this->Auth->user();
		
		if (!empty($user)) {
			return $this->redirect('/');
		}

		$this->_login();
		return;

	}


/**
 * Admin login.
 *
 * @return void
 */
	public function admin_login() {

		$this->_login();

		$this->layout = "admin_login";

		$this->view = 'EvCore./Users/admin_login';

		return;

	}


/**
 * Login - with login attempt time-out built in.
 *
 * @return void
 */
	protected function _login() {

		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validateLogin();

		if (!empty($this->request->data)) {

			$Model->set($this->request->data);

			// Check if a delay is being applied to the login
			$loginDelay = $this->Session->read('Login.delay');

			if ($loginDelay && $loginDelay>time()) {

				$minutes = intval(($loginDelay - time()) / 60) + 1;

				$this->Session->setFlash(
					array(
						'title' => __('Login failed'),
						'description' => __('You will not be able to login for the next %s minute(s)!', $minutes)
					),
					'flash_fail'
				);

				return;

			}

			if ($Model->validates()) {

				if ($this->Auth->login()) {

					// Unset the login attempt limits
					$this->Session->write('Login.attempt', 0);
					$this->Session->write('Login.delay', 0);

					//Get a better version of the user model out in the format $data['User']
					$user = $this->Auth->user();

					$fullUser = $Model->getUserForLogin($user['id']);

					$this->Auth->login($fullUser);

					return $this->redirect($this->Auth->redirect());

				}
				else {

					// Define the time delays for limiting login attempts
					$loginDelays = array(0, 1, 5, 30, 60, 90);

					// Check login limits and write updated ones to session
					$loginAttempt = $this->Session->read('Login.attempt') ? $this->Session->read('Login.attempt') + 1 : 1;
					$loginDelay = isset($loginDelays[intval($loginAttempt/5)]) ? $loginDelays[intval($loginAttempt/5)] : 90;
					$this->Session->write('Login.attempt', $loginAttempt);
					$this->Session->write('Login.delay', time() + $loginDelay*60);

					if ($loginDelay>0) {

						$this->Session->setFlash(
							array(
								'title' => __('Login failed'),
								'description' => __('You will not be able to login for the next %s minute(s)!', $loginDelay)
							),
							'flash_fail'
						);

					} else {

						$this->Session->setFlash(__('Username or password is incorrect'), 'flash_fail');

					}

				}

			} else {

				$validationErrors = array();

				foreach ($this->Instructor->validationErrors as $key => $errors) {

					$validationErrors[] = $errors['0'];
				}

				$errors = array(
					'description' => '<p>There were some errors with the data you entered.</p>',
					'list' => $validationErrors
				);

				$this->Session->setFlash($errors, 'flash_fail', array());
				unset($this->request->data[$Model->alias]['password']);
			}

		}

		return;
	}


/**
 * Handles User logout. Redirects to Auth::logoutRedirect (defaults as the homepage).
 *
 * @return void
 */
	public function logout() {

		return $this->redirect($this->Auth->logout());

	}


/**
 * Password reset.
 *
 * @return void
 */
	public function password_reset() {

		return $this->_password_reset();

	}


/**
 * Password reset when in admin section
 *
 * @return void;
 */
	public function admin_password_reset() {

		$this->_password_reset();

		$this->layout = 'EvCore./admin_login';

		return;

	}


/**
 * User password reset form.
 *
 * @return void
 */
	protected function _password_reset() {

		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$user = $Model->findByEmail($this->request->data[$Model->alias]['email']);

			if (!$user) {

				$this->Session->setFlash('Email not found', 'flash_fail');

			} else {

				$this->_password_reset_email($user[$Model->alias]['id']);

				$this->view = 'EvCore.password_reset_email';

				return;

			}

		}

		$this->view = 'EvCore.password_reset';

		return;

	}


/**
 * Creates a reset code and emails it to the user
 *
 * @param integer $userId ID of user to reset password for
 * @return void
 */
	protected function _password_reset_email($userId) {

		$Model = $this->{$this->modelClass};

		if ($Model->resetPassword($userId)===false) {

			$this->Session->setFlash('User not found.', 'flash_fail');
			$this->redirect(
				array(
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
			return;

		}

		$this->set('resetUser', $Model->findById($userId));

		return;

	}


/**
 * Handles callback from password reset email. Allows a user to change their
 * password.
 *
 * @param string $code
 * @return void
 */
	public function password_reset_callback($code) {

		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validatePassword();


		$user = $Model->findByPasswordResetCode($code);

		if (!$user) {

			$this->Session->setFlash('Password reset code not found', 'flash_fail');
			$this->redirect(
				array(
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
		}

		$this->_password_reset_callback($code, $user);

		return;
	}


/**
 * Handles callback from password reset email
 * Allows user to change their password
 *
 * @param unknown_type $code
 * @return void
 */
	public function admin_password_reset_callback($code) {

		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validatePassword();


		$user = $Model->findByPasswordResetCode($code);

		if (!$user) {

			$this->Session->setFlash('Password reset code not found', 'flash_fail');
			$this->redirect(
				array(
					'admin' => true,
					'prefix' => 'admin',
					'controller' => strtolower($this->name),
					'action' => 'password_reset'
				)
			);
			return;
		}

		$this->_password_reset_callback($code, $user);

		$this->layout = 'EvCore./admin_login';
		$this->view = 'EvCore./Users/password_reset_callback';

		return;

	}


/**
 * Handles callback from password reset email
 * Allows user to change their password
 *
 * @param unknown_type $code
 * @param array user array from findByPasswordResetCode
 * @return void
 */
	protected function _password_reset_callback($code, $user) {

		$Model = $this->{$this->modelClass};

		if (!empty($this->request->data)) {

			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];
			$this->request->data[$Model->alias]['password_reset_code'] = "";

			if ($Model->save($this->request->data)) {

				$user = $Model->findByPasswordResetCode($code);
				$this->Session->setFlash('Password changed', 'flash_success');

				$this->redirect(array(
					'action' => 'login'
				));
				return;

			} else {

				$this->Session->setFlash('Please correct the errors:', 'flash_fail');

			}
		}

		return;

	}


/**
 * Populates drop down lists for admin forms.
 *
 * @return void
 */
	protected function _adminPopulateLookups() {

		$Model = $this->{$this->modelClass};

		$this->set('userGroups', $Model->UserGroup->find('list'));

		return;

	}


/**
 * Defines the fields used in the admin filter form.
 *
 * @return array
 */
	protected function _adminFilterFields() {

		$Model = $this->{$this->modelClass};

		$this->_adminPopulateFilterLookups();

		$fields = parent::_adminFilterFields();

		$fields[$Model->alias . '.user_group_id'] = array(
			'label' => 'User Group',
			'type' => 'select',
			'compare' => array($Model->alias . '.user_group_id' => '%s')
		);

		unset($fields[$Model->alias.'.password']);
		unset($fields[$Model->alias.'.password_reset_code']);
		unset($fields[$Model->alias.'.password_reset_code_expires']);

		return $fields;

	}


/**
 * Defines the columns to display for admin_index().
 *
 * @return array
 */
	protected function _adminIndexColumns() {

		$Model = $this->{$this->modelClass};

		return array(
			$Model->alias.'.id'=>array(
				'label'=>'ID',
				'type'=>'integer',
			),
			$Model->alias.'.email'=>array(
				'label'=>'Email',
				'type'=>'string',
			),
			'UserGroup.name'=>array(
				'label'=>'User Group',
				'type'=>'string'
			)
		);

	}


	protected function _adminIndexPaginate() {

		$Model = $this->{$this->modelClass};

		$paginate = parent::_adminIndexPaginate();
		$paginate['contain'][] = 'UserGroup';

		if ($this->Auth->user($Model->alias . '.id') != 1) {

			$paginate['conditions'][$Model->alias . '.id !='] = 1;
		}

		return $paginate;
	}


	protected function _adminFormFields() {

		$fields = parent::_adminFormFields();

		$Model = $this->{$this->modelClass};

		$fields[$Model->alias.'.password']['div'] = 'input text required';
		$fields[$Model->alias.'.password']['type'] = 'password';

		unset(
			$fields[$Model->alias.'.is_guest_user'],
			$fields[$Model->alias.'.password_reset_code'],
			$fields[$Model->alias.'.password_reset_code_expires']
		);

		return $fields;
	}


/**
 * Add or edit a user.
 *
 * @param  integer $id User ID
 * @return void
 */
	public function admin_edit($id = null) {

		$Model = $this->{$this->modelClass};

		// unset password fields if they're not needed
		if (!empty($this->request->data)) {

			if (empty($this->request->data[$Model->alias]['password'])) {

				unset($this->request->data[$Model->alias]['password'], $this->request->data[$Model->alias]['confirm_password']);
			}

		}

		// call the parent admin_edit to do the add / edit
		parent::admin_edit($id);

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		return;
	}


/**
 * Admin 'My Account'.
 *
 * @return void
 */
	public function admin_account() {

		$user = $this->Auth->user();
		$Model = $this->{$this->modelClass};

		$Model->validate = $Model->validateEdit();

		if (!empty($this->request->data)) {

			// Ensure the user is updating their own record only.
			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];

			if ($Model->saveAll($this->request->data)) {

				$this->Session->setFlash(array(
					'title' => "Account Updated",
					'description' => "Your account has been successfully updated!"
				), 'flash_success');
				return $this->redirect(array('action'=>'account'));

			} else {

				$this->Session->setFlash(array(
					'title' => 'Save failed',
					'description' => 'Failed to update your account'
				), 'flash_fail');

			}

		} else {

			$this->request->data = $Model->readForEdit($user[$Model->alias]['id']);

		}

		$this->_adminPopulateLookups();

		// unset again incase it was prefilled
		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'My Account');

		$this->view = 'EvCore./Users/admin_account';

		return;

	}


/**
 * Edit current user account.
 *
 * @return void
 */
	public function account() {

		$Model = $this->{$this->modelClass};

		$user = $this->Auth->user();

		$Model->validate = $Model->validateEdit();

		if (!empty($this->request->data)) {

			$this->request->data[$Model->alias]['id'] = $user[$Model->alias]['id'];

			if ($this->request->data[$Model->alias]['password'] == "") {

				unset($this->request->data[$Model->alias]['password']);
				unset($this->request->data[$Model->alias]['conform_password']);

			} else {

				$Model->validate = array_merge_recursive($Model->validate, $Model->validatePassword());

			}

			if ($Model->save($this->request->data)) {

				$this->Session->setFlash(__('Information updated'), 'flash_success');
				return $this->redirect(
					array(
						'action' => 'account'
					)
				);
			}

		} else {

			$this->request->data = $Model->findById($user[$Model->alias]['id']);

		}

		$this->request->data[$Model->alias]['password'] = "";
		$this->request->data[$Model->alias]['confirm_password'] = "";

		$this->set('title_for_layout', 'Update Account Information | '.Configure::read('SiteSetting.site_title'));

		return;

	}

}
