<?php

App::uses('Model', 'Model');
App::uses('Document', 'EvCore.Model');
App::uses('Image', 'EvCore.Model');

/**
 * EvCore model for Cake.
 *
 * All logic which is default to the Evoluted CMS is in here
 *
 * @package       app.Model
 */
class EvCoreAppModel extends Model {

	// Switch off automatic recursize queries app wide.
	public $recursive = -1;

	public $displayField = "name";

	// Name used in the admin theme for the model (override in your models or
	// leave as null to default to the model name).
	public $displayName = null;


/**
 * Define the number of images associated with the model.
 *
 * @var  integer|array
 *
 * It is now possible to have unlimited image slots
 * Image Slots
 * 		-1 = 	Unlimited Image Slots
 *		0  = 	No Image Slots
 *		X  = 	X Amount of Image Slots
 *
 * This can be done in 2 ways
 * 		1. If the model only has one set of images
 *			"public $imageSlots = 0;"
 *
 *		2. If there are multiple image areas to a page
 *			public $imageSlots = array(
 *				'main' => 4,
 *				'listing' => 1,
 *			);
 *
 *		public $imageSlots = array(
 *			'main' => array(
 *				'fields' => array(
 *					'caption' => array(
 *						'label' => 'Alt', //This array is passed through to FormHelper::input so can be used to  add class etc.
 *					),
 *					'button_text'
 *				),
 *				'slots' => -1
 *			)
 *		);
 *
 *
 */
	public $imageSlots = 0;


/**
 * Define the number of files/documents associated with the model.
 *
 * @see  AppModel::imageSlots
 * @var  integer|array
 */
	public $documentSlots = 0;


/**
 * Define the number of content blocks associated with the model.
 *
 * There are two ways of defining the blocks (like image slots):-
 *
 * 1. Using an integer when the blocks use all the block fields and
 *    all belong to the same set:
 *
 *    public $blocks = 2; // Two blocks per record
 *
 * 2. Using an array to define sets of blocks (and optionally their
 *    fields):
 *
 *    public $blocks = array(
 *        'main' => array(
 *            'fields' => array(
 *	                'name' => true,
 *					'body' => true
 *            ),
 *            'slots' => 1
 *        )
 *    );
 *
 * @var integer|array
 */
	public $blocks = 0;


	// Define the menu content in this model belong in when using the
	// Navigatable behaviour.
	public $parentMenu = null;


/**
 * Setup the model
 */
	public function __construct($id = false, $table = null, $ds = null) {

		parent::__construct($id, $table, $ds);

		if ($this->displayName == null) {
			$this->displayName = $this->alias;
		}

		$this->initialiseAssociateModels();

		return;

	}

/**
 * Initialises the associated models such as Image, Document and Block
 */
	public function initialiseAssociateModels() {

		// Set up model associations for images automagically based on the
		// image slots.
		$this->_associateModels('EvCore.Image', $this->imageSlots);

		// Set up model associations for documents automagically based on the
		// document slots.
		$this->_associateModels('EvCore.Document', $this->documentSlots);

		// Set up model associations for content blocks automagically.
		$this->_associateModels('EvCore.Block', $this->blocks, 'block_type');

	}


/**
 * Automagically associate attachments and blocks with the model based on the
 * number of slots defined. This method is called from the constructor.
 *
 * @param  string $modelAlias either 'Document', 'Image' or 'Block'
 * @param  mixed  $modelSlots  array or integer defining the number of slots required
 */
	protected function _associateModels($modelAlias, $modelSlots, $type = 'attachment_type') {

		list($plugin, $modelName) = pluginSplit($modelAlias);

		if (is_array($modelSlots)) {

			foreach ($modelSlots as $section => $slots) {

				if ($slots != 0) {
					$model =  ($section == 'main') ? $modelName : Inflector::camelize($section) . $modelName;
					$this->hasMany[$model] = array(
						'className' => $modelAlias,
						'foreignKey' => 'model_id',
						'conditions' => array(
							'model' => $this->name,
							$type => $model
						),
						'dependent' => true,
						'cascade' => true
					);
				}
			}

		} elseif ($modelSlots != 0) {

			$this->hasMany[$modelName] = array(
				'className' => $modelAlias,
				'foreignKey' => 'model_id',
				'conditions' => array(
					'model' => $this->name,
					$type => $modelName
				),
				'dependent' => true,
				'cascade' => true
			);

		}

		return;

	}


/**
 * Override in your on Model to customise
 */
	public function validate() {

		return $this->validate;

	}


/**
 * Query used to retrieve a record ready for edit
 *
 * @param integer $id ID of row to edit
 * @param array $query The db query array - can be used to pass in additional parameters such as contain
 * @return array
 */
	public function readForEdit($id, $query = array()) {

		$alias =  $this->alias;

		$query['conditions'][$alias.'.id'] = $id;

		if ($this->hasField('is_removed')) {

			$query['conditions']['is_removed'] = false;

		}

		$query = array_merge_recursive($query, $this->_containAssociateModel('Block', $this->blocks));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Image', $this->imageSlots));
		$query = array_merge_recursive($query, $this->_containAssociateModel('Document', $this->documentSlots));

		$data = $this->find('first', $query);

		return $data;

	}


	protected function _containAssociateModel($modelAlias, $modelSlots) {

		$query = array();

		if (is_array($modelSlots)) {
			foreach ($modelSlots as $section=>$slots) {
				if ($slots != 0) {
					$model = ($section == "main") ? $modelAlias : Inflector::camelize($section).$modelAlias;
					$query['contain'][$model] = array(
						'order' => 'sequence'
					);
				}
			}
		} elseif (is_numeric($modelSlots) && $modelSlots != 0) {
			$query['contain'][$modelAlias] = array(
				'order' => 'sequence'
			);
		}

		return $query;

	}


/**
 * Query used to retrieve a record ready for view.
 *
 * By default calls readForEdit but can be extended to bring out more
 * complicated data.
 *
 * @param integer $id ID of row to edit
 * @param array $params The db query array - can be used to pass in additional parameters such as contain
 * @return array
 */
	public function readForView($id, $params = array()) {

		if ($this->hasField('is_active')) {
			$params['conditions'][$this->escapeField('is_active')] = true;
		}

		return $this->readForEdit($id, $params);

	}


/**
 * Checks if the record is protected
 * Removes associated images and documents
 *
 * @param boolean $cascade true if the delete is to cascade to children
 * @return boolean true if the opperation was successful, otherwise false
 * @see Model::beforeDelete()
 */
	public function beforeDelete($cascade=true) {

		if ($this->hasField('is_protected') && $this->field('is_protected')) {

			return false;

		}

		return parent::beforeDelete($cascade);

	}


/**
 * Returns a list of dependant records (usually displayed on the delete dialog)
 *
 * @param integer $id ID of the row we are checking dependencies for
 */
	public function findDependents($id) {

		$dependents = array();

		$models = array_merge_recursive($this->hasOne, $this->hasMany);

		foreach($models as $model=>$attr)
		{
			//If model is set as not dependent then dont need to look in it for dependents
			if((isset($attr['dependent']) && !$attr['dependent']) || (isset($attr['cascade']) && $attr['cascade']))
			{
				continue;
			}

			$params = array();

			//If default conditions are set on the relationship we need to listen to them
			if(isset($attr['conditions']))
				$params['conditions'] = $attr['conditions'];

			$params['conditions'][$model . "." . $attr['foreignKey']] = $id;

			$found = $this->$model->find('all', $params);

			foreach($found as $item) {

				$value = $item[$model][$this->$model->displayField];
				$dependents[$model][] = $value;

			}

		}

		return $dependents;

	}


/**
 * Custom validate rule, checks if two fields are identical
 * (useful for comparing password and confirm_password)
 *
 * @param array $check Field to trigger the check
 * @param array $otherField Other field to compare with
 */
	public function checkMatches($check, $otherField) {

		$value = array_pop($check);

		return $value == $this->data[$this->alias][$otherField];

	}


/**
 * Checks if a combination of field values are unique in the DB
 *
 * @param $check array field the check is triggered by
 * @param $fields array list of fields that form part of the unique key
 *
 * @return boolean true if combination doesn't exists in the database, otherwise false
 */
	public function checkCombinationUnique($check, $fields) {

		$fieldToTest = key($check);
		$value = array_pop($check);

		// build up the list of fields and their values into a condition array
		$conditions = array(
			$this->alias . "." . $fieldToTest => $value
		);

		foreach($fields as $field) {

			$conditions[$this->alias . "." . $field] = $this->data[$this->alias][$field];

		}

		// don't include the current row in the duplicate check
		if (isset($this->data[$this->alias][$this->primaryKey]) && !empty($this->data[$this->alias][$this->primaryKey])) {

			$conditions[$this->alias . "." . $this->primaryKey . " <>"] = $this->data[$this->alias][$this->primaryKey];

		}

		// if we find a match, the combination already exists
		$found = $this->find('first', array(
			'conditions'=>$conditions
		));

		return empty($found);

	}


/**
 * Toggles a field on or off.
 *
 * @param string $field field to toggle
 * @param integer $id ID of row
 *
 * @return result of updateAll()
 */
	public function toggleField($field, $id=null) {

		$id = empty($id) ? $this->id : $id;

		$field = $this->escapeField($field);

		$fields = array($field => "NOT $field");
		$conditions = array($this->escapeField() => $id);

		return $this->updateAll($fields, $conditions);

	}


/**
 * Custom delete method that checks whether the current model has an is_removed flag
 * to mark as true rather than delete before actually deleting content
 * @param  int  $id
 * @param  boolean $cascade
 * @return boolean
 */
	public function delete($id = null, $cascade = true) {

		if ($this->hasField('is_removed')) {

			$this->id = $id;
			$this->saveField('is_removed', true);

			if ($this->hasField('is_active')) {

				$this->saveField('is_active', false);

			}

			$result = true;

		} else {

			$result = parent::delete($id, $cascade);

		}

		return $result;

	}


/**
 * Gets the next order number in a sequence.
 *
 * @param array $params
 * @return integer
 */
	public function getNextSequenceNo($params = array()) {

		$defaults = array(
			'order' => 'sequence DESC',
			'limit' => 1
		);

		$params = array_merge($params, $defaults);

		$result = $this->find('first', $params);

		return $result ? $result[$this->alias]['sequence'] + 1 : 0;
	}


/**
 * @todo: Implement an sql error log
 */
	public function sqlLog() {

	}

/**
 * @todo: Implement an sql error log
 */
 	public function getLastQuery() {
 		$log = $this->getDataSource()->getLog(false, false);
		// return the first element of the last array (i.e. the last query)
		return $log['log'][(count($log['log'])-1)];
 	}


/**
 * § Behaviours
 */

	/**
	 * Helper function to check if the model has a particular behaviour
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function hasBehaviour($behaviour) {

		if (empty($this->actsAs)) {
			return false;
		}

		$behaviourKeys = array_keys($this->actsAs);
		$behaviourValues = array_values($this->actsAs);

		return in_array($behaviour, array_merge($behaviourKeys, $behaviourValues), true);
	}


	/**
	 * Helper function to remove a behaviour from a model
	 *
	 * @param string $behaviour name of the behaviour
	 * @return boolean
	 */
	public function removeBehaviour($behaviour) {

		if ($this->hasBehaviour($behaviour)) {

			unset($this->actsAs[$behaviour]);

			$this->Behaviors->unload($behaviour);

		}

		return true;

	}


/**
 * Helper function to add a behaviour to a model
 *
 * @param string $behaviour name of the behaviour
 * @param array $config (optional) behaviour's config
 * @param boolean $overrideBehaviour (optional) pass as true to reload the behaviour if already loaded
 * @return boolean
 */
	public function addBehaviour($behaviour, $config = array(), $overrideBehaviour = false) {

		if ($overrideBehaviour===true || !$this->hasBehaviour($behaviour)) {

			$this->actsAs[$behaviour] = $config;

			$this->Behaviors->load($behaviour, $config);

		}

		return true;

	}


}
