<?php

App::uses('Controller', 'Controller');
App::uses('InflectorExt', 'Lib');
App::uses('ArrayUtil', 'EvCore.Lib');

/**
 * Evoluted Core Application Controller
 *
 * This is the main controller for the Evoluted CMS.
 * AppController extends from this controller to allow all methods and functionality to be accessible
 *
 * @package       app.Controller
 * @link http://book.cakephp.org/2.0/en/controllers.html#the-app-controller
 */
class EvCoreController extends Controller {

/**
 * Default components
 */
	public $components = array(
		'Auth' => array(
			'loginAction' => '/users/login',
			'authenticate' => array(
				'Form' => array(
					'fields' => array('username' => 'email'),
					'scope' => array('User.is_active' => true)
				)
			),
			'authorize' => 'Controller' // override isAuthorized() to customise
		),
		'Session',
		'SiteSettings.SiteSettings',
		'MetaData.Meta',
		'DebugKit.Toolbar'
	);


/**
 * Default helpers
 */
	public $helpers = array(
		'Html' => array(
			'className' => 'HtmlExt'
		),
		'Form' => array(
			'className' => 'FormExt'
		),
		'Session',
		'Number',
		'InflectorExt',
		'Image',
		'Listing',
		'Navigation.Navigation',
		'Routable.Route',
	);


/**
 * List of admin actions the controller supports
 * Used when you want to stop any administrator from performing an action
 * e.g. If you don't want an admin to delete something, override this in the controller and remove 'admin_delete'
 */
	public $adminActions = array(
		'admin_index',
		'admin_edit',
		'admin_delete',
		'admin_toggle',
		'admin_copy',
		'admin_view',
		'admin_reorder'
	);

	public $reordering = true;

/**
 * Paginator defaults for admin.
 *
 * @var array
 */
	public $adminPaginate = array(
		'limit' => 25
	);


/**
 * Ensures only administrators are allowed to access /admin areas
 *
 * Override in your controller to customise this behaviour
 *
 * @param mixed $user   User attempting to access the controller/action
 * @return boolean
 */
	public function isAuthorized($user) {

		if (!isset($user['User'])) {

			$user['User'] = $user;

		}

		if (isset($this->request->params['admin']) && $this->request->params['admin'] == true) {

			// Administrators and Super Users can access admin section, everyone else cannot.
			if (isset($user['User']['user_group_id']) && $user['User']['user_group_id'] <= 2) {

				return true;

			} else {

				$this->Session->setFlash('You are not authorized to access that location', 'flash_fail');

				$this->redirect(array('controller' => 'users', 'action' => 'admin_login', 'plugin' => false));

				return false;

			}

		} else {

			return true;

		}

	}


/**
 * Handles Theme selection and admin access
 *
 * @see Controller::beforeFilter()
 */
	public function beforeFilter() {

		// Set the theme based on prefix routing
		$this->theme = isset($this->params['admin']) ? 'Admin' : Configure::read('theme');

		if (isset($this->params['admin'])) {

			// if accessing the admin section, use the admin login screens
			$this->Auth->loginAction = "/admin/users/login";
			$this->Auth->loginRedirect = "/admin";

			// check the current admin action is permitted for this controller
			if (!in_array($this->action, $this->adminActions)) {

				$Model = $this->{$this->modelClass};
				$this->Session->setFlash('Action ' . $this->action . ' not supported for ' . InflectorExt::humanize($Model->displayName), 'flash_fail');
				return $this->redirect('/admin');

			}


		}

		// Load the Navigation menu model ready for building up site
		// navigation.
		$this->loadModel('Navigation.Menu');

		return;

	}


/**
 * Passes useful variables to the view
 *
 * @see Controller::beforeRender()
 */
	public function beforeRender() {

		// Model name
		$Model = $this->{$this->modelClass};

		$this->set('primaryModel', $Model->alias);

		$this->set('displayName', $Model->displayName);

		// Model primaryKey (check for existance as error pages don't have models)
		// Model displayField
		if (isset($Model->primaryKey)) {

			$this->set('primaryKey', $Model->primaryKey);
			$this->set('displayField', $Model->displayField);

		}

		// Logged in User
		AuthComponent::$sessionKey = 'Auth.User';
		$user = $this->Auth->user();
		$this->set('authUser', $user);

		// Set the primary navigation.
		if ($this->theme == 'Admin' && !empty($this->Menu)) {
			$this->set('primary_nav', $this->Menu->getMenu(1));
		} elseif (!empty($this->Menu)) {
			$this->set('primary_nav', $this->Menu->getMenu(7));
		} else {
			$this->set('primary_nav', array());
		}

		return;

	}


/**
 * Processes a limit passed for number of results to page by.
 *
 * @return void
 */
	protected function _processLimit() {

		$Model = $this->{$this->modelClass};

		$url = $this->request->params;

		if (!empty($this->request->data[$Model->alias]['results_per_page'])) {

			$url['named']['limit'] = $this->request->data[$Model->alias]['results_per_page'];
			foreach ($url['named'] as $key => $value) {
				$url[$key] = $value;
			}
			unset($url['named']);

			return $this->redirect($url);

		}

		if (!empty($this->request->params['named']['limit'])) {
			$this->request->data[$Model->alias]['results_per_page'] = $this->request->params['named']['limit'];
		} else {
			$this->request->data[$Model->alias]['results_per_page'] = $this->adminPaginate['limit'];
		}

		return;

	}


/**
 * Converts a filter form POST submission into a bookmarkable GET request
 * Converts a GET request into a conditions array suitable for filtering paginate
 *
 * @return array
 */
	protected function _processFilter($adminFilter = true) {

		if (isset($this->request->data['Filter'])) {

			// convert to get
			$url['action'] = $this->request->action;

			foreach ($this->request->data['Filter'] as $model=>$fields) {

				foreach ($fields as $field=>$value) {

					if ($value !== "") {

						$url['Filter'][$model][$field] = $value;

					}

				}

			}

			// redirect to bookmarkable get
			$this->redirect($url);

		}

		// build filter conditions
		$conditions = array();

		// only these fields will be concidered as part of the filter
		if ($adminFilter) {

			$filterFields = $this->_adminFilterFields();
		} else {

			$filterFields = $this->_frontFilterFields();
		}

		// Set any defined default filters.
		foreach ($filterFields as $field => $filter) {

			if (isset($filter['default']) && !empty($filter['default'])) {

				$conditions[$field] = $filter['default'];

			}

		}

		if (isset($this->request->params['named']['Filter'])) {
			foreach ($this->request->params['named']['Filter'] as $model=>$fields) {
				foreach ($fields as $field=>$value) {

					if (is_array($value)) {
						$value = $value['year'] . "-" . $value['month'] . "-" . $value['day'];
					}

					if (isset($filterFields[$model . "." . $field])) {
						$key = $filterFields[$model . "." . $field]['compare'];
						$isDate = $filterFields[$model . "." . $field]['type'] == 'date';
						$isNull = (isset($filterFields[$model.".".$field]['is_null']) && $filterFields[$model.".".$field]['isnull'] = true) ?  true : false;

						//If there are multiple fields to compare join them together with ORs
						if (count($key) > 1) {
							$or_array = array();
							foreach ($key as $fieldi => $condition) {
								$or_array[$fieldi] = sprintf($condition, $value);
							}
							$conditions[]['OR'] = $or_array;
						} else {
							if ($isDate) {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column." >="] = $value. " 00:00:00";
								$conditions[$column." <="] = $value. " 24:00:00";

							} elseif ($isNull) {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								if ($value == 1) {
									$conditions[] = $column." IS NULL";
								} else {
									$conditions[] = $column." IS NOT NULL";
								}

							} elseif($filterFields[$model . "." . $field]['type'] == 'daterange_from') {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column] = $value. " 00:00:00";

							} elseif($filterFields[$model . "." . $field]['type'] == 'daterange_to') {

								$column = key($key);
								$value = sprintf(array_pop($key), $value);
								$conditions[$column] = $value. " 24:00:00";

							} else {

								$conditions[key($key)] = sprintf(array_pop($key), $value);
							}
						}

						$this->request->data['Filter'][$model][$field] = $value;

					}
				}
			}
		}

		// If the model has an is_removed column then we want to ensure removed
		// rows are filtered out of the admin_index listings
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if ($Model->hasField('is_removed')) {
			$conditions[$modelAlias. '.is_removed'] = false;
		}

		return $conditions;

	}


/**
 * Does a recursive str_replace on the $compare array, replacing instances of %s with $value
 * in both the keys and values
 *
 * @param array $compare
 * @param string $value
 * @return array
 */
	protected function _buildFilterCompare($compare, $value) {

		$key = key($compare);

		if (in_array(strtolower($key), array('or', 'and'))) {

			$compare[$key] = $this->_buildFilterCompare($compare[$key], $value);

		}
		else {

			$result = array();

			foreach($compare as $compareKey=>$compareValue) {

				$compareKey = str_replace('%s', $value, $compareKey);
				$compareValue = str_replace('%s', $value, $compareValue);

				$result[$compareKey] = $compareValue;

			}

			$compare = $result;

		}

		return $compare;

	}


/**
 * Defines the toolbar buttons displayed in admin_index
 *
 * @return array
 */
	protected function _adminIndexToolbar() {

		$toolbar =  array(
			'Add New'=>array(
				'url'=>array('action'=>'add')
			)
		);

		if ($this->reordering) {

			$show_reordering = false;
			$model_columns = $this->{$this->modelClass}->getColumnTypes();

			if (
				array_key_exists('lft', $model_columns) ||
				array_key_exists('orderno', $model_columns) ||
				array_key_exists('sequence', $model_columns)
			) {

				$show_reordering = true;

			}

			if ($show_reordering) {

				$toolbar['Reorder'] = array(
					'url' => array('action' => 'reorder'),
					'class' => 'toolbar__btn--reorder'
				);

			}

		}


		return $toolbar;

	}


/**
 * Pagination settings for admin_index
 *
 * Override in your own controller to customise
 *
 * @return array
 */
	protected function _adminIndexPaginate() {

		$Model = $this->{$this->modelClass};
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions
		);

		// Include defaults.
		$paginate = array_merge_recursive($this->adminPaginate, $paginate);

		$model_columns = $Model->getColumnTypes();

		if (array_key_exists('orderno', $model_columns)) {

			$paginate['order'] = $Model->alias . '.orderno ASC';

		} elseif (array_key_exists('sequence', $model_columns)) {

			$paginate['order'] = $Model->alias . '.sequence ASC';

		} elseif (array_key_exists('lft', $model_columns)) {

			$paginate['order'] = $Model->alias . '.lft ASC';

		}

		return $paginate;


	}


/**
 * Defines which actions are available to each row of the admin_index table results
 *
 * Override in your controller to customise
 * Customise the handlers for these actions in /view/Admin/Elements/index_results.ctp
 *
 * @return array
 */
	protected function _adminIndexActions() {

		$actions = array(
			'Delete',
			'Edit',
			//'Admin View'
		);

		if (method_exists($this, 'view')) {
			$actions[] = 'View';
		}

		return $actions;

	}


/**
 * Defines the fields displayed in a filter form.
 *
 * Defaults to id, display name, is_active, created and modified fields.
 *
 * Override in your controller to customise.
 *
 * Format:
 *
 * 'id' => array(
 *         'label' => label text,
 *         'type' => data type of control
 *         'compare' => SQL query. e.g. Model.field='%s' (where %s is replace by the value of the field)
 *         'default' => value you want the data filtering by by default
 * )
 */
	protected function _adminFilterFields() {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = array(
			"$modelAlias.id" => array(
				'label' => 'ID',
				'type' => 'integer',
				'compare' => array("$modelAlias.id" => "%s")
			)
		);

		if ($Model->displayField) {

			$fields[$modelAlias. '.' . $Model->displayField] = array(
				'label' => InflectorExt::humanize($Model->displayField),
				'type' => 'string',
				// We use double '%' in the comparison as we don't want them
				// treating as literal %.
				'compare' => array($modelAlias. '.' . $Model->displayField . ' LIKE' => "%%%s%%")
			);

		}

		if ($Model->hasField('is_active')) {

			$fields[$modelAlias. '.is_active'] = array(
				'label' => 'Active',
				'type' => 'boolean',
				'compare' => array("$modelAlias.is_active" => "%s")
			);

		}

		if ($Model->hasField('created')) {

			$fields[$modelAlias. '.created'] = array(
				'label' => 'Created',
				'type' => 'date',
				'compare' => array("$modelAlias.created" => "%s")
			);

		}

		if ($Model->hasField('modified')) {

			$fields[$modelAlias. '.modified'] = array(
				'label' => 'Modified',
				'type' => 'date',
				'compare' => array("$modelAlias.modified" => "%s")
			);
		}


		return $fields;

	}


/**
 * Stub function
 * Used to populate form drop down selects
 *
 * Override in your controller to customise
 */
	protected function _adminPopulateLookups() {

	}


/**
 * Stub function
 * Used to populate filter form drop down selects
 *
 * Override in your controller to customise
 */
	protected function _adminPopulateFilterLookups() {

		$this->_adminPopulateLookups();

	}


/**
 * Defines the columns displayed in the admin_index results table
 * using a columns whitelist.
 *
 * Override in your controller to customise
 */
	protected function _adminIndexColumns() {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = array();

		$columnsWhitelist = $this->_adminIndexColumnsWhitelist();

		foreach ($Model->schema() as $field => $attr) {

			if (in_array("$modelAlias.$field", $columnsWhitelist)) {

				$attr['label'] = InflectorExt::humanize($field);
				$attr['label'] = str_replace('Is ', '', $attr['label']);
				$columns[$modelAlias. "." . $field] = $attr;

			}

		}

		return $columns;

	}


/**
 * Defines a whitelist of columns to be displayed on the admin index.
 * @return array whitelisted columns
 */
	protected function _adminIndexColumnsWhitelist() {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$columns = array();

		$columns[] = "$modelAlias.id";

		if ($Model->displayField) {
			$columns[] = "$modelAlias.{$Model->displayField}";
		}

		if ($Model->hasField('is_active')) {
			$columns[] = "$modelAlias.is_active";
		}
		if ($Model->hasField('created')) {
			$columns[] = "$modelAlias.created";
		}
		if ($Model->hasField('modified')) {
			$columns[] = "$modelAlias.modified";
		}

		return $columns;

	}


/**
 * Defines the fields displayed in an admin_form for this model
 *
 * Defaults to all fields in the db table
 *
 * Override in your controller to customise
 */
	protected function _adminFormFields() {

		$Model = $this->{$this->modelClass};

	  	$data = $Model->schema();

		$fields = array();

		foreach ($data as $key => $value) {
			$fields["{$Model->alias}.$key"] = $value;
		}


		// If the model is routable we want to include the route fields
		if ($Model->hasBehaviour('Routable.Routable')) {

			if(isset($fields["{$Model->alias}.name"])) {

				$insertRoutesAfter = "{$Model->alias}.name";

			} elseif(isset($fields["{$Model->alias}.title"])) {

				$insertRoutesAfter = "{$Model->alias}.title";

			}

			$routeFields = array(
				'Route.id' => array(
					'type' => 'hidden'
				),
				'Route.alias' => array(
					'type' => 'route-alias',
					'length' => 128
				)
			);

			if(isset($insertRoutesAfter)) {

				$fields = ArrayUtil::addAfter($fields, $insertRoutesAfter, $routeFields);

			} else {

				$fields += $routeFields;

			}

		}

		// is_removed columns shouldn't be exposed to the user
		unset($fields[$Model->alias.'.is_removed']);

		unset($fields[$Model->alias.'.lft']);
		unset($fields[$Model->alias.'.rght']);

		return $fields;

	}


/**
 * Defines the block fields displayed in an admin_form for this model.
 *
 * Override in your controller to customise.
 *
 * @return array
 */
	protected function _adminFormBlockFields() {


		$Model = $this->{$this->modelClass};

		$blocks = array();

		if (!empty($Model->blocks)) {

			if (is_numeric($Model->blocks)) {
				$blockSections = array(
					'main' => array(
						'slots' => $Model->blocks
					)
				);
			} else {
				$blockSections = $Model->blocks;
			}

			foreach ($blockSections as $section => $slots) {

				$modelAlias = $section=='main' ? 'Block' : Inflector::camelize($section) . 'Block';

				// Build up the fields for the blocks using the Block model's schema
				$fields = $Model->$modelAlias->schema();
				unset($fields['created']);
				unset($fields['modified']);
				unset($fields['is_active']);

				for ($block = 0; $block < $slots['slots']; $block++) {

					foreach ($fields as $fieldKey => $fieldValues) {

						if (!isset($slots['fields']) || isset($slots['fields'][$fieldKey])) {
							$blocks["$modelAlias.$block.$fieldKey"] = $fieldValues;
							$label = Inflector::humanize($section);
							$label .= $slots['slots'] > 1 ? ' ' . ($block+1) : null;
							$label .= ': ' . Inflector::humanize(Inflector::underscore($fieldKey));
							$blocks["$modelAlias.$block.$fieldKey"]['label'] = $label;

							if (!empty($slots['fields'][$fieldKey]['type'])) {
								$blocks["$modelAlias.$block.$fieldKey"]['type'] = $slots['fields'][$fieldKey]['type'];
							}
						}

					}

					$blocks["$modelAlias.$block.id"]['type'] = 'hidden';

					// Hide the association fields and make sure the model is getting set.
					$blocks["$modelAlias.$block.block_type"]['type'] = 'hidden';
					$blocks["$modelAlias.$block.block_type"]['value'] = $modelAlias;
					$blocks["$modelAlias.$block.model"]['type'] = 'hidden';
					$blocks["$modelAlias.$block.model"]['value'] = $Model->alias;
					$blocks["$modelAlias.$block.model_id"]['type'] = 'hidden';
					$blocks["$modelAlias.$block.plugin"]['type'] = 'hidden';
					$blocks["$modelAlias.$block.sequence"]['type'] = 'hidden';

				}

			}

		}

		return $blocks;

	}


/**
 * Defines the meta data fields in an admin_form for this model
 * when it has the Meta behaviour.
 */
	protected function _adminFormMetaFields() {

		$Model = $this->{$this->modelClass};

		$fields = array();

		if ($Model->hasBehaviour('MetaData.Meta')) {


			$fields['MetaData.id'] = array(
				'type' => 'hidden'
			);

			$fields['MetaData.title'] = array(
				'type' => 'string',
				'length' => 155
			);

			$fields['MetaData.description'] = array(
				'type' => 'string',
				'length' => 155
			);

		}

		return $fields;

	}


/**
 * Defines the menu fields in an admin_form for this model
 * when it has the Navigatable behaviour.
 *
 * @param  integer $id ID of the current menu to exclude
 * @return array
 */
	protected function _adminFormMenuFields($id) {

		$Model = $this->{$this->modelClass};

		$fields = array();

		if ($Model->hasBehaviour('Navigation.Navigatable')) {

			$menu = $this->Menu->findTreeBranches($Model->parentMenu, $id);

			$fields['Menu.is_active'] = array(
				'type' => 'boolean',
				'label' => 'Add to menu'
			);

			$fields['Menu.id'] = array(
				'type' => 'hidden'
			);

			$fields['Menu.name'] = array(
				'type' => 'string',
				'length' => 45
			);

			$fields['Menu.parent_id'] = array(
				'type' => 'integer',
				'options' => $menu
			);

		}

		return $fields;

	}


/**
 * Defines the buttons in the toolbar displayed on an admin_form
 *
 * Override in your controller to customise
 *
 * @param integer $id
 * @return array
 */
	protected function _adminFormToolbar($id = null) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$actions = array(
			'Add New' => array(
				'url' => array('action' => 'add')
			)
		);

		$data = null;

		if ($id) {

			$data = $Model->findById($id);

		}

		if (!isset($data[$modelAlias]['is_protected']) || !$data[$modelAlias]['is_protected']) {

			if ($Model->hasBehaviour('EvCore.CustomCopyable')) {

				$actions['Copy'] = array(
					'url' => array('action' => 'copy', $id),
					'class' => 'toolbar__btn--copy'
				);

			}

			$actions['Delete'] = array(
				'url' => array('action' => 'delete', $id),
				'class' => 'toolbar__btn--delete'
			);

		}

		return $actions;

	}


/**
 * Defines the fields displayed in an admin_form for this model
 *
 * Defaults to all fields in the db table
 *
 * Override in your controller to customise
 *
 * @return array
 */
	protected function _adminViewFields() {

		return $this->_adminFormFields();
	}


/**
 * Defines the buttons in the toolbar displayed on an admin_form
 *
 * Override in your controller to customise
 *
 * @param integer $id
 * @return array
 */
	protected function _adminViewToolbar($id = null) {

		return $this->_adminFormToolbar($id);
	}


/**
 * Defines the toolbar buttons displayed in admin_reorder
 *
 * @return array
 */
	protected function _adminReorderToolbar() {

		$toolbar = array(
			'Listing' => array(
				'url' => array(
					'action' => 'index'
				),
				'class' => 'toolbar__btn--listing'
			),
			'Add New' => array(
				'url' => array(
					'action' => 'add'
				)
			)
		);

		return $toolbar;
	}


/**
 * Manages a default admin index view of this model
 * Includes a filter form and paginated results
 *
 * Override in your controller to customise
 *
 * @return void
 */
	public function admin_index() {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$this->_processLimit();
		$this->_adminPopulateFilterLookups();
		$this->set('filter', $this->_adminFilterFields());
		$this->set('columns', $this->_adminIndexColumns());
		$this->set('toolbar', $this->_adminIndexToolbar());
		$this->set('actions', $this->_adminIndexActions());

		$this->paginate = $this->_adminIndexPaginate();
		$this->set('data', $this->paginate());

		$this->set('title_for_layout', InflectorExt::pluralize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_index';

		return;

	}


/**
 * Manages an admin edit form for this model
 *
 * @param integer $id ID of row to edit
 * @return void
 */
	public function admin_edit($id = null) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {

			$this->request->data = $this->_processImages($this->request->data);

			if ($Model->saveAll($this->request->data, array('deep' => true))) {

				$action = ($id===null) ? 'created' : 'updated';
				$title = empty($this->data[$modelAlias][$Model->displayField]) ? InflectorExt::humanize($Model->displayName) : '&lsquo;' . $this->data[$modelAlias][$Model->displayField] . '&rsquo;';

				$this->Session->setFlash(array(
					'title' => InflectorExt::humanize($Model->displayName) . " $action",
					'description' => "$title has been successfully $action!"
				), 'flash_success');

				if (isset($this->request->data['return']) && $this->request->data['return'] == 1) {

					$this->Session->write("Admin.$modelAlias.return", true);

					$return = array('action' => 'edit', $Model->id);

				} else {

					$this->Session->delete("Admin.$modelAlias.return");

					$redirect = $this->Session->read("Admin.$modelAlias.filter");
					if (strpos($redirect, 'Filter') !== false) {
						$return = $redirect;
					}
					else {
						$return = array('action' => 'index');
					}
				}

				return $this->redirect($return);

			} else {

				$this->Session->setFlash(array(
					'title' => 'Save failed',
					'description' => 'Failed to save ' . InflectorExt::humanize($Model->displayName) . ', see below for details'
				), 'flash_fail');

			}

		} elseif ($id !== null) {

			$this->request->data = $Model->readForEdit($id);

			if (!$this->request->data) {

				throw new NotFoundException();

			}
			$this->Session->write("Admin.$modelAlias.filter", $this->referer());
		}

		$actionLabel = ($id===null) ? 'Add' : 'Edit';

		$this->set('title_for_layout', "$actionLabel " . InflectorExt::humanize($Model->displayName));
		$this->set('fields', $this->_adminFormFields());
		$this->set('toolbar', $this->_adminFormToolbar($id));
		$this->set('blocks', $this->_adminFormBlockFields());
		$this->set('imageSlots', $Model->imageSlots);
		$this->set('documentSlots', $Model->documentSlots);
		$this->set('menuFields', $this->_adminFormMenuFields(isset($this->request->data['Menu']['id']) ? $this->request->data['Menu']['id'] : null));
		$this->set('metaData', $this->_adminFormMetaFields());
		$this->_adminPopulateLookups();

		$this->set('return', $this->Session->read("Admin.$modelAlias.return"));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_form';

		return;

	}


/**
 * View data for this model
 *
 * @param integer $id ID of row to edit
 * @return void
 */
	public function admin_view($id) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$this->request->data = $Model->readForEdit($id);

		if (!$this->request->data) {

			throw new NotFoundException();

		}

		$actionLabel = 'View';

		$this->set('title_for_layout', "$actionLabel " . InflectorExt::humanize($Model->displayName));
		$this->set('fields', $this->_adminViewFields());
		$this->set('toolbar', $this->_adminViewToolbar($id));
		$this->set('imageSlots', $Model->imageSlots);
		$this->set('documentSlots', $Model->documentSlots);
		$this->_adminPopulateLookups();

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_view';

		return;

	}


/**
 * Manages an admin delete for this model
 *
 * Override in your controller to customise
 *
 * @todo: check $Model->cascade - if true we don't need to check for
 * dependants as they will be automatically deleted when the parent is
 * deleted
 *
 * @param integer $id ID of row to delete
 * @return void
 */
	public function admin_delete($id) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {

			$Model->delete($id);

			$this->Session->setFlash(array(
				'title' => InflectorExt::humanize($Model->displayName) . ' deleted',
				'description' => InflectorExt::humanize($Model->displayName) . ' has been successfully deleted!'
			), 'flash_success');
			return $this->redirect(array('action'=>'index'));

		} else {

			$this->Session->write('delete_referrer', $this->request->referrer);

		}

		$this->request->data = $Model->findById($id);

		$this->set('dependants', $Model->findDependents($id));

		$this->set('title_for_layout', 'Delete ' . InflectorExt::humanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_delete';

		return;

	}


/**
 * Copies an item
 *
 * @param integer $id id of item to copy
 * @return void
 */
	public function admin_copy($id) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		if (!empty($this->request->data)) {

			if ($Model->copy($id)) {

				$newId = $Model->getInsertID();

				$this->Session->setFlash(array(
					'title' => InflectorExt::humanize($Model->displayName) . " copied",
					'description' => 'Item has been successfully copied!'
				), 'flash_success');

				return $this->redirect(array('action' => 'edit', $newId));

			} else {

				$this->Session->setFlash(array(
					'title' => 'Copy failed',
					'description' => 'Failed to copy ' . InflectorExt::humanize($Model->displayName)
				), 'flash_fail');

				return $this->redirect(array('action' => 'edit', $id));

			}

		}

		$this->request->data = $Model->findById($id);

		if (isset($this->request->data[$modelAlias]['is_protected']) && $this->request->data[$modelAlias]['is_protected']) {

			return $this->redirect(array('action' => 'index'));

		}

		$this->set('title_for_layout', 'Copy ' . InflectorExt::humanize($Model->displayName));

		// Set the default template, this can be overridden in controllers using $this->view
		// or $this->render().
		$this->view = '../Scaffolds/admin_copy';

		return;

	}


/**
 * Toggles a field
 *
 * @param string $field Field to toggle
 * @param integer $id id of row to toggle
 * @return void
 */
	public function admin_toggle($field, $id) {

		$Model = $this->{$this->modelClass};

		// Cleanse the data before updating the database
		$id = intval($id);

		$Model->toggleField($field, $id);

		return $this->redirect($this->referer());

	}


/**
 * Admin reordering by Ollie ┗(＾0＾)┓
 * Put public max_depth in the controller to control how many levels it goes down
 * 0 = Unlimited
 *
 * @return void
 */
	public function admin_reorder() {

		$Model = $this->{$this->modelClass};

		//Find out which key to use for ordering...
		$colomns = $Model->getColumnTypes();

		if (array_key_exists('orderno', $colomns)) {

			$order_key = 'orderno';

		} else if (array_key_exists('sequence', $colomns)) {

			$order_key = 'sequence';

		} else if (array_key_exists('lft', $colomns)) {

			$order_key = 'lft';

		} else {

			//Fail I need an ordering column
			$this->Session->setFlash(array(
				'title' => 'Reordering Failed',
				'description' => 'Unable to find an ordering column on the table'
			), 'flash_fail');
			$this->redirect('/admin');

		}

		//See if its a multi-level dealy
		$hasMultiLevel = false;
		if (array_key_exists('parent_id', $colomns)) {

			$hasMultiLevel = true;

		} else {

			$this->max_depth = 1;

		}

		//Save data if post
		if ($this->request->data) {

			if ($Model->hasBehaviour('Routable.Routable')) {

				$Model->Behaviors->unload('Routable.Routable');
			}

			unset($this->request->data['log']);
			$Model->validate = array();

			foreach($this->request->data['reorder'] as &$data) {

				if (empty($data['parent_id'])) {

					$data['parent_id'] = null;

				}

			}

			$Model->removeBehaviour('Tree');

			if (isset($this->request->data['reorder']) && $Model->saveAll($this->request->data['reorder'], array('callbacks' => false))) {

				$this->Session->setFlash(array(
					'title' => 'Order Saved',
					'description' => 'Reordering has been saved successfully!'
				), 'flash_success');

			} else {

				$this->Session->setFlash(array(
					'title' => 'Reordering Failed',
					'description' => 'There was an error saving the data'
				), 'flash_fail');

			}

			$Model->addBehaviour('Tree');
			$this->redirect(array('action'=>'index'));
			exit;

		}

		//Use only ID and the display field
		$columns = array($Model->alias.'.id' => '', $Model->alias  .'.' . $Model->displayField  => '');
		$this->set('columns', $columns);

		//If its multi-level recursion!
		if ($hasMultiLevel) {

			if ($order_key == 'lft') {

				$data = $Model->find('threaded', array(
					'order' => $order_key
				));

			} else {

				$data = $Model->find('all', array(
					'conditions' => array(
						'OR' => array(
							'parent_id IS NULL',
							'parent_id' => 0
						)
					),
					'order' => $order_key
				));
				$this->_getChildren($data, $Model->alias, $order_key);

			}

		} else {
			//No fun.
			$data = $Model->find('all', array(
				'order' => $order_key
			));
		}

		//Lets javascript know whats going on too
		$this->set('order_key', $order_key);
		$this->set('max_depth', $this->max_depth);
		$this->set('toolbar', $this->_adminReorderToolbar());
		$this->set('data', $data);

		$this->view = '../Scaffolds/admin_reorder';

		return;

	}

/**
 * Recursive Finding of children - Takes array by reference
 * @param data array by reference
 * @param model name string
 * @param order key string
 * @return void
 */
	protected function _getChildren(&$data, $Model, $order_key) {

		foreach ($data as &$d) {

			$children = $this->$Model->find('all', array(
				'conditions' => array(
					'parent_id' => $d[$Model]['id']
				),
				'order' => $order_key

			));
			$this->_getChildren($children, $Model, $order_key);

			$d[$Model]['children'] = $children;
		}

		return;

	}


/**
 * Toggles a field via an AJAX call
 *
 * @param string $field Field to toggle
 * @param integer $id id of row to toggle
 * @return boolean
 */
	public function ajax_toggle($field, $id) {

		$Model = $this->{$this->modelClass};

		// Cleanse the data before updating the database
		$id = intval($id);

		$this->render('../../Admin/Scaffolds/ajax_toggle', 'ajax');

		return ($id) ? $Model->toggleField($field, $id) : false;

	}


/**
 * Remove any image slots that are not been used in an admin form
 *
 * @param $this->request->data
 * @return $this->request->data with surplus image slots removed
 */
	protected function _processImages($data) {

		$rtnArray = $data;

		foreach ($data as $key=>$value) {
			if ($key == 'Image' && isset($data['Image'])) {
				foreach ($data['Image'] as $keyi=>$image) {
					if (empty($image['id']) && (empty($image['Image']['filename']['name']) && empty($image['filename']['name']))) {
						unset($rtnArray['Image'][$keyi]);
					}
				}
			} else {
				if (is_array($value)) {
					$rtnArray[$key] = $this->_processImages($value);
				} else {
					$rtnArray[$key] = $value;
				}
			}

		}

		return $rtnArray;

	}


/**
 * Returns true if a form submission contains a defined Form.process
 *
 * Used to identify which form to process on pages with multiple forms (for
 * example a product page with an add to basket and review forms).
 *
 * @param string $process
 * @return boolean
 */
	public function checkProcessForm($process) {

		return !empty($this->request->data['Form']['process']) && $this->request->data['Form']['process']===$process;

	}


	/**
	 * add page data to a page
	 * Used for none page screens that need page titles / meta functionality
	 * such as module listing pages
	 *
	 * @param 	string|int 		page constant to load or page ID
	 * @return 	bool
	 */
	public function assignPage($page, $tpl_var='data')
	{
		$this->loadModel('Page');

		if (! is_int($page)) {

			$page = constant('Page::' . $page);
		}

		$data = $this->Page->readForView($page);

		if (! $data) {
			return false;
		}

		$this->set($tpl_var, $data);
		$this->Meta->set($data, 'Page');

		return true;
	}

}
