<?php

App::uses('FieldFormatter', 'EvContactForm.Lib');

class ShowContactFormHelper extends AppHelper {

	protected $_name;
/**
 * Display a contact form on the page using the variables set in the controller. Need to use the same form name
 * as the one used to inject the form.
 * @param  string $name      Name of the form to display.
 * @param  array  $extraVars extra variables to pass to the contact form template
 * @return string            A HTML string to display the contact form
 */
	public function display($name, $extraVars = array()) {
		$this->_name = $name;
		$template = 'EvContactForm.contact_form';
		$formVars = $this->_View->viewVars[$name . '_form'];
		if (!empty($formVars['template'])) {
			$template = $formVars['template'];
		}

		if (!empty($formVars['use_field_sections']) && !empty($formVars['fields'])) {
			$formVars['fields'] = $this->_sectionAndGroupFields($formVars['fields'], $useFieldGroups = !empty($formVars['use_field_groups']));
		}

		return $this->_View->Element($template,
			array_merge_recursive($extraVars,
			array(
				'form' => $formVars,
				'name' => $name,
			))
		);
	}

/**
 * Display a contact form field using the config settings. Used on contact form templates.
 * @param  string $fieldName       The name to be given to the field.
 * @param  array  $fieldAttributes The field attributes to be given to the field
 * @param  bool   $useInlineErrors Whether the form is displaying errors inline or not.
 * @return string                  A HTML string to display the field.
 */
	public function displayField($fieldName, $fieldAttributes = [], $useInlineErrors = false) {
		$fieldHtml = '';

		// reset unwanted $fieldAttributes
		unset($fieldAttributes['validate']);
		unset($fieldAttributes['admin-label']);
		unset($fieldAttributes['admin-type']);

		// assign default values for the loop
		$fieldDiv = false;
		$fieldError = false;

		if (
			! empty($this->_View->validationErrors['ContactFormResponse'][$fieldName]) &&
			$useInlineErrors === true
		):
			$fieldError = $this->_View->Form->error('ContactFormResponse.' . $fieldName);
			$fieldAttributes['afterInput'] = $fieldError;
		endif;

		// extract the div class so error messages can be echo after the input
		// when the field has failed validation
		if (! empty($fieldAttributes['div'])):
			$fieldDiv = $fieldAttributes['div'];
			$fieldAttributes['div'] = false;

			// assign a generic error class when the field has errored on save
			if (is_array($fieldError) && ! empty($fieldError)):
				$fieldDiv .= ' error';
			endif;

			$fieldHtml .= $this->_View->Html->tag('div', null, ['class' => $fieldDiv]);
		endif;

		if ($fieldAttributes['type'] == 'file') {
			$fieldHtml .= $this->_View->Form->input('FilesDocument.' . $fieldName . '.model', [
				'type' => 'hidden',
				'value' => 'ContactFormResponse'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.' . $fieldName . '.attachment_type', [
				'type' => 'hidden',
				'value' => 'FilesDocument'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.' . $fieldName . '.id', [
				'type' => 'hidden',
				'value' => '0'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.' . $fieldName . '.filename', $fieldAttributes);

		} else {

			if (!empty($this->_View->viewVars[$this->_name . '_submitted_data'][$fieldName])) {
				$fieldAttributes['value'] = $this->_View->viewVars[$this->_name . '_submitted_data'][$fieldName];
			}

			$fieldHtml .= $this->_View->Form->input($fieldName, $fieldAttributes);
		}

		if (! empty($fieldDiv)):
			$fieldHtml .= $this->_View->Html->tag('/div');
		endif;

		return $fieldHtml;
	}

/**
 * Display a field to catch spam. Used on contact form templates
 * @return string A HTML string to display the spam field.
 */
	public function displaySpamField() {
		$spamHtml = '';

		$spamHtml .= $this->_View->Form->input('email_confirm', array(
			'type' => 'text',
			'div' => array(
				'style' => 'display: none;'
			)
		));

		return $spamHtml;
	}

/**
 * Display a submit button. Used on contact form templates
 * @param  string $formName           The name of the form to attach the button to. Used if no button text has been provided
 * @param  array  $buttonAttributes   The field attributes to be given to the button
 * @return string                     A HTML string to display the submit button.
 */
	public function displaySubmit($formName, $buttonAttributes = []) {
		$submitHtml = '';

		if (!empty($buttonAttributes['button_text'])):
			$inputName = $buttonAttributes['button_text'];
			unset($buttonAttributes['button_text']);
		else:
			$inputName = $formName . '.save';
		endif;

		if (!isset($buttonAttributes['type'])) {
			$buttonAttributes['type'] = 'submit';
		}

		$submitHtml .= $this->_View->Form->input($inputName, $buttonAttributes);

		return $submitHtml;
	}

/**
 * Check if a field can be displayed on a notification email. The target should be either 'admin' or 'user'. A field
 * can't be displayed if the data is missing or the field is for a specific target but does not match the current
 * target.
 *
 * @param string $fieldName The name of the field to display.
 * @param array  $field     Field attributes.
 * @param array  $formData  Submitted form data.
 * @param string $target    The target of the email. 'admin' or 'user'.
 * @return bool True if the field can be displayed, false otherwise.
 */
	public function canDisplayNotificationField($fieldName, $field, $formData, $target) {
		return isset($formData[$fieldName])
			&& (
				(empty($field['user_only']) && empty($field['admin_only'])) //Field isn't limited to target
				|| (!empty($field['user_only']) && $target == 'user') //Field is limited to user target and matches
				|| (!empty($field['admin_only']) && $target == 'admin') //Field is limited to admin target and matches
			);
	}

/**
 * Display a label for a field to be used on the notification emails.
 *
 * @param string $fieldName The name of the field being displayed.
 * @param array  $field     Field attributes.
 * @param string $target    The target of the email. 'admin' or 'user'.
 * @return string Label text.
 */
	public function displayNotificationLabel($fieldName, $field, $target) {
		if (!empty($field['admin-label']) && $target == 'admin') {
			return $field['admin-label'];
		}

		if (!empty($field['label'])) {
			if (is_array($field['label']) && !empty($field['label']['text'])) {
				return $field['label']['text'];
			}

			return $field['label'];
		}

		return Inflector::humanize($fieldName);
	}

/**
 * Collate the contact form fields into sections and groups if the form config has them enabled and fields have the
 * correct attribute set.
 *
 * @param array $fields         Current contact form fields.
 * @param bool  $useFieldGroups True if the fields should be group inside of sections.
 * @return array Modified fields array.
 */
	protected function _sectionAndGroupFields($fields, $useFieldGroups = false) {
		$FieldFormatter = new FieldFormatter();
		return $FieldFormatter->sectionAndGroupFields($fields, $useFieldGroups);
	}
}
