<?php

class ResponseFormatter {

	/*
	 * Set to true to escape the value before it is returned. Set to false to not escape or to make sure it isn't
	 * double escaped.
	 */
	private $__escapeValue = false;

/**
 * Format a value on a contact form response based on it's field attributes.
 *
 * @param string $fieldName       The name of the field being formatted.
 * @param array  $fieldAttributes The attributes of the field as defined in the config.
 * @param mixed  $fieldValue      The value to format.
 * @param bool   $escape          True to escape the formatted value. Defaults to true.
 * @return mixed The formatted contact form response value.
 */
	public function format($fieldName, $fieldAttributes, $fieldValue, $escape = true) {
		$formattedValue = null;

		if (isset($fieldAttributes['escape'])) {
			$escape = $fieldAttributes['escape'];
		}

		$this->__setEscape($escape);

		if (is_array($fieldValue)) {
			foreach ($fieldValue as $value) {
				if ($escape) {
					$value = h($value);
				}

				$formattedValue .= $value . '<br/>';
			}

			//This can't be escaped because it contains HTML markup.
			return $formattedValue;
		}

		if (!empty($fieldAttributes['type'])) {
			$formattedValue = $this->formatOnFieldType($fieldValue, $fieldAttributes['type']);
		}

		if ($this->__escapeValue) {
			$formattedValue = h($formattedValue);
		}

		return $formattedValue;
	}

/**
 * Format a value on a contact form response based on the field type.
 *
 * @param mixed  $fieldValue The value to format.
 * @param string $fieldType  The field type of the field the value is for.
 * @return mixed The formatted contact form response value.
 */
	public function formatOnFieldType($fieldValue, $fieldType) {
		switch ($fieldType) {
			case 'boolean':
			case 'checkbox':
				$fieldValue = $fieldValue ? 'Yes' : 'No';
				break;
			case 'textarea':
			case 'text_plain':
				$fieldValue = nl2br(h($fieldValue));
				$this->__setEscape(false);
				break;
		}

		return $fieldValue;
	}

/**
 * Set whether to escape a value or not.
 *
 * @param bool $escape True to escape the value, false to not escape the value.
 * @return void.
 */
	private function __setEscape($escape) {
		$this->__escapeValue = $escape;
	}
}
