<?php

App::uses('EvContactFormAppController', 'EvContactForm.Controller');
App::uses('InflectorExt', 'EvInflector.Lib');

class ContactFormResponsesController extends EvContactFormAppController {

	public $uses = "EvContactForm.ContactFormResponse";

	public $helpers = array(
		'EvInflector.InflectorExt'
	);

	public $adminActions = array(
		'admin_index',
		'admin_actioned',
		'admin_unactioned',
		'admin_view',
		'admin_delete',
	);

	/**
	 * index action
	 */
	protected function _adminIndexActions() {
		$actions = array(
			'Delete' => array(
				'cell' => array(
					'class' => 'action'
				),
				'link' => array(
					'url' => array(
						'action' => 'delete'
					),
					'text' => '<i class="s s--delete"></i>',
					'options' => array(
						'escape' => false
					)
				),
				'helper' => 'deleteAction',
				'protected' => '<i class="s s--delete-disabled" title="Disabled"></i>',
			),
			'View' => array(
				'cell' => array(
					'class' => 'action'
				),
				'link' => array(
					'url' => array(
						'admin' => true,
						'action' => 'view'
					),
					'text' => '<i class="s s--view"></i>',
					'options' => array(
						'escape' => false
					)
				)
			)
		);

		return $actions;
	}

	/**
	 * Manages a default admin index view of this model
	 * Includes a filter form and paginated results
	 *
	 * Override in your controller to customise
	 *
	 * @return void
	 */
	public function admin_index() {
		parent::admin_index();

		// get current user to check Permissions
		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			$Model = $this->{$this->modelClass};
			$modelAlias = $Model->alias;

			$data = $this->viewVars['data'];
			$gclids = array();
			foreach ($data as &$response) {
				$fields = unserialize($response[$modelAlias]['form_data']);
				if(isset($fields['adwords_tracking']) && !empty($fields['adwords_tracking'])) {
					$response[$modelAlias]['form_data'] = $fields['adwords_tracking'];
				} else {
					$response[$modelAlias]['form_data'] = '-';
				}
			}
			$this->set('data', $data);
		}
	}

	/**
	 * remove the filter
	 */
	protected function _adminFilterFields() {
		if (Configure::read('EvContactForm.enable_filters')) {

			$filters = parent::_adminFilterFields();
			unset($filters['ContactFormResponse.id']);
			unset($filters['ContactFormResponse.created']);

			$filters['ContactFormResponse.contact_form_id'] = array(
				'label' => 'Contact Form',
				'type' => 'select',
				'compare' => array(
					'ContactFormResponse.contact_form_id' => '%s'
				)

			);

			return $filters;
		}

		return array();
	}

	/**
	 * populate the contact form filter list if in use
	 */
	protected function _adminPopulateLookups() {
		if (Configure::read('EvContactForm.enable_filters')) {

			$this->loadModel('EvContactForm.ContactForm');
			$contactForms = $this->ContactForm->find('list');
			$this->set('contactForms', $contactForms);
		}
	}

	/**
	 * define the columns to show on the listing
	 */
	protected function _adminIndexColumns() {
		$columns = array(
			'ContactFormResponse.name' => array(
				'label' => 'Name',
				'type' => 'string'
			),
			'ContactForm.name' => array(
				'label' => 'Type',
				'type' => 'string'
			),
			'ContactFormResponse.created' => array(
				'label' => 'Submitted',
				'type' => 'datetime_nice'
			),
			'ContactFormResponse.actioned' => array(
				'label' => 'Actioned',
				'type' => 'datetime_nice'
			)
		);

		// check permissions
		if ($this->Permissions->hasSuperAdminPermission($this->Auth->user())) {
			$columns['ContactFormResponse.form_data'] = array(
				'label' => 'GCLIDs',
				'type' => 'string'
			);
		}

		return $columns;
	}

	/**
	 * Pagination settings for admin_index
	 *
	 * Override in your own controller to customise
	 */
	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			//'fields'=>join(",", array_keys($this->_adminIndexColumns())),
			'conditions' => $conditions,
			'contain' => array('ContactForm'),
			'order' => 'ContactFormResponse.created DESC'
		);

		return $paginate;
	}

	/**
	 * remove the add button
	 */
	protected function _adminIndexToolbar() {
		return array();
	}

	/**
	 * Defines the buttons in the toolbar displayed on an admin_form
	 *
	 * Override in your controller to customise
	 */
	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$toolbar = array();
		if (isset($this->request->data[$modelAlias]['id'])) {
			if (is_null($this->request->data[$modelAlias]['actioned'])) {
				$toolbar = array(
					'Mark as Actioned' => array(
						'url' => array(
							'controller' => 'contact_form_responses',
							'plugin' => 'ev_contact_form',
							'action' => 'actioned',
							$this->request->data[$modelAlias]['id']
						),
						'class' => 'toolbar__btn--tick'
					)
				);
			} else {
				$toolbar = array(
					'Mark as Unactioned' => array(
						'url' => array(
							'controller' => 'contact_form_responses',
							'plugin' => 'ev_contact_form',
							'action' => 'unactioned',
							$this->request->data[$modelAlias]['id']
						),
						'class' => 'toolbar__btn--delete'
					)
				);
			}
		}

		$toolbar['Delete'] = array(
			'url' => array('action' => 'delete', $this->request->data[$modelAlias]['id']),
			'class' => 'toolbar__btn--delete'
		);

		return $toolbar;
	}

	protected function _adminViewFieldsCustom($formType = null) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$fields = [
			$modelAlias . '.actioned' => [
				'label' => 'Actioned',
				'type' => 'datetime_nice',
				'displayonly' => 'displayonly',
			],
			$modelAlias . '.created' => [
				'label' => 'Submitted',
				'type' => 'datetime_nice',
				'displayonly' => 'displayonly',
			],
		];

		$config = Configure::read('EvContactForm');
		$contactFormFields = [];
		if (!empty($config['forms'][$formType]['fields'])) {

			$configFields = $config['forms'][$formType]['fields'];

			if (!empty($configFields)) {
				foreach ($configFields as $fieldName => $field) {

					if (empty($field['user_only']) || $field['user_only'] !== true) {
						$fieldLabel = null;
						if (!empty($field['admin-label'])) {
							$fieldLabel = $field['admin-label'];
						} elseif (!empty($field['label'])) {
							if (is_array($field['label']) && !empty($field['label']['text'])) {
								$fieldLabel = $field['label']['text'];
							} else {
								$fieldLabel = $field['label'];
							}
						}

						$contactFormFields[$modelAlias . '.' . $fieldName] = array(
							'label' => $fieldLabel,

							//Data gets formatted before it is displayed so every value is a string.
							'type' => 'string',
							'displayonly' => 'displayonly',

							//Data will be escaped when it is formatted
							'escape' => false,
						);
					}
				}
			}
		}

		$fields = ArrayUtil::addAfter($fields, $modelAlias . '.actioned', $contactFormFields);

		return $fields;
	}

	/**
	 * admin view
	 *
	 * actually just a view with an "mark as actioned button"
	 */
	public function admin_view($id = null) {
		parent::admin_view($id);

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$formData = $Model->ContactForm->find('first', array(
			'conditions' => array(
				'ContactForm.id' => $this->request->data['ContactFormResponse']['contact_form_id']
			)
		));

		$this->set('fields', $this->_adminViewFieldsCustom($formData['ContactForm']['form_data']));

		return;
	}

	/**
	 * mark the message as actioned
	 *
	 * @param   int     contact form id
	 */
	public function admin_actioned($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$data = array(
			'ContactFormResponse' => array(
				'actioned' => date('Y-m-d G:i:s')
			)
		);
		$Model->id = $id;

		if ($Model->save($data)) {

			$this->Session->setFlash(array(
				'title' => "Message Updated",
				'description' => 'Message has been marked as actioned'
			), 'flash_success');

			$this->redirect(array(
				'action' => 'index'
			));

		} else {

			$this->Session->setFlash(array(
				'title' => 'Save Failed',
				'description' => 'Failed to mark the message as actioned'
			), 'flash_fail');

			$this->redirect(array(
				'action' => 'edit',
				$id
			));
		}
	}

	/**
	 * mark the message as unactioned
	 *
	 * @param   int     contact form id
	 */
	public function admin_unactioned($id) {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$data = array(
			'ContactFormResponse' => array(
				'actioned' => ''
			)
		);
		$Model->id = $id;

		if ($Model->save($data)) {

			$this->Session->setFlash(array(
				'title' => "Message Updated",
				'description' => 'Message has been marked as unactioned'
			), 'flash_success');

			$this->redirect(array(
				'action' => 'index'
			));

		} else {

			$this->Session->setFlash(array(
				'title' => 'Save Failed',
				'description' => 'Failed to mark the message as unactioned'
			), 'flash_fail');

			$this->redirect(array(
				'action' => 'edit',
				$id
			));
		}
	}
}
