<?php

App::uses('CustomEmail', 'Lib');

App::uses('FieldFormatter', 'EvContactForm.Lib');

class InjectContactFormComponent extends Component {

	const FORM_SUCCESS = 1;
	const FORM_ERROR = 2;
	const FORM_VALIDATION = 3;

	protected $_Controller;

	private $__form;

/**
 * Called before the Controller::beforeFilter().
 *
 * @param Controller $Controller Controller with components to initialize
 * @return void
 * @link https://book.cakephp.org/2.0/en/controllers/components.html#Component::initialize
 */
	public function initialize(Controller $Controller) {
		$this->_Controller = $Controller;
		$this->__form = null;
	}

/**
 * Get the current form config when using a subclass
 *
 * @return array The form data from $this->__form
 */
	protected function _getForm() {
		return $this->__form;
	}

/**
 * Sets the flash message for the form submission.
 *
 * @param string $status Form status
 * @return void
 */
	protected function _showFlash($status = ', ') {
		// Define the default flash settings.
		$defaultFlash = [
			'message' => 'An error occurred',
			'element' => 'default',
			'params' => [],
			'key' => 'flash'
		];
		// Determine the status of the form submission and therefore the type of flash message
		// to show.
		switch ($status) {
			case self::FORM_SUCCESS:
				$flash = $this->__form['config']['flash']['success'];
				break;
			case self::FORM_ERROR:
				$flash = $this->__form['config']['flash']['error'];
				break;
			case self::FORM_VALIDATION:
				$flash = $this->__form['config']['flash']['validation'];

				if (!empty($this->Model->validationErrors) && empty($this->__form['config']['use_inline_errors'])) {
					if (is_array($flash['message'])) {
						$flash['message']['list'] = $this->Model->validationErrors;
					} else {
						//Convert flash message into array
						$flash['message'] = [
							'description' => $flash['message'],
							'list' => $this->Model->validationErrors,
						];
					}
				}
				break;
		}
		// Combine the default flash settings with those for the current message.
		$flash = !empty($flash) ? Hash::merge($defaultFlash, $flash) : $defaultFlash;
		// Set the flash message.
		$this->_Controller->Flash->set(
			$flash['message'],
			[
				'key' => $flash['key'],
				'params' => $flash['params'],
				'element' => $flash['element'],
				'clear' => true,
			]
		);
	}

/**
 * Handle the validation of the form data submitted by a user.
 *
 * @param array $fields The field config with validation rules on.
 * @param array $data   The user submitted data to validate.
 * @return array|bool If atomic: True on success, or false on failure.
 *    Otherwise: array similar to the $data array passed, but values are set to true/false
 *    depending on whether each record validated successfully.
 */
	protected function _handleValidation($fields, $data) {
		// setup the dynamic validation
		foreach ($fields as $field => $options) {

			if (!empty($options['validate'])) {

				foreach ($options['validate'] as $rule => $ruleData) {
					$this->Model->validator()->add($field, $rule, $ruleData);
				}
			}
		}
		// validate!
		return $this->Model->validator()->validateAssociated($data);
	}

/**
 * Remove validation from fields so that they aren't applied when saving the user submitted data.
 *
 * @param array $fields The field config with validation rules on.
 * @return void. Validation rules on the model are updated.
 */
	protected function _removeValidation($fields) {
		foreach ($fields as $field => $options) {

			if (!empty($options['validate'])) {

				$this->Model->validator()->remove($field);
			}
		}
	}

/**
 * Save user submitted data as a contact form response.
 *
 * @param array  $form            The form config.
 * @param array  $contactFormData User submitted form data.
 * @param string $responseName    The name of the user responding.
 * @return mixed If atomic: True on success, or false on failure.
 *    Otherwise: array similar to the $data array passed, but values are set to true/false
 *    depending on whether each record saved successfully.
 */
	protected function _saveResponse($form, $contactFormData, $responseName) {
		$data = array(
			'ContactFormResponse' => array(
				'contact_form_id' => $form['ContactForm']['id'],
				'name' => $responseName,
				'form_data' => $contactFormData,
				'created' => date('Y-m-d H:i:s')
			)
		);

		$this->_Controller->request->data = array_merge_recursive($this->_Controller->request->data, $data);

		return $this->Model->saveAll($this->_Controller->request->data, array('deep' => true));
	}

/**
 * Set the from details for the notification emails.
 *
 * @return array Email from details.
 */
	protected function _setEmailFrom() {
		return array(
			'name' => Configure::read('SiteSetting.general.site_title'),
			'email' => Configure::read('SiteSetting.general.admin_email'),
		);
	}

/**
 * Set the to data for the notification email. Settings can be set via the config form or the CMS form. The config
 * form setting has precedence over the CMS form. If the value is missing an email will be attempted to be gotten
 * from the site settings. If it is still missing the null is returned which should stop the notification
 * email being sent. Emails will be returned as an array. The first index will be the recipient of the
 * notification email, any additional emails will be added to the CC list.
 *
 * @param string $formName The name of the form to get the email to from.
 * @return array|null To data containing emails for notification email. Null if no email is found.
 */
	protected function _setNotificationEmailTo($formName) {
		if (Configure::check('SiteSetting.EvContactForm.' . $formName . '_email_to')) {
			//EvContactForm config form setting.
			return explode(',', Configure::read('SiteSetting.EvContactForm.' . $formName . '_email_to'));
		} elseif (!empty($this->__form['ContactForm']['email_to'])) {
			//CMS form setting.
			return explode(',', $this->__form['ContactForm']['email_to']);
		} elseif (Configure::check('SiteSetting.general.admin_email')) {
			//Site setting admin email
			return [Configure::read('SiteSetting.general.admin_email')];
		} else {
			//Email can't be found.
			return null;
		}
	}

/**
 * Get the subject for the notification email. It will attempt to use the config from the current form. If that can't
 * be found then the passed subject will be used. If a subject still can't be set it will use the site title.
 *
 * @param string $subjectFromConfig A subject to use if the current form is missing a subject.
 * @return string The subject for the notification email.
 */
	protected function _setNotificationEmailSubject($subjectFromConfig = null) {
		if (!empty($this->__form['ContactForm']['subject_to_staff'])) {
			// Look in the CMS field first
			return $this->__form['ContactForm']['subject_to_staff'];
		} elseif (!empty($subjectFromConfig)) {
			// Then to the config value
			return $subjectFromConfig;
		}

		// Fall back to using the site title
		return Configure::read('SiteSetting.general.site_title');
	}

/**
 * Get the subject for the user email. It will attempt to use the config from the current form. If that can't
 * be found then the passed subject will be used. If a subject still can't be set it will use the site title.
 *
 * @param string $subjectFromConfig A subject to use if the current form is missing a subject.
 * @return string The subject for the user email.
 */
	protected function _setEmailSubject($subjectFromConfig = null) {
		if (!empty($this->__form['ContactForm']['subject_to_user'])) {
			// Look in the CMS field first
			return $this->__form['ContactForm']['subject_to_user'];
		} elseif (!empty($subjectFromConfig)) {
			// Then to the config value
			return $subjectFromConfig;
		}

		// Fall back to using the site title
		return Configure::read('SiteSetting.general.site_title');
	}

/**
 * Send a notification email to a CMS user. The email will be sent to a saved address against the contact form in the
 * CMS or to the general admin email.
 *
 * @param array $emailFrom The email from details.
 * @param array $emailTo   The email to details.
 * @param array $form      The form config.
 * @param array $formData  The user submitted form data.
 * @param array $fields    The fields on the form.
 * @return void.
 */
	protected function _sendNotificationEmail($emailFrom, $emailTo, $form, $formData, $fields) {
		$fields = $this->_formatFields($fields, $formConfig = $form, $formData);

		$email = new CustomEmail();
		$email->template('EvContactForm.contact_form_notification')
			->from($emailFrom['email'], (isset($emailFrom['name'])) ? $emailFrom['name'] : null)
			->emailFormat('html')
			->subject($this->_setNotificationEmailSubject())
			->helpers(
				[
					'InlineCss.InlineCss',
					'EvContactForm.ShowContactForm',
				]
			)->viewVars(
				[
					'form_data' => $formData,
					'fields' => $fields,
					'body' => $form['ContactForm']['body_to_staff'],
				]
			);

		// build the to / cc's
		$to = array_shift($emailTo);

		$email->to($to);

		if (!empty($emailTo)) {
			foreach ($emailTo as $recipient) {
				$email->addCc($recipient);
			}
		}

		$attachFiles = Configure::read('EvContactForm.attach_files_to_notification_email');
		if ($attachFiles === true) {
			$this->_attachFilesToEmail($email, $this->Model->id);
		}

		$this->_modifyEmail(
			$email,
			$target = 'admin',
			$emailInfo = [
				'from' => $emailFrom,
				'to' => $emailTo
			],
			$formConfig = $form,
			$formData
		);

		$email->send();
	}

/**
 * Attach files to an email.
 *
 * @param CustomEmail &$email        Email object.
 * @param int         $contactFormId Contact form ID.
 * @return CustomEmail
 */
	protected function _attachFilesToEmail(&$email, $contactFormId) {
		$this->_Controller->loadModel('Document');
		$documents = $this->_Controller->Document->find('all', [
			'conditions' => [
				'Document.model_id' => $contactFormId,
				'Document.model' => 'ContactFormResponse'
			]
		]);
		$attachments = [];
		$attachmentsSize = 0;
		foreach ($documents as $document) {
			$attachments[] = WWW_ROOT . $document['Document']['filepath'];
			$attachmentsSize += filesize(WWW_ROOT . $document['Document']['filepath']);
		}

		$maxFilesize = Configure::read('EvContactForm.notification_email_attachments_max_filesize');
		if (empty($maxFilesize) || $attachmentsSize <= $maxFilesize) {
			$email->attachments($attachments);
		}

		$email->viewVars(compact('documents'));

		return $email;
	}

/**
 * Modify an email before it is sent. Use the email object to change of the current values for that email. Change
 * to/from addresses, templates and variables. The target is provided so checks can be made to see if the email will be
 * sent to the admin or to the user who submitted the contact form. The $emailInfo contains the to, from and subject
 * passed through to the respective parent calls. The $formConfig contains the CMS form data and the config. The
 * $formData contains the data entered by the user who submitted the form.
 *
 * @param CustomEmail &$email     The email object to modify. See app/Lib/CustomEmail for available methods.
 * @param string      $target     The email's target, 'admin' or 'user'.
 * @param array       $emailInfo  Info already added to the email: 'to', 'from' and 'subject'.
 * @param array       $formConfig Form data from the CMS and the settings from the config.
 * @param array       $formData   User submitted form data.
 * @return void.
 */
	protected function _modifyEmail(
		&$email,
		$target,
		$emailInfo,
		$formConfig,
		$formData
	) {
		$email->viewVars(
			[
				'target' => $target,
			]
		);

		$email->viewRender('EvCore.EvCustom');
	}

/**
 * Send a notification email to the user who submitted the form. A field can be specified in the form config to be used
 * to acquire an email address form the submitted form data.
 *
 * @param array  $emailFrom The email from details.
 * @param array  $emailTo   The email to details.
 * @param array  $form      The form config.
 * @param string $subject   The subject of the email.
 * @param array  $formData  The user submitted form data.
 * @param array  $fields    The fields on the form.
 * @return void.
 */
	protected function _sendUserEmail($emailFrom, $emailTo, $form, $subject, $formData, $fields) {
		$fields = $this->_formatFields($fields, $formConfig = $form, $formData);

		$email = new CustomEmail();
		$email->template('EvContactForm.user_email_notification')
			->from($emailFrom['email'], (isset($emailFrom['name'])) ? $emailFrom['name'] : null)
			->emailFormat('html')
			->subject($subject)
			->helpers(
				[
					'InlineCss.InlineCss',
					'EvContactForm.ShowContactForm',
				]
			)->viewVars(
				[
					'form_data' => $formData,
					'fields' => $fields,
					'body' => $form['ContactForm']['body_to_user'],
				]
			);

		$email->to($emailTo);

		$this->_modifyEmail(
			$email,
			$target = 'user',
			$emailInfo = [
				'from' => $emailFrom,
				'to' => $emailTo,
				'subject' => $subject,
			],
			$formConfig = $form,
			$formData
		);

		$email->send();
	}

/**
 * Load the CMS saved form configuration.
 *
 * @param int|string $id The id or system name of the contact form.
 * @return array ContactForm data.
 */
	protected function _loadForm($id) {
		$this->_Controller->loadModel('EvContactForm.ContactForm');

		if (is_numeric($id)) {
			$form = $this->_Controller->ContactForm->findById($id);
		} else {
			$form = $this->_Controller->ContactForm->findBySystemName($id);
		}

		if (empty($form['ContactForm'])) {
			return false;
		}
		return $form;
	}

/**
 * Load a contact form configuration from the plugin configuration file.
 *
 * @param string $formName The form name in the config to find.
 * @return null|array The contact form config if found, null otherwise.
 */
	protected function _loadConfig($formName) {
		$config = Configure::read('EvContactForm.forms');

		if (empty($config[$formName])) {
			return null;
		}

		return $config[$formName];
	}

/**
 * Check if the submitted user data is spam. The "email_confirm" field is a hidden field so shouldn't be populated by a
 * human user as they wouldn't be able to see it.
 *
 * @param array $formData User submitted form data.
 * @return bool
 */
	protected function _isSpamComment($formData) {
		//Check if the commit is flagged as spam
		if (isset($formData['email_confirm']) && !empty($formData['email_confirm'])) {
			return true;
		} else {
			return false;
		}
	}

/**
 * Checks form submissions and returns true or false based oin whether it contains Spam
 * characters of phrases.
 *
 * @param array $formData contains request data.
 * @return bool True if $formData contains spam, false otherwise.
 */
	protected function _isSpam($formData) {
		$spamChars = array(
			'Ð', 'Ñ', 'Ð', 'µ', 'Ð', '´', 'Ñ', 'Ñ', 'Ð', '°', 'Ð', '²', 'Ð', '»', 'Ñ', 'Ð', 'µ', 'Ð', '¼', 'Ð', '½', 'Ð', '¾', 'Ð', '²', 'Ñ', 'Ð', '¹', 'Ð', '¼', 'Ð', 'µ', 'Ð', '´', 'Ð', '¸', 'Ñ', 'Ð', '¸', 'Ð', '½', 'Ñ', 'Ð', 'º', 'Ð', '¸', 'Ð', '¹', 'Ð', '¿', 'Ð', '¾', 'Ñ', 'Ñ', 'Ð', '°', 'Ð', '»'
		);

		$spamPhrases = array(
			'without prescription',
			'cheap acomplia',
			'Buy Clonidine',
			'Isotretinoin,',
			'Tretinoin',
			'hydrochlorothiazide',
			'losartan',
			'Generic propecia costs',
			'Amoxicillin',
			'500mg',
			'Antibiotics',
			'erotic',
			'xxx',
			' por ',
			'fuck my',
			'test post',
			'sissy',
			'transvestite',
			'feminising',
			'facking',
			'hot sale',
			'oral',
			'Cuny',
			'Blowjob',
			'free gay',
			'Bitter enemy',
			'porn',
			' cum ',
			'rimonabant',
			'viagra',
			'Sildenafil Citrate',
			'java games',
			'effexor',
			'revia',
			'Kopieer',
			'plak',
			'naar',
			' een',
			'korte',
			'registratie',
			'maken',
			'Anmeldung',
			'Farmacia',
			'Bang me',
			'Laminine',
			'hubbub',
			'my nickname',
			'communiquâ',
			'Fidget Spinner',
			'diclofenac',
			'you have won',
			'congratulations',
			'What are you waiting for',
			'You have been selected',
			'Removes wrinkles',
			'Reverses aging',
			'Be your own boss',
			'Can’t live without',
			'Cashcashcash',
			'Earn extra cash',
			'Human growth hormone',
			'Explode your business',
			'horny',
			'F r e e',
			'Meet singles',
			'local singles',
			'no catch'
		);

		foreach ($formData as $requestData) {
			if (is_array($requestData)) {
				foreach ($requestData as $data) {
					// check for spam chars
					foreach ($spamChars as $spamChar) {
						if (is_string($data) && strpos($data, $spamChar) !== false) {
							return true;
						}
					}
					foreach ($spamPhrases as $spamPhrase) {
						if (is_string($data) && strpos(strtoupper($data), strtoupper($spamPhrase)) !== false) {
							return true;
						}
					}
				}
			}
		}

		return false;
	}

/**
 * Display a contact form. Load the configuration data form the plugin configuration file and form the database and
 * inject the contact form info into the view variables. When a user submits data, it is validated and their response
 * saved here also.
 *
 * @param int|string $id        The id or system name of the contact form.
 * @param string     $name      The name of the form to use in the view.
 * @param string     $modelName The plugin notation of the model to use to save a response to.
 * @return void. View variables set with config data or redirect.
 */
	public function display($id, $name, $modelName = 'EvContactForm.ContactFormResponse') {
		// Load configs, overridable!
		$this->__form = $this->_loadForm($id);

		if (empty($this->__form)) {
			return null;
		}

		$formData = $this->_loadConfig($this->__form['ContactForm']['form_data']);

		if (empty($formData)) {
			return null;
		}

		$this->Model = EvClassRegistry::init($modelName);

		$this->_getErrors($name);

		$this->__form['config'] = $formData;

		$fields = $formData['fields'];

		if (
			$this->_Controller->request->is('post') &&
			!empty($this->_Controller->request->data[$name]) &&
			isset($this->_Controller->request->data[$name]['contact-form-send']) &&
			$this->_Controller->request->data[$name]['contact-form-send'] === $name
		) {

			if (isset($this->_Controller->request->data['g-recaptcha-response'])) {
				$this->_Controller->request->data[$name]['g-recaptcha-response'] = $this->_Controller->request->data['g-recaptcha-response'];
			}

			if ($this->_handleValidation($fields, $this->_Controller->request->data[$name])) {

				// passed validation, remove rules so we can add to the database
				$this->_removeValidation($fields);

				// Remove any recaptcha response now it has been validated
				unset($this->_Controller->request->data[$name]['g-recaptcha-response']);

				// serialize it
				$contactFormData = serialize($this->_Controller->request->data[$name]);

				// name label
				$nameLabel = $formData['name_label'];

				if (!empty($this->_Controller->request->data[$name][$nameLabel])) {
					$responseName = $this->_Controller->request->data[$name][$nameLabel];
				} else {
					$responseName = 'Unknown Name';
				}

				$this->_Controller->set($name . '_completed', true);

				// save it or pretend to if it is spam
				if (
					empty($formData['skip_spam_detection'])
					&& (
						$this->_isSpamComment($this->_Controller->request->data[$name])
						|| $this->_isSpam($this->_Controller->request->data)
					)
				) {

					//The comment has been flagged as spam so we pretend to save it but don't
					if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
						$this->_showFlash(self::FORM_SUCCESS);
					}

					$successRedirect = $this->_setupRedirect(Hash::get($formData, 'success_redirect'));

					if (isset($successRedirect) && !empty($successRedirect)) {
						if ($successRedirect === true) {
							$successRedirect = Router::currentRoute()->defaults;
						}

						return $this->_Controller->redirect($successRedirect);
					} else {
						$this->_Controller->set($name . '_sent', true);
					}
				} elseif ($this->_saveResponse($this->__form, $contactFormData, $responseName)) {
					$formattedData = $this->_Controller->ContactForm->ContactFormResponse->formatResponseData(
						$this->_Controller->request->data[$name],
						$this->__form['config']['fields']
					);

					$emailFrom = $this->_setEmailFrom();
					if (! empty($formData['admin_email_notification']['from'])) {
						$emailFrom = $formData['admin_email_notification']['from'];
					}

					$emailTo = $this->_setNotificationEmailTo($name);

					if (!empty($emailFrom) && !empty($emailTo)) {

						// send the email!
						$this->_sendNotificationEmail($emailFrom, $emailTo, $this->__form, $formattedData, $fields);
					}

					// If this form is configured to send an email to the user to say thanks
					if (!empty($formData['user_email_notification'])) {
						$userEmailConfig = $formData['user_email_notification'];

						//Default to taking the email field as the to address, otherwise use specified.
						$userEmailField = 'email';
						if (!empty($userEmailConfig['email_field'])) {
							$userEmailField = $userEmailConfig['email_field'];
						}

						$userEmailSubject = null;
						if (!empty($userEmailConfig['subject'])) {
							$userEmailSubject = $userEmailConfig['subject'];
						}

						// Get the user's email, so we can mail them and say thanks
						$userEmailAddress = array($this->_Controller->request->data[$name][$userEmailField]);
						$userFromEmailAddress = (!empty($userEmailConfig['from']) ? $userEmailConfig['from'] : $this->_setEmailFrom());
						$subject = $this->_setEmailSubject( $userEmailSubject );
						$this->_sendUserEmail($userFromEmailAddress, $userEmailAddress, $this->__form, $subject, $formattedData, $fields);
					}

					// unset form data and show success
					unset($this->_Controller->request->data[$name]);
					if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
						$this->_showFlash(self::FORM_SUCCESS);
					}

					$successRedirect = $this->_setupRedirect(Hash::get($formData, 'success_redirect'));

					if (empty($successRedirect)) {
						$successRedirect = $this->_Controller->request->here();
					}

					//  Since we're doing redirects, the variables will need to be put in the Session so they
					//  can be used elsewhere.
					$this->_Controller->Session->write('ev_contact_form_' . $name . '_sent', true);
					$this->_Controller->Session->write('ev_contact_form_' . $name . '_adwords_tracking', $this->__form['ContactForm']['adwords_tracking']);

					return $this->_Controller->redirect($successRedirect);
				} else {
					if (isset($formData['flash']['error']) && !empty($formData['flash']['error'])) {
						$this->_showFlash(self::FORM_ERROR);
					}
					// failed to save, error
					$contactFormErrors = array(
						'There was an issue saving your message, please try again'
					);

					$this->_Controller->set($name . '_errors', $contactFormErrors);

					$errorRedirect = Hash::get($formData, 'error_redirect');
					if (!empty($errorRedirect)) {
						$this->_setErrors($name, $contactFormErrors);
						return $this->_Controller->redirect($errorRedirect);
					}
				}
			} else {

				// compile the validate errors
				$this->_showFlash(self::FORM_VALIDATION);

				$contactFormErrors = array();

				foreach ($this->Model->validationErrors as $errors) {
					foreach ($errors as $msg) {
						$contactFormErrors[] = $msg;
					}
				}

				// I dont know why I need this
				if (isset($this->_Controller->request->data[$name][$name])) {
					$temp = $this->_Controller->request->data[$name][$name];
					$this->_Controller->request->data[$name] = $temp;
				}

				$this->_Controller->set($name . '_errors', $contactFormErrors);

				$errorRedirect = $this->_setupRedirect(Hash::get($formData, 'error_redirect'));
				if (!empty($errorRedirect)) {
					$this->_setErrors($name, $contactFormErrors);
					return $this->_Controller->redirect($errorRedirect);
				}
			}
		}

		if (!empty($formData['fields'])) {
			$formData['fields'] = $this->_formatFields($formData['fields'], $this->__form);
		}

		$this->_Controller->set($name . '_form', $formData);
	}

/**
 * Format the fields on a contact form. If fields sections are to be used then fields will be grouped under their
 * section name. If sections and groups are to be used then fields will be grouped under their section name
 * and group name.
 *
 * @param array $fields     The fields of the contact form.
 * @param array $formConfig The config of the contact form.
 * @param array $formData   The submitted user data of the contact form.
 * @return array Formatted fields.
 */
	protected function _formatFields($fields, $formConfig, $formData = []) {
		if (!empty($fields) && !empty($formConfig['config']['use_field_sections'])) {
			$FieldFormatter = new FieldFormatter();
			$fields = $FieldFormatter->sectionAndGroupFields(
				$fields,
				$groupFields = !empty($formConfig['config']['use_field_groups'])
			);
		}

		return $fields;
	}

/**
 * Work out if redirect is literal or needs looking up.
 *
 * @param mixed $data The redirect data from the plugin config.
 * @return mixed $redirect The final determined redirect.
 * @throws NotFoundException when redirecting to a page and the page can't be found.
 */
	protected function _setupRedirect($data) {
		if ($data === true) {
			return Router::currentRoute()->defaults;
		}

		//If the redirect is relative to the page the user is currently on.
		if (!empty($data['relative'])) {
			$currentRoute = Router::currentRoute()->defaults;

			unset($data['relative']);

			return array_merge($currentRoute, $data);
		}

		//Redirect to page using internal title
		if (!empty($data['internal_title'])) {
			$Page = EvClassRegistry::init('EvCore.Page');
			$page = $Page->findByInternalTitle($data['internal_title']);
			if (empty($page['Page']['id'])) {
				throw new NotFoundException();
			} else {
				return array(
					'plugin' => false,
					'controller' => isset($data['controller']) ? $data['controller'] : 'pages',
					'action' => 'view',
					$page['Page']['id']
				);
			}
		}

		return $data;
	}

/**
 * When the user submitted data fails to validate or there is an error while saving the contact form response, errors
 * are saved to the user's session so that they can be displayed on the form.
 *
 * @param string $name   The name of the contact form used in the view.
 * @param array  $errors An array of errors.
 * @return void.
 */
	protected function _setErrors($name, $errors) {
		CakeSession::write($name . '_errors', $errors);
		CakeSession::write($name . '_submitted_data', $this->_Controller->request->data[$name]);

		if (!empty($this->Model->validationErrors) && !empty($this->__form['config']['use_inline_errors'])) {
			// Need to save the validation errors from the response
			CakeSession::write($name . '_inline_errors', $this->Model->validationErrors);
		}
	}

/**
 * Check if the session contains errors and the data that errored and set it against the controller and model so that
 * errors are displayed correctly on the form.
 *
 * @param string $name The name of the contact form used in the view.
 * @return void.
 */
	protected function _getErrors($name) {
		$errorsInSession = CakeSession::consume($name . '_errors');
		if (!empty($errorsInSession)) {
			$this->_Controller->set($name . '_errors', $errorsInSession);
		}

		$submittedData = CakeSession::consume($name . '_submitted_data');
		if (!empty($submittedData)) {
			$this->_Controller->set($name . '_submitted_data', $submittedData);
		}

		$inlineErrors = CakeSession::consume($name . '_inline_errors');
		if (!empty($inlineErrors)) {
			$this->Model->validationErrors = $inlineErrors;
		}
	}

}
