<?php

class ShowContactFormHelper extends AppHelper {

/**
 * Display a contact form on the page using the variables set in the controller. Need to use the same form name
 * as the one used to inject the form.
 * @param  string $name      Name of the form to display.
 * @param  array  $extraVars extra variables to pass to the contact form template
 * @return string            A HTML string to display the contact form
 */
	public function display($name, $extraVars = array()) {
		$template = 'EvContactForm.contact_form';
		$formVars = $this->_View->viewVars[$name . '_form'];
		if (!empty($formVars['template'])) {
			$template = $formVars['template'];
		}
		return $this->_View->Element($template,
			array_merge_recursive($extraVars,
			array(
				'form' => $formVars,
				'name' => $name,
			))
		);
	}

/**
 * Display a contact form field using the config settings. Used on contact form templates.
 * @param  string $fieldName       The name to be given to the field.
 * @param  array  $fieldAttributes The field attributes to be given to the field
 * @param  bool   $useInlineErrors Whether the form is displaying errors inline or not.
 * @return string                  A HTML string to display the field.
 */
	public function displayField($fieldName, $fieldAttributes = [], $useInlineErrors = false) {
		$fieldHtml = '';

		// reset unwanted $fieldAttributes
		unset($fieldAttributes['validate']);
		unset($fieldAttributes['admin-label']);

		// assign default values for the loop
		$fieldDiv = false;
		$fieldError = false;

		if (
			! empty($this->_View->validationErrors['ContactFormResponse'][$fieldName]) &&
			$useInlineErrors === true
		):
			$fieldError = $this->_View->Form->error('ContactFormResponse.' . $fieldName);
			$fieldAttributes['afterInput'] = $fieldError;
		endif;

		// extract the div class so error messages can be echo after the input
		// when the field has failed validation
		if (! empty($fieldAttributes['div'])):
			$fieldDiv = $fieldAttributes['div'];
			$fieldAttributes['div'] = false;

			// assign a generic error class when the field has errored on save
			if (is_array($fieldError) && ! empty($fieldError)):
				$fieldDiv .= ' error';
			endif;

			$fieldHtml .= $this->_View->Html->tag('div', null, ['class' => $fieldDiv]);
		endif;

		if ($fieldAttributes['type'] == 'file') {
			$fieldHtml .= $this->_View->Form->input('FilesDocument.0.model', [
				'type' => 'hidden',
				'value' => 'ContactFormResponse'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.0.attachment_type', [
				'type' => 'hidden',
				'value' => 'FilesDocument'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.0.id', [
				'type' => 'hidden',
				'value' => '0'
			]);

			$fieldHtml .= $this->_View->Form->input('FilesDocument.0.filename', $fieldAttributes);

		} else {
			$fieldHtml .= $this->_View->Form->input($fieldName, $fieldAttributes);
		}

		if (! empty($fieldDiv)):
			$fieldHtml .= $this->_View->Html->tag('/div');
		endif;

		return $fieldHtml;
	}

/**
 * Display a field to catch spam. Used on contact form templates
 * @return string A HTML string to display the spam field.
 */
	public function displaySpamField() {
		$spamHtml = '';

		$spamHtml .= $this->_View->Form->input('email_confirm', array(
			'type' => 'text',
			'div' => array(
				'style' => 'display: none;'
			)
		));

		return $spamHtml;
	}

/**
 * Display a submit button. Used on contact form templates
 * @param  string $formName 		  The name of the form to attach the button to. Used if no button text has been provided
 * @param  array  $buttonAttributes   The field attributes to be given to the button
 * @return string           		  A HTML string to display the submit button.
 */
	public function displaySubmit($formName, $buttonAttributes = []) {
		$submitHtml = '';

		if (!empty($buttonAttributes['button_text'])):
			$inputName = $buttonAttributes['button_text'];
			unset($buttonAttributes['button_text']);
		else:
			$inputName = $formName . '.save';
		endif;

		$submitHtml .= $this->_View->Form->input($inputName, $buttonAttributes);

		return $submitHtml;
	}

}
