<?php

	App::uses('CustomEmail', 'Lib');

	class InjectContactFormComponent extends Component {

		protected $_Controller;

		const FORM_SUCCESS = 1;
		const FORM_ERROR = 2;
		const FORM_VALIDATION = 3;

		private $__form;

		public function initialize(Controller $Controller) {
			$this->_Controller = $Controller;
			$this->__form = null;
		}

		/**
		 * Sets the flash message for the form submission.
		 * @param string $status Form status
		 * @return void
		 */
		protected function _showFlash($status = ', ') {
			// Define the default flash settings.
			$defaultFlash = [
				'message' => 'An error occurred',
				'element' => 'default',
				'params' => [],
				'key' => 'flash'
			];
			// Determine the status of the form submission and therefore the type of flash message
			// to show.
			switch ($status) {
				case self::FORM_SUCCESS:
					$flash = $this->__form['config']['flash']['success'];
					break;
				case self::FORM_ERROR:
					$flash = $this->__form['config']['flash']['error'];
					break;
				case self::FORM_VALIDATION:
					$flash = $this->__form['config']['flash']['validation'];
					break;
			}
			// Combine the default flash settings with those for the current message.
			$flash = !empty($flash) ? Hash::merge($defaultFlash, $flash) : $defaultFlash;
			// Set the flash message.
			$this->_Controller->Flash->set(
				$flash['message'],
				[
					'key' => $flash['key'],
					'params' => $flash['params'],
					'element' => $flash['element']
				]
			);
			return;
		}

		protected function _handleValidation($fields, $data) {
			// setup the dynamic validation
			foreach ($fields as $field => $options) {

				if (!empty($options['validate'])) {

					foreach ($options['validate'] as $rule => $ruleData) {
						$this->Model->validator()->add($field, $rule, $ruleData);
					}
				}
			}
			// validate!
			return $this->Model->validator()->validateAssociated($data);
		}

		protected function _removeValidation($fields) {
			foreach ($fields as $field => $options) {

				if (!empty($options['validate'])) {

					$this->Model->validator()->remove($field);
				}
			}
		}

		protected function _saveResponse($form, $contactFormData, $responseName) {
			$data = array(
				'ContactFormResponse' => array(
					'contact_form_id' => $form['ContactForm']['id'],
					'name' => $responseName,
					'form_data' => $contactFormData,
					'created' => date('Y-m-d H:i:s')
				)
			);

			$this->_Controller->request->data = array_merge($this->_Controller->request->data, $data);

			return $this->Model->saveAll($this->_Controller->request->data, array('deep' => true));
		}

		protected function _setEmailFrom() {
			return array(
				'name' => Configure::read('SiteSetting.general.site_title'),
				'email' => Configure::read('SiteSetting.general.admin_email'),
			);
		}

/**
 * Set the to data for the notification email. Settings can be set via the config form or the CMS form. The config
 * form setting has precedence over the CMS form. If the value is missing an email will be attempted to be gotten
 * from the site settings. If it is still missing the null is returned which should stop the notification
 * email being sent. Emails will be returned as an array. The first index will be the recipient of the
 * notification email, any additional emails will be added to the CC list.
 *
 * @param string $formName The name of the form to get the email to from.
 * @return array|null To data containing emails for notification email. Null if no email is found.
 */
		protected function _setNotificationEmailTo($formName) {
			if (Configure::check('SiteSetting.EvContactForm.' . $formName . '_email_to')) {
				//EvContactForm config form setting.
				return explode(',', Configure::read('SiteSetting.EvContactForm.' . $formName . '_email_to'));
			} elseif (!empty($this->__form['ContactForm']['email_to'])) {
				//CMS form setting.
				return explode(',', $this->__form['ContactForm']['email_to']);
			} elseif (!empty(Configure::check('SiteSetting.general.admin_email'))) {
				//Site setting admin email
				return [Configure::read('SiteSetting.general.admin_email')];
			} else {
				//Email can't be found.
				return null;
			}
		}

		protected function _setEmailSubject($subjectFromConfig = null) {
			if (!empty($this->__form['ContactForm']['subject_to_user'])) {
				// Look in the CMS field first
				return $this->__form['ContactForm']['subject_to_user'];
			} elseif (!empty($subjectFromConfig)) {
				// Then to the config value
				return $subjectFromConfig;
			}

			// Fall back to using the site title
			return Configure::read('SiteSetting.general.site_title');
		}

		protected function _sendNotificationEmail($emailFrom, $emailTo, $form, $formData, $fields) {
			$email = new CustomEmail();
			$email->template('EvContactForm.contact_form_notification')
					->from($emailFrom['email'], (isset($emailFrom['name'])) ? $emailFrom['name'] : null)
					->emailFormat('html')
					->subject($form['ContactForm']['subject_to_staff'])
					->helpers(['InlineCss.InlineCss'])
					->viewVars(array(
						'form_data' => $formData,
						'fields' => $fields,
						'body' => $form['ContactForm']['body_to_staff']
			));
			// build the to / cc's
			$to = array_shift($emailTo);

			$email->to($to);

			if (!empty($emailTo)) {
				foreach ($emailTo as $recipient) {
					$email->addCc($recipient);
				}
			}
			$email->send();
		}

		protected function _sendUserEmail($emailFrom, $emailTo, $form, $subject, $formData, $fields) {
			$email = new CustomEmail();
			$email->template('EvContactForm.user_email_notification')
					->from($emailFrom['email'], (isset($emailFrom['name'])) ? $emailFrom['name'] : null)
					->emailFormat('html')
					->subject($subject)
					->helpers(['InlineCss.InlineCss'])
					->viewVars(array(
						'form_data' => $formData,
						'fields' => $fields,
						'body' => $form['ContactForm']['body_to_user']
			));

			$email->to($emailTo);
			$email->send();
		}

		protected function _loadForm($id) {
			$this->_Controller->loadModel('ContactForm');

			if (is_numeric($id)) {
				$form = $this->_Controller->ContactForm->findById($id);
			} else {
				$form = $this->_Controller->ContactForm->findBySystemName($id);
			}

			if (empty($form['ContactForm'])) {
				return false;
			}
			return $form;
		}

		protected function _loadConfig($formName) {
			$config = Configure::read('EvContactForm');
			return $config[$formName];
		}

		protected function _isSpamComment($formData) {
			//Check if the commit is flagged as spam
			if (isset($formData['email_confirm']) && !empty($formData['email_confirm'])) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Checks form submissions and returns true or false based oin whether it contains Spam
		 * characters of phrases.
		 * @param array $formData contains request data
		 * @return boolean
		 */
		protected function _isSpam($formData) {
			$spamChars = array(
				'Ð', 'Ñ', 'Ð', 'µ', 'Ð', '´', 'Ñ', 'Ñ', 'Ð', '°', 'Ð', '²', 'Ð', '»', 'Ñ', 'Ð', 'µ', 'Ð', '¼', 'Ð', '½', 'Ð', '¾', 'Ð', '²', 'Ñ', 'Ð', '¹', 'Ð', '¼', 'Ð', 'µ', 'Ð', '´', 'Ð', '¸', 'Ñ', 'Ð', '¸', 'Ð', '½', 'Ñ', 'Ð', 'º', 'Ð', '¸', 'Ð', '¹', 'Ð', '¿', 'Ð', '¾', 'Ñ', 'Ñ', 'Ð', '°', 'Ð', '»'
			);

			$spamPhrases = array(
				'without prescription',
				'cheap acomplia',
				'Buy Clonidine',
				'Isotretinoin,',
				'Tretinoin',
				'hydrochlorothiazide',
				'losartan',
				'Generic propecia costs',
				'Amoxicillin',
				'500mg',
				'Antibiotics',
				'erotic',
				'xxx',
				' por ',
				'fuck my',
				'test post',
				'sissy',
				'transvestite',
				'feminising',
				'facking',
				'hot sale',
				'oral',
				'Cuny',
				'Blowjob',
				'free gay',
				'Bitter enemy',
				'porn',
				'cum',
				'rimonabant',
				'viagra',
				'Sildenafil Citrate',
				'java games',
				'effexor',
				'revia',
				'Kopieer',
				'plak',
				'naar',
				' een',
				'korte',
				'registratie',
				'maken',
				'Anmeldung',
				'Farmacia',
				'Bang me',
				'Laminine',
				'hubbub',
				'my nickname',
				'communiquâ',
				'Fidget Spinner',
				'diclofenac',
				'you have won',
				'congratulations',
				'What are you waiting for',
				'You have been selected',
				'Removes wrinkles',
				'Reverses aging',
				'Be your own boss',
				'Can’t live without',
				'Cashcashcash',
				'Earn extra cash',
				'Human growth hormone',
				'Explode your business',
				'horny',
				'F r e e',
				'Meet singles',
				'local singles',
				'no catch'
			);

			foreach ($formData as $requestData) {
				if (is_array($requestData)) {
					foreach ($requestData as $data) {
						// check for spam chars
						foreach ($spamChars as $spamChar) {
							if (is_string($data) && strpos($data, $spamChar) !== false) {
								return true;
							}
						}
						foreach ($spamPhrases as $spamPhrase) {
							if (is_string($data) && strpos(strtoupper($data), strtoupper($spamPhrase)) !== false) {
								return true;
							}
						}
					}
				}
			}

			return false;
		}

		public function display($id, $name, $modelName = 'EvContactForm.ContactFormResponse') {
			// Load configs, overridable!
			$this->__form = $this->_loadForm($id);
			$formData = $this->_loadConfig($this->__form['ContactForm']['form_data']);
			$this->__form['config'] = $formData;

			$fields = $formData['fields'];

			if (
				$this->_Controller->request->is('post') &&
				!empty($this->_Controller->request->data[$name]) &&
				isset($this->_Controller->request->data[$name]['contact-form-send']) &&
				$this->_Controller->request->data[$name]['contact-form-send'] === $name
			) {
				$this->Model = ClassRegistry::init($modelName);

				if (isset($this->_Controller->request->data['g-recaptcha-response'])) {
					$this->_Controller->request->data[$name]['g-recaptcha-response'] = $this->_Controller->request->data['g-recaptcha-response'];
				}

				if ($this->_handleValidation($fields, $this->_Controller->request->data[$name])) {

					// passed validation, remove rules so we can add to the database
					$this->_removeValidation($fields);

					// Remove any recaptcha response now it has been validated
					unset($this->_Controller->request->data[$name]['g-recaptcha-response']);

					// serialize it
					$contactFormData = serialize($this->_Controller->request->data[$name]);

					// name label
					$nameLabel = $formData['name_label'];

					if (!empty($this->_Controller->request->data[$name][$nameLabel])) {
						$responseName = $this->_Controller->request->data[$name][$nameLabel];
					} else {
						$responseName = 'Unknown Name';
					}

					$this->_Controller->set($name . '_completed', true);

					// save it or pretend to if it is spam
					if ($this->_isSpamComment($this->_Controller->request->data[$name]) || $this->_isSpam($this->_Controller->request->data)) {

						//The comment has been flagged as spam so we pretend to save it but don't
						if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
							$this->_showFlash(self::FORM_SUCCESS);
						}

						$successRedirect = Hash::get($formData, 'success_redirect');

						if (isset($successRedirect) && !empty($successRedirect)) {
							return $this->_Controller->redirect($successRedirect);
						} else {
							$this->_Controller->set($name . '_sent', true);
						}
					} elseif ($this->_saveResponse($this->__form, $contactFormData, $responseName)) {

						$emailFrom = $this->_setEmailFrom();
						$emailTo = $this->_setNotificationEmailTo($name);

						if (!is_null($emailFrom) && !is_null($emailTo)) {

							// send the email!
							$this->_sendNotificationEmail($emailFrom, $emailTo, $this->__form, $this->_Controller->request->data[$name], $fields);
						}

						// If this form is configured to send an email to the user to say thanks
						if (isset($formData['user_email_notification']) && !empty($formData['user_email_notification'])) {
							$userEmailConfig = $formData['user_email_notification'];

							// Get the user's email, so we can mail them and say thanks
							$userEmailAddress = array($this->_Controller->request->data[$name]['email']);
							$userFromEmailAddress = (!empty($userEmailConfig['from']) ? $userEmailConfig['from'] : $this->_setEmailFrom());
							$subject = $this->_setEmailSubject( $userEmailConfig['subject'] );
							$this->_sendUserEmail($userFromEmailAddress, $userEmailAddress, $this->__form, $subject, $this->_Controller->request->data[$name], $fields);
						}

						// unset form data and show success
						unset($this->_Controller->request->data[$name]);
						if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
							$this->_showFlash(self::FORM_SUCCESS);
						}
						$successRedirect = Hash::get($formData, 'success_redirect');
						if (isset($successRedirect) && !empty($successRedirect)) {
							if ($successRedirect === true) {
								$successRedirect = Router::currentRoute()->defaults;
							}

							return $this->_Controller->redirect($successRedirect);
						} else {
							$this->_Controller->set($name . '_sent', true);
							$this->_Controller->set($name . '_adwords_tracking', $this->__form['ContactForm']['adwords_tracking']);
						}
					} else {
						if (isset($formData['flash']['error']) && !empty($formData['flash']['error'])) {
							$this->_showFlash(self::FORM_ERROR);
						}
						// failed to save, error
						$contactFormErrors = array(
							'There was an issue saving your message, please try again'
						);

						$this->_Controller->set($name . '_errors', $contactFormErrors);
					}
				} else {

					// compile the validate errors
					$this->_showFlash(self::FORM_VALIDATION);

					$contactFormErrors = array();

					foreach ($this->Model->validationErrors as $errors) {
						foreach ($errors as $msg) {
							$contactFormErrors[] = $msg;
						}
					}

					// I dont know why I need this
					if (isset($this->_Controller->request->data[$name][$name])) {
						$temp = $this->_Controller->request->data[$name][$name];
						$this->_Controller->request->data[$name] = $temp;
					}
					$this->_Controller->set($name . '_errors', $contactFormErrors);
				}
			}

			$this->_Controller->set($name . '_form', $formData);
		}

	}
