<?php

App::uses('EvContactFormAppController', 'EvContactForm.Controller');
App::uses('InflectorExt', 'Lib');

class ContactFormResponsesController extends EvContactFormAppController
{

	public $uses = "EvContactForm.ContactFormResponse";

	public $helpers = array(
		'InflectorExt'
	);

	public $adminActions = array(
		'admin_index',
		'admin_actioned',
		'admin_unactioned',
		'admin_view',
		'admin_delete',
		'admin_export'
	);


	/**
	 * index action
	 */
	public function _adminIndexActions() {

		return array(
			'Admin View',
			'Delete'
		);
	}

/**
 * Used to populate filter form drop down selects
 *
 * @return void
 */
	protected function _adminPopulateFilterLookups() {
		// assign the contact form type options
		$this->set(
			'contactForms',
			$this->{$this->modelClass}->ContactForm->find('list', [
				'order' => [
					'ContactForm.name' => 'ASC'
				]
			])
		);
	}

/**
 * Converts a filter form POST submission into a bookmarkable GET request
 * Converts a GET request into a conditions array suitable for filtering paginate
 *
 * @return array
 */
	protected function _processFilter($adminFilter = true) {
		$conditions = parent::_processFilter($adminFilter);

		$Model = $this->{$this->modelClass};

		// convert the date filter request to a from and to conditions
		if (
			! empty($this->request->data['Filter'][$Model->alias]['created_from']) ||
			! empty($this->request->data['Filter'][$Model->alias]['created_to'])
		) {
			// remove the previously assigned 'created' condition
			unset($conditions[$Model->alias . '.created']);

			// compare the two created filters
			if (
				! empty($this->request->data['Filter'][$Model->alias]['created_from'])
			) {
				$createdFrom = $this->request->data['Filter'][$Model->alias]['created_from'];
				$conditions[$Model->alias . '.created >='] = CakeTime::toServer(
					$createdFrom . '00:00:00',
					Configure::read('Config.timezone'),
					'Y-m-d  H:i:s'
				);
			}

			if (
				! empty($this->request->data['Filter'][$Model->alias]['created_to'])
			) {
				$createdTo = $this->request->data['Filter'][$Model->alias]['created_to'];
				$conditions[$Model->alias . '.created <='] = CakeTime::toServer(
					$createdTo . '24:00:00',
					Configure::read('Config.timezone'),
					'Y-m-d  H:i:s'
				);
			}

		}

		return $conditions;
	}

/**
 * Defines the fields displayed in a filter form.
 *
 * @return array Contains the filter fields to display on the form
 */
	protected function _adminFilterFields() {
		$adminFilterFields = [];

		if (! empty(Configure::read('EvContactForm.settings.allowAdminExport'))) {
			// add filter field for form type
			$adminFilterFields['ContactFormResponse.contact_form_id'] = [
				'label' => 'Form Type',
				'type' => 'select',
				'compare' => ["ContactFormResponse.contact_form_id" => "%s"]
			];

			// add fitler field for created_from date
			$adminFilterFields['ContactFormResponse.created_from'] = [
				'label' => 'Date From',
				'type' => 'daterange_from',
				'compare' => ["ContactFormResponse.created" => "%s"]
			];

			// add fitler field for created_to date
			$adminFilterFields['ContactFormResponse.created_to'] = [
				'label' => 'Date To',
				'type' => 'daterange_to',
				'compare' => ["ContactFormResponse.created" => "%s"]
			];
		}

		return $adminFilterFields;
	}

	/**
	 * define the columns to show on the listing
	 */
	public function _adminIndexColumns() {

		return array(
			'ContactFormResponse.name' => array(
				'label' => 'Name',
				'type' => 'string'
			),
			'ContactForm.name' => array(
				'label' => 'Type',
				'type' => 'string'
			),
			'ContactFormResponse.created' => array(
				'label' => 'Submitted',
				'type' => 'datetime_nice'
			),
			'ContactFormResponse.actioned' => array(
				'label' => 'Actioned',
				'type' => 'datetime_nice'
			)
		);
	}

	/**
	 * Pagination settings for admin_index
	 *
	 * Override in your own controller to customise
	 */
	public function _adminIndexPaginate() {

		$conditions = $this->_processFilter();

		$paginate = array(
			//'fields'=>join(",", array_keys($this->_adminIndexColumns())),
			'conditions'=>$conditions,
			'contain' => array('ContactForm'),
			'order' => 'ContactFormResponse.created DESC'
		);

		return $paginate;
	}

/**
 * Defines the toolbar buttons displayed in admin_index
 *
 * @return array
 */
	protected function _adminIndexToolbar() {
		$adminIndexToolbar = [];

		if (! empty(Configure::read('EvContactForm.settings.allowAdminExport'))) {
			$adminIndexToolbar['CSV Export'] = [
				'url' => [
					'action' => 'export',
				]
			];

			if (! empty($this->request->params['named'])) {
				$adminIndexToolbar['CSV Export']['url'] += $this->request->params['named'];
			}
		}

		return $adminIndexToolbar;
	}

	/**
	 * Defines the buttons in the toolbar displayed on an admin_form
	 *
	 * Override in your controller to customise
	 */
	public function _adminFormToolbar($id = NULL) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$toolbar = array();
		if(isset($this->request->data[$modelAlias]['id']))
		{
			if(is_null($this->request->data[$modelAlias]['actioned'])) {
				$toolbar = array(
					'Mark as Actioned' => array(
						'url' => array(
							'controller' => 'contact_form_responses',
							'plugin' => 'ev_contact_form',
							'action' => 'actioned',
							$this->request->data[$modelAlias]['id']
						),
						'class' => 'toolbar__btn--tick'
					)
				);
			} else {
				$toolbar = array(
					'Mark as Unactioned' => array(
						'url' => array(
							'controller' => 'contact_form_responses',
							'plugin' => 'ev_contact_form',
							'action' => 'unactioned',
							$this->request->data[$modelAlias]['id']
						),
						'class' => 'toolbar__btn--delete'
					)
				);
			}
		}

		$toolbar['Delete'] = array(
			'url' => array('action' => 'delete', $this->request->data[$modelAlias]['id']),
			'class' => 'toolbar__btn--delete'
		);

		return $toolbar;
	}


	protected function _adminViewFieldsCustom($formType = null) {

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$config = Configure::read('EvContactForm');
		if (isset($config[$formType]['fields']) && !empty($config[$formType]['fields'])) {

			$configFields = $config[$formType]['fields'];

			if (isset($configFields) && !empty($configFields)) {

				$fields = array();

				$fields[$modelAlias . '.actioned'] = array(
					'label' => 'Actioned',
					'type' => 'datetime_nice'
				);

				foreach ($configFields as $keyName => $field) {

					$fields[$modelAlias . '.' . $keyName] = array(
						'label' => $field['admin-label'],
						'type' => $field['type'],
					);
				}

				$fields[$modelAlias . '.created'] = array(
					'label' => 'Submitted',
					'type' => 'datetime_nice'
				);
			}
		}

		return (!empty($fields)) ? $fields : array();
	}

/**
 * Accepts an array of data containing a named filter array and converts to
 * a CakePHP based conditions array
 *
 * @param array $data The data to convert into a conditions array
 * @return array The conditions to pass through to the find declaration
 */
	protected function _buildExportConditions($data = []) {
		$Model = $this->{$this->modelClass};

		$conditions[$Model->alias . '.contact_form_id'] = $data[$Model->alias]['contact_form_id'];

		if (
			! empty($data[$Model->alias]['created_from'])
		) {
			$createdFrom = $data[$Model->alias]['created_from'];
			$conditions[$Model->alias . '.created >='] = CakeTime::toServer(
				$createdFrom . '00:00:00',
				Configure::read('Config.timezone'),
				'Y-m-d  H:i:s'
			);
		}

		if (
			! empty($data[$Model->alias]['created_to'])
		) {
			$createdTo = $data[$Model->alias]['created_to'];
			$conditions[$Model->alias . '.created <='] = CakeTime::toServer(
				$createdTo . '24:00:00',
				Configure::read('Config.timezone'),
				'Y-m-d  H:i:s'
			);
		}

		return $conditions;
	}

	/**
	 * admin view
	 *
	 * actually just a view with an "mark as actioned button"
	 */
	public function admin_view($id = null) {

		parent::admin_view($id);

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$formData = $Model->ContactForm->find('first', array(
			'conditions' => array(
				'ContactForm.id' => $this->request->data['ContactFormResponse']['contact_form_id']
			)
		));
		$this->set('fields', $this->_adminViewFieldsCustom($formData['ContactForm']['form_data']));


		return;
	}

	/**
	 * mark the message as actioned
	 *
	 * @param 	int 	contact form id
	 */
	public function admin_actioned($id)
	{

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$data = array(
			'ContactFormResponse' => array(
				'actioned' => date('Y-m-d G:i:s')
			)
		);
		$Model->id = $id;

		if ($Model->save($data)) {

			$this->Session->setFlash(array(
				'title' => "Message Updated",
				'description' => 'Message has been marked as actioned'
			), 'flash_success');

			$this->redirect(array(
				'action' => 'index'
			));

		} else {

			$this->Session->setFlash(array(
				'title' => 'Save Failed',
				'description' => 'Failed to mark the message as actioned'
			), 'flash_fail');

			$this->redirect(array(
				'action' => 'edit',
				$id
			));
		}
	}

	/**
	 * mark the message as unactioned
	 *
	 * @param 	int 	contact form id
	 */
	public function admin_unactioned($id)
	{

		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		$data = array(
			'ContactFormResponse' => array(
				'actioned' => ''
			)
		);
		$Model->id = $id;

		if ($Model->save($data)) {

			$this->Session->setFlash(array(
				'title' => "Message Updated",
				'description' => 'Message has been marked as unactioned'
			), 'flash_success');

			$this->redirect(array(
				'action' => 'index'
			));

		} else {

			$this->Session->setFlash(array(
				'title' => 'Save Failed',
				'description' => 'Failed to mark the message as unactioned'
			), 'flash_fail');

			$this->redirect(array(
				'action' => 'edit',
				$id
			));
		}
	}

/**
 * Enables an admin user to export the current admin index results
 *
 * @return void
 */
	public function admin_export() {
		$Model = $this->{$this->modelClass};

		// check that a form type has been selected before continuing
		if (empty($this->request->params['named']['Filter'][$Model->alias]['contact_form_id'])) {
			$this->Session->setFlash('Please select a form type before continuing', 'flash_fail');
			return $this->redirect($this->referer());
		}

		$contactFormId = $this->request->params['named']['Filter'][$Model->alias]['contact_form_id'];

		// fetch the contact form data for the requested form
		$contactFormData = $Model->ContactForm->find('first', [
			'conditions' => [
				'ContactForm.id' => $contactFormId
			]
		]);

		if (! empty($contactFormData['ContactForm']['form_data'])) {
			$formFieldsKey = 'EvContactForm.' . $contactFormData['ContactForm']['form_data'] . '.fields';
			$formFields = Configure::read($formFieldsKey);

			// remove the default message, origin and adwords_tracking fields so the
			// entries can be imported into 3rd party software easily
			unset(
				$formFields['message'],
				$formFields['origin'],
				$formFields['adwords_tracking']
			);

			$formFields = array_keys($formFields);
		}

		// send the user back to the referring page if no headings can be established
		if (empty($formFields)) {
			$this->Session->setFlash('An error occurred and the CSV file could not be created', 'flash_fail');
			return $this->redirect($this->referer());
		}

		// assign the CSV fiel headings
		$_header = $formFields;

		// build the find conditions
		$params['conditions'] = $this->_buildExportConditions($this->request->params['named']['Filter']);

		$contactFormResponsesData = $Model->find('all', $params);

		if (empty($contactFormResponsesData)) {
			$this->Session->setFlash('No data is available to export', 'flash_fail');
			return $this->redirect($this->referer());
		}

		$data = [];

		foreach ($contactFormResponsesData as $item) {
			if (! empty($item[$Model->alias]['form_data'])) {
				$itemFormData = @unserialize($item[$Model->alias]['form_data']);
				if (! empty($itemFormData)) {
					$row = [];

					foreach ($formFields as $formFieldsKey) {
						if (isset($itemFormData[$formFieldsKey])) {
							$row[] = $itemFormData[$formFieldsKey];
						} else {
							$row[] = '';
						}
					}

					$data[] = $row;
				}
			}
		}

		$_serialize = 'data';

		$this->viewClass = 'CsvView.Csv';

		$this->set(compact('data', '_serialize', '_header'));

		$this->response->download(
			str_replace(' ', '', $contactFormData['ContactForm']['form_data']) . '-' . date("j-M-Y") . '.csv'
		);
	}

/**
 * Manages a default admin index view of this model
 *
 * @return void
 */
	public function admin_index() {
		parent::admin_index();

		// set the filter field options
		$this->_adminPopulateFilterLookups();
	}

}
