<?php

	App::uses('CustomEmail', 'Lib');

	class InjectContactFormComponent extends Component {

		protected $Controller;

		const FORM_SUCCESS = 1;
		const FORM_ERROR = 2;
		const FORM_VALIDATION = 3;

		protected $form;

		public function initialize(Controller $Controller) {
			$this->Controller = $Controller;
			$this->form = null;
		}

		protected function _showFlash($status = '') {

			$defaultFlash = array(
				'message' => 'An error occurred',
				'element' => 'default',
				'params' => array(),
				'key' => 'flash'
			);

			switch ($status) {

				case self::FORM_SUCCESS:
					$flash = $this->form['config']['flash']['success'];
					break;

				case self::FORM_ERROR:
					$flash = $this->form['config']['flash']['errror'];
					break;

				case self::FORM_VALIDATION:
					$flash = $this->form['config']['flash']['validation'];
					break;
			}

			if (isset($flash) && !empty($flash)) {
				$defaultFlash = array_merge($defaultFlash, $flash);
			}

			$this->Controller->Session->setFlash(
				$defaultFlash['message'],
				$defaultFlash['element'],
				$defaultFlash['params'],
				$defaultFlash['key']
			);
		}

		protected function _handleValidation($fields, $data) {
			// setup the dynamic validation
			foreach ($fields as $field => $options) {

				if (!empty($options['validate'])) {

					foreach ($options['validate'] as $rule => $rule_data) {

						$this->Model->validator()->add($field, $rule, $rule_data);
					}
				}
			}
			// validate!
			return $this->Model->validator()->validateAssociated($data);
		}

		protected function _removeValidation($fields) {
			foreach ($fields as $field => $options) {

				if (!empty($options['validate'])) {

					$this->Model->validator()->remove($field);
				}
			}
		}

		protected function _saveResponse($form, $contact_form_data, $responseName) {

			$data = array(
				'ContactFormResponse' => array(
					'contact_form_id' => $form['ContactForm']['id'],
					'name' => $responseName,
					'form_data' => $contact_form_data,
					'created' => date('Y-m-d H:i:s')
				)
			);

			$this->Controller->request->data = array_merge($this->Controller->request->data, $data);

			return $this->Model->saveAll($this->Controller->request->data, array('deep' => true));
		}

		protected function _setEmailFrom() {
			return array(
				'name' => Configure::read('SiteSetting.site_title'),
				'email' => Configure::read('SiteSetting.admin_email'),
			);
		}

		protected function _sendNotificationEmail($email_from, $email_to, $form, $formData, $fields) {
			$email = new CustomEmail();
			$email->template('EvContactForm.contact_form_notification')
					->from($email_from['email'], (isset($email_from['name'])) ? $email_from['name'] : null)
					->emailFormat('html')
					->subject($form['ContactForm']['subject'])
					->viewVars(array(
						'form_data' => $formData,
						'fields' => $fields
			));
			// build the to / cc's
			$to = array_shift($email_to);

			$email->to($to);

			if (!empty($email_to)) {
				foreach ($email_to as $recipient) {
					$email->addCc($recipient);
				}
			}
			if (! empty($this->form['config']['attach_files_to_notification_email'])) {
				$this->_attachFilesToEmail($email, $this->Model->id);
			}

			$email->send();
		}

		/**
		 * Attach files to an email
		 * @param CustomEmail $email Email object
		 * @param int $contactFormId Contact form ID
		 * @return CustomEmail
		 */
		protected function _attachFilesToEmail(&$email, $contactFormId) {
			$this->Controller->loadModel('Document');
			$documents = $this->Controller->Document->find('all', [
				'conditions' => [
					'Document.model_id' => $contactFormId,
					'Document.model' => 'ContactFormResponse'
				]
			]);
			$attachments = [];
			$attachmentsSize = 0;
			foreach ($documents as $document) {
				$attachments[] = WWW_ROOT . $document['Document']['filepath'];
				$attachmentsSize += filesize(WWW_ROOT . $document['Document']['filepath']);
			}
			if (! empty($this->form['config']['notification_email_attachments_max_filesize'])) {
				$maxFilesize = $this->form['config']['notification_email_attachments_max_filesize'];
			}
			if (empty($maxFilesize) || $attachmentsSize <= $maxFilesize) {
				$email->attachments($attachments);
			}
			$email->viewVars(compact('documents'));

			return $email;
		}

		protected function _sendUserEmail($email_from, $email_to, $subject, $form_data, $fields) {
			$email = new CustomEmail();
			$email->template('EvContactForm.user_email_notification')
					->from($email_from['email'], (isset($email_from['name'])) ? $email_from['name'] : null)
					->emailFormat('html')
					->subject($subject)
					->viewVars(array(
						'form_data' => $form_data,
						'fields' => $fields
			));

			$email->to($email_to);
			$email->send();
		}

		protected function _loadForm($id) {

			$this->Controller->loadModel('ContactForm');
			$form = $this->Controller->ContactForm->find('first', array('conditions' => array(
					'ContactForm.id' => $id
			)));
			if (empty($form['ContactForm'])) {
				return false;
			}
			return $form;
		}

		protected function _loadConfig($form_name) {
			$config = Configure::read('EvContactForm');
			return $config[$form_name];
		}

		protected function _isSpamComment($form_data) {

			//Check if the commit is flagged as spam
			if (isset($form_data['email_confirm']) && !empty($form_data['email_confirm'])) {
				return true;
			} else {
				return false;
			}
		}

		public function display($id, $name, $modelName = 'EvContactForm.ContactFormResponse') {
			// Load configs, overridable!
			$this->form = $this->_loadForm($id);
			$formData = $this->_loadConfig($this->form['ContactForm']['form_data']);
			$this->form['config'] = $formData;

			$fields = $formData['fields'];

			if (
				$this->Controller->request->is('post') &&
				!empty($this->Controller->request->data[$name]) &&
				isset($this->Controller->request->data[$name]['contact-form-send']) &&
				$this->Controller->request->data[$name]['contact-form-send'] === $name
			) {

				$this->Model = ClassRegistry::init($modelName);

				if ($this->_handleValidation($fields, $this->Controller->request->data[$name])) {

					// passed validation, remove rules so we can add to the database
					$this->_removeValidation($fields);

					// serialize it
					$contact_form_data = serialize($this->Controller->request->data[$name]);

					// name label
					$name_lbl = $formData['name_label'];


					if (!empty($this->Controller->request->data[$name][$name_lbl])) {
						$responseName = $this->Controller->request->data[$name][$name_lbl];
					} else {
						$responseName = 'Unknown Name';
					}

					$this->Controller->set($name . '_completed', true);

					// save it or pretend to if it is spam
					if ($this->_isSpamComment($this->Controller->request->data[$name])) {

						//The comment has been flagged as spam so we pretend to save it but don't
						if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
							$this->_showFlash(self::FORM_SUCCESS);
						}

						$successRedirect = Configure::read('ContactForm.success_redirect');
						if (isset($successRedirect) && !empty($successRedirect)) {
							return $this->Controller->redirect($successRedirect);
						} else {
							$this->Controller->set($name . '_sent', true);
						}
					} elseif ($this->_saveResponse($this->form, $contact_form_data, $responseName)) {

						$email_from = $this->_setEmailFrom();

						// Email to can be overriden using site settings (for when we choose not to show the forms CMS section to clients)
						if (Configure::read('SiteSetting.' . $name . '_email_to')) {
							$email_to = explode(',', Configure::read('SiteSetting.' . $name . '_email_to'));
						} else {
							$email_to = explode(',', $this->form['ContactForm']['email_to']);
						}

						if (!is_null($email_from) && !is_null($email_to)) {

							// send the email!
							$this->_sendNotificationEmail($email_from, $email_to, $this->form, $this->Controller->request->data[$name], $fields);
						}

						// If this form is configured to send an email to the user to say thanks
						if (isset($formData['user_email_notification']) && is_array($formData['user_email_notification'])) {
							$userEmailConfig = $formData['user_email_notification'];

							// Get the user's email, so we can mail them and say thanks
							$userEmailAddress = array($this->Controller->request->data[$name]['email']);
							$userFromEmailAddress = (isset($userEmailConfig['from']) ? $userEmailConfig['from'] : $this->_setEmailFrom());
							$subject = $userEmailConfig['subject'];
							$this->_sendUserEmail($userFromEmailAddress, $userEmailAddress, $subject, $formData, $fields);
						}

						// unset form data and show success
						unset($this->Controller->request->data[$name]);
						if (isset($formData['flash']['success']) && !empty($formData['flash']['success'])) {
							$this->_showFlash(self::FORM_SUCCESS);
						}
						$successRedirect = Configure::read('ContactForm.success_redirect');
						if (isset($successRedirect) && !empty($successRedirect)) {
							return $this->Controller->redirect($successRedirect);
						} else {
							$this->Controller->set($name . '_sent', true);
						}
					} else {
						if (isset($formData['flash']['error']) && !empty($formData['flash']['error'])) {
							$this->_showFlash(self::FORM_ERROR);
						}
						// failed to save, error
						$contact_form_errors = array(
							'There was an issue saving your message, please try again'
						);

						$this->Controller->set($name . '_errors', $contact_form_errors);
					}
				} else {

					// compile the validate errors
					$this->_showFlash(self::FORM_VALIDATION);

					$contact_form_errors = array();

					foreach ($this->Model->validationErrors as $field => $errors) {
						foreach ($errors as $msg) {
							$contact_form_errors[] = $msg;
						}
					}

					// I dont know why I need this
					if (isset($this->Controller->request->data[$name][$name])) {
						$temp = $this->Controller->request->data[$name][$name];
						$this->Controller->request->data[$name] = $temp;
					}
					$this->Controller->set($name . '_errors', $contact_form_errors);
				}
			}

			$this->Controller->set($name . '_form', $formData);
		}

	}
