# Evoluted coding standards

Because we all have standards.

1. [Backend / PHP](#backend-php)
2. [Frontend](#frontend)
    1. [ESLint](#eslint)
    2. [Stylelint](#stylelint)

If you only need frontend standards just follow [that section](#frontend), you can install them without needing to install the [backend / PHP ones](#backend-php)

# Backend / PHP

## Quick start

* Grab the latest version of [PHP-CS](https://github.com/PHPCSStandards/PHP_CodeSniffer)
* Add this repo to your project: `composer require --dev evoluted/coding-standard:version` - _always lock to a specific version_
* Add a `phpcs.xml.dist` file to the root of your project, make it look something like [`phpcs-sample.xml`](phpcs-sample.xml)
* Run `phpcs`

## About

[PSR-2 coding standards](https://github.com/php-fig/fig-standards/blob/master/accepted/PSR-2-coding-style-guide.md) are a great starting point for standards but are, by their nature, very broad.

**[Very broad](examples/BadPsr2.php)**

The Evoluted PHP coding standard builds on PSR-12 with a series of standards aimed at making all the code we produce similarly styled. This means that when it comes to code reviews or bug fixes you can focus on what the code does, and not about how the code looks.

## Details

The Evoluted coding standards are:

* Built on top of PSR-12
* Designed to reduce the dissonance when switching projects
* For new and modern projects only - Cake has its own *particular* style
* Built upon consensus and best practice
* Runnable from the PHP-CS command line tool
* Versioned
* Always evolving

## Current standards

* [PSR-2](https://www.php-fig.org/psr/psr-2/)
* [PSR-12](https://www.php-fig.org/psr/psr-12/)
* [Slevomat coding style](https://github.com/slevomat/coding-standard)

## Issues

If upgrading from version 3.x to 4.x of PHP-CS (or the latest version of the coding standards) you may get the error:

```
ERROR: Referenced sniff "SlevomatCodingStandard.TypeHints.ParameterTypeHint" does not exist.
<other similar errors>
```

The solution is to remove the following line from your project `phpcs.xml.dist` file:

```xml
<config name="installed_paths" value="../../evoluted/coding-standard"/>
```

As long as the "SlevomatCodingStandard" and "Evoluted" coding standards are installed and accessible by PHP-CS (check by running `vendor/bin/phpcs -i` from your project directory) this should remove the errors. This line is no longer included in the recommended [`phpcs.xml.dist`](phpcs-sample.xml) file.

# Frontend

To use our standards in your project:

```shell
npm install --save-dev git+https://git.evoluted.net/evoluted-public/coding-standard.git 
```

or if you're using yarn as your project's package manager:

```shell
yarn add --dev git+https://git.evoluted.net/evoluted-public/coding-standard.git 
```

You will want to lock the version of the coding standard to a specific version so that newer versions - with different linting rules - aren't applied incorrectly. To do this, append the version with a `#` at the end of the URL:

```
git+https://git.evoluted.net/evoluted-public/coding-standard.git#v2.1.0
```

If you do this, be sure to apply the version to the `.gitlab-ci.yml` instance of `npm install`.

## ESLint

We use [ESLint](https://eslint.org/) to check JavaScript / TypeScript code.

Create a `.eslintrc.json` in the root of your project and make it look like:

```json
{
    "root": true,
    "extends": [
        "./node_modules/evoluted-coding-standard/.eslintrc.json"
    ]
}
```

To run ESLint from the command line, add an entry in your `package.json`'s `script` section:

```json
{
   "scripts": {
      "check:script": "eslint --color <your files go here>"
   }
}
```

Then you can run `npm run check:script` or `yarn check:script` to run the ESLint command locally.

Check out the [ESLint CLI documentation](https://eslint.org/docs/user-guide/command-line-interface) for how to specify multiple files / directories, e.g. `resources/assets/site/js/**/*.scss`. You may also want to set up your IDE to automatically lint your code, there are plugins for [VSCode](https://marketplace.visualstudio.com/items?itemName=dbaeumer.vscode-eslint) and [Jetbrains](https://plugins.jetbrains.com/plugin/7494-eslint).

To run ESLint during the CI/CD pipeline, use the following config in your project's `.gitlab-ci.yml` file:

```yaml
stages:
  - linting
  # other stages

eslint:
  stage: linting
  image: registry.evoluted.net/evoluted/coding-standard/eslint:latest
  script:
    - npm install 'git+https://://git.evoluted.net/evoluted-public/coding-standard.git'
     # match this to what's run in package.json
    - eslint --color <your files go here>
  artifacts:
    expire_in: 1 week
```

Make sure the path that you use in your `package.json` script matches the one in `.gitlab-ci.yml`.

## Stylelint

We use [Stylelint](https://stylelint.io/) to check CSS / SCSS code.

Create a `.stylelintrc` in the root of your project and make it look like:

```json
{
   "extends": [
      "./node_modules/evoluted-coding-standard/.stylelintrc"
   ]
}
```

To run Stylelint from the command line, add an entry in your `package.json`'s `script` section:

```json
{
   "scripts": {
      "check:style": "stylelint <your files go here>"
   }
}
```

You can then run `npm run check:style` or `yarn check:style` to run the Stylelint command locally.

Check out the [Stylelint CLI documentation](https://stylelint.io/user-guide/usage/cli/#usage-examples) for how to specify multiple files / directories e.g. `resources/assets/site/scss/**/*.scss`. You may also want to set up your IDE to automatically lint your code, there are plugins for [VSCode](https://marketplace.visualstudio.com/items?itemName=stylelint.vscode-stylelint) and [Jetbrains](https://plugins.jetbrains.com/plugin/9276-intellij-stylelint-plugin).

To run Stylelint during the CI/CD pipeline, use the following config in your `.gitlab-ci.yml` file:

```yaml
stages:
  - linting
  # other stages

stylelint:
  stage: linting
  image: registry.evoluted.net/evoluted/coding-standard/stylelint:latest
  script:
    - npm install 'git+https://://git.evoluted.net/evoluted-public/coding-standard.git'
    # if you're using the webpack plugin, be sure to use your "context" path below
    - stylelint --color --config='.stylelintrc' <your files go here>
  artifacts:
    expire_in: 1 week
```

Make sure the path that you use in your `package.json` script matches the one in `.gitlab-ci.yml`.

If you have files that you want to ignore, you **cannot** place them within the `.stylelintrc` file, you have to either specify them on the command line above, or (better), create a `.stylelintignore` file with the paths / patterns to the files.
