# EvCheckout

## Installation

Simply add EvCheckout to the `install.php` file within the Config directory and run `Console/cake installer core` once you have added all the other plugins you will need.

If this has already been run, simply run `Console/cake installer plugin EvCheckout `.

The installer will automatically setup the database tables, add a link within the admin to view orders, setup templates for success, fail, cancelled order pages and writes the ID numbers to an `ev_checkout.php` config. It finally then adds in some default order statuses along with some site settings.

## How it works

The checkout plugin works on the concept of "stages" for the main checkout process, with each stage being made up of a template and then `PreStage` and `PostStage` methods which are called to make each stage function. This should allow for a totally customisable checkout process for each and every site. The process works by making sure each stage submits a form back to itself, the stage handler will take care of the processing from then on.

The `PreStage` methods are called on every page load of that stage, the stage will load the list of items and loop through in order to process. If a single method returns false the stage is aborted and it will go back one stage.

The `PostStage` methods are called upon POSTing to a stage. These are then used to check the submitted data, if a single method returns false then that stage is reloaded from the beginning allowing you to go back and show things such as validation errors. If every stage returns true, the system will try to get the next Stage and redirect to it, in the event the current stage is the last one it will redirect to an Order success page instead.

Upon completion of the last stage as defined by the config, `EvCheckout` will load the order success page upon which the order is built and saved into the system. To improve flexibility the concept of `OrderBuilders` was introduced.

The `OrderBuilder` is defined within a config and is essentially a class which is run by `EvCheckout` upon an order being completed, with the class essentially building and saving the order. This concept was introduced to allow this Checkout plugin be used in various ways, simply be creating a new class to turn a membership subscription request into a valid order and actually create the subscription (for example). Built in is an OrderBuilder to transform an `EvBasket` basket into a full Order.

The checkout process also triggers various events at the end of a checkout process allowing you to hook into these events to add extra functionality. It will also send out admin and user email notifications of the order, using the Queue plugin in doing so.

A toggle (`allowOrderItemStatuses`) can be found and set in the config file of the plugin to turn on order statuses for individual order items, which runs independantly of the traditional order order status configuration. Switching the `allowOrderItemStatuses` toggle to `true` displays a select input element alongside each order item on the `admin_edit` template.

## Setup

Once the installer has run and the config file generated, there are a few config items needing setting up.

The first is `stages`, this will be an associative array defining all the stages that need to be run for this process, the template it should use and all the Pre / Post Stage methods.

You can also define multiple processes within this array should the checkout have a standard checkout and a 1 page checkout setup.

The following is a default example which starts with a login / register page combo, delivery address selection page and finally a confirmation page. All the templates are relative to `/Plugin/EvCheckout/View` or `/app/View/Plugin/EvCheckout` in the event of overridden templates.

	'stages' => array(
		// this is the name of stage, for use when setting the default stage to use or switching processes
		'default' => array(
			// key is the name of this stage and is used in the url /checkout/stage/login
			'login' => array(
				// any PreStage methods to run, this checks to see if they are logged in
				// if so we can skip this stage
				'pre' => array(
					'alreadyLoggedIn'
				),
				// define the PostStage methods to run, this will check if we are registering
				// or logging in and process accordingly.
				'post' => array(
					'processRegister',
					'processLogin'
				),
				// define the template we wish to use
				'template' => 'Checkout/login'
			),
			// checkout/stage/delivery-address
			'delivery-address' => array(
				// PreStage - check we are logged in, and load all the users addresses if so
				'pre' => array(
					'authUser',
					'loadAddresses'
				),
				// PostStage - process delivery addresses, either creating new one or selecting
				// an existing address
				'post' => array(
					'processDeliveryAddress'
				),
				'template' => 'Checkout/delivery_address'
			),
			// checkout/stage/confirm
			'confirm' => array(
				// PreStage - check the user is logged in, load the selected addresses
				// Auto process the users shipping now we have loaded the address
				// Reload the basket to show confirmation page
				'pre' => array(
					'authUser',
					'loadSelectedAddresses',
					'processBasketShippingFromAddress',
					'loadBasket'
				),
				// PostStage - We need to punch out to Paypal so build up the transaction from
				// a basket row and send to Paypal
				'post' => array(
					'buildTransactionFromBasket'
				),
				'template' => 'Checkout/basket_confirm'
			)
		)
	),

The second attribute to define will be the default process to use if none are defined within the URL / session. `stageProcess` holds the default process to use which itself is labeled 'default' so amend if you change your process name from above.

The third attribute to define is the order builder to use. These are defined in PluginName.OrderBuilderName format. These are located either at `/app/Lib/OrderBuilder/OrderBuilderName.php` or `/Plugin/PluginName/Lib/OrderBuilder/OrderBuilderName.php`

A fourth optional parameter you can configure (more exist but are self explanatory) is `orderItemData`. The attribute allows you to define an extra data the order needs to try and save. Using the Basket Builder as an example, when saving the basket into an Order we are only able to store the basic details of name, unit prices, quantity etc... In some cases you may wish to store extra data such as the SKU or the weight / dimensions of an item and save this on the order in the event that the details eventually changed.

This is defined in the format of having the model name as the array key and the value being a sub array containing a label and the Hash path Syntax to retrieve that data.

The following example is from Brighton Tools and is used to retrieve the SKU code for each variant when we transform it into an Order.

	'orderItemData' => array(
		'EvShop.Variant' => array(
			'sku' => 'Variant.sku'
		)
	),

## Usage

To get started on the checkout process you can simply redirect the customer to `/checkout/`, either via the string or the routing array

	array(
		'plugin' => 'ev_checkout',
		'controller' => 'checkout',
		'action' => 'index'
	);

Then make sure each stage contains a form which posts to itself and the stage handler will take care of the rest.

When using the `OrderBuilder` as described below, there is an `OrderManager` Component available which can be used to perform a range of actions from setting up and creating an order to marking the order as paid and triggering the events associated with that.

Bundled is also an Order History section that can be used on the front end as an extension to the my account section that comes with EvCore. This can be linked with the address management functionality bundled with EvAddressBook.

## Assigning Pages
Sometimes you might want to pull page information into a checkout step. You can do this by adding a `assignPage` into your stage. e.g. 

	'stages' => [
		'default' => [
			'login' => [
				'assignPage' => 'Checkout Login'
			],
		],
	],
		
The `Checkout Login` content is now available within the template under `$checkoutPage`

## OrderBuilder

To create an OrderBuilder simply create a class at the following location `/app/Lib/OrderBuilder` with the class name matching the filename as per CakePHP standards.

* MyBuilder -> /app/Lib/OrderBuilder/MyBuilder.php

The class should extend the base order builder and implement the `OrderBuilderInterface`, this ensures your Order Builder will have all the methods needed to run correctly.

Below is an example class to get started.

	<?php

	App::uses('OrderBuilderInterface', 'EvCheckout.Lib');
	App::uses('OrderBuilder', 'EvCheckout.Lib');

	class MyBuilder extends OrderBuilder implements OrderBuilderInterface {

		/**
		 * build an order from a basket
		 *
		 * @param 	int 	$transactionId The successful transaction ID just completed
		 * @return  int 	$orderId 			The id number of the order we just created
		 */
		public function build($transactionId) { }
	}

The builder method is the one which the Order Processor will attempt to load when trying to build up the order. As part of the base `OrderBuilder` class the constructor for it automatically loads up the Controller from which called it (usually `OrdersController`) and also it will initiate and load the `OrderManager` Component and make it available at `$this->_controller->OrderManager` from within the builder allowing you to build the checkout into an actual Order.

## Custom Stage Methods

You can create custom stage methods by creating the method in an extended version of the PreStage/PostStage componenet and adding the method to your stages config array. Access to the controller is available via `$this->_controller` and checkout flow is controlled by returning true (to continue with the stage) or false (to return to the current stage). An array of attributes can also be returned to effect the current method loop, the following attributes are available:
 - success: True if the method succeeded, false otherwise.
 - skipAll: Skip any remaining methods, returning the current overall method success.
 - skipNext: skip the next method in the loop.

If returning an array of attributes from a custom stage method, ensure to set the success attribute, if it is missing then it is assumed that the method has failed.

## PreStage Methods

Below is a list of the built in available `PreStage` methods.

### alreadyLoggedIn

Used to check if the customer is already logged in, automatically redirects the user to the next stage if so

### authUser

Used to protect that stage from used that are not yet logged in.

### loadAddresses

Used with `EvAddressBook`. Loads all a logged in users stored address and sets to template so they can select an existing address without adding a new one.

Used in conjunction with PostStage - processDeliveryAddress / processBillingAddress(TBC)

### processBasketShippingFromAddress

Used with `EvBasket` and `EvShipping`. Used to calculate the shipping based on the users basket and the selected delivery address

### processBasketVATFromAddress

Used with `EvBasket` to check whether the current customer's delivery address is non-eu. When the customer is non-eu based a session variable of `EvCheckout.omitOrderVat` is set to `true`, all VAT is then removed from the order when the `removeVatOutsideEU` is also set to `true`.

### loadBasket

Used with `EvBasket`. Loads a users basket for use on a confirm page.

### loadSelectedAddresses

Load the saved addresses within the session `EvCheckout.address`, for displaying on the page.

## PostStage Methods

Below is a list of the built in available `PostStage ` methods.

### processRegister

Used to process a new users registration. Can handle full registration or guest checkout

### processLogin

Used to login an existing user

### processDeliveryAddress

Used to process a delivery address, either by selecting an existing one or adding a new one and then selecting it.

Used with `EvAddressBook` and the PreStage method - loadAddresses

### buildTransactionFromBasket

Used in conjunction with `EvBasket`, `EvAddressBook` and `EvTransactions`. This gets the basket and selected addresses and passes it all to `EvTransactions` Component and the `takePayment()` method. This will either redirect off site to Paypal type gateway or potentially return to the method for the checkout handler to process the redirecting to the success page.

#### Working with multiple payment gateways

The `_processPaymentTaking` method is available for extension, called from within `buildTransactionFromBasket`. The plugin is configured to work with one payment gateway as default but a second payment gateway can be simply added by checking the `$cardDetails['PaymentMethod']['name]` value. The following example shows how Puckstop allows for Paypal and Sagepay gateways.

		protected function _processPaymentTaking(
			$transactionData = array(),
			$modelId = null,
			$requestData = array(),
			$additionalData = array()
		) {
			// get the order totals; NOT basket
			$transactionTotals = $this->_controller->OrderManager->buildTransactionTotals();

			if ($requestData['PaymentMethod']['name'] == 'paypal') {
				// assign payment method specific $data here
				...

				// assign payment gateway
				$gateway = 'EvPaypal.Paypal';
			} elseif ($requestData['PaymentMethod']['name'] == 'card') {
				// assign payment method specific $data here
				...

				// assign payment gateway
				$gateway = 'EvSagePay.SagePay';
			}

			// take the payment
			$this->_controller->TransactionsManager->takePayment(...
		}

## Sessions

### EvCheckout.stageProcess

A session can be set with this name with the contents of the session being the key of one of the stage processes from the EvCheckout Config.

This should allow things like a custom admin process to be run.

Upon completion of any Checkouts in that scenario remember to delete the session

### EvCheckout.address

A session set to hold to the selected delivery / billing address ID numbers (used in conjunction with EvAddressBook).

* EvCheckout.address.delivery - contains the id selected for delivery address
* EvCheckout.address.billing - contains the id selected for billing address

### EvCheckout.omitOrderVat

Set to `true` when the customer is non-eu based and the `removeVatOutsideEU` config variable is also set to `true`.

## Events

### EvCheckout.Component.Order.paid

Called by the OrderManager component when an order has been marked as paid

#### Parameters

* `orderId` - The order that has been marked as paid

### EvCheckout.Component.Order.statusChanged

Called by the OrderManager component when an orders status has been updated

#### Parameters

* `orderId` - The order that has been updated
* `statusId` - The new order status id

### EvCheckout.Controller.Order.success

Called by OrderController::complete() when an order was successfully processed

#### Parameters

* `orderId` - The order that has been created

### EvCheckout.Controller.Order.fail

Called by OrderController::complete() when an order failed to process

#### Parameters

No parameters passed

### EvCheckout.Controller.Order.cancel

Called by OrderController::cancel() when an order was cancelled

#### Parameters

No parameters passed

### EvCheckout.Component.Order.statusChanged

Called from the Order Items Controller and can used to update the Order's status

#### Parameters

* `OrderItem` - Contains the order items that have been update - includes each order item's parent order_id


#Additional Features

## Order Data Table
The order data table is used to store any 'extra' details about an order. So this might include things like the discount code used, or any special custom fields you might need to be storing against the order.

This works in conjunction with the data table used on EvBasket. Anything in the EvBasket one gets mirrored over to EvCheckout's order data table.

Examples of where this could be useful:

- Storing an order note that the customer might need to provide
- Storing any special delivery information

Each item can be optionally marked as visible. When this option is true, the data row will be shown on the admin edit page. It'll automatically format the data by using the cakephp inflector. For example the discount one that's already built in, will store data as:

- Name: 'discount_code'
- Data: 'MYDISCOUNTCODE'

and output it as:

**Discount Code:** MYDISCOUNT CODE

## Removing VAT for Non-EU based customers

To remove VAT from all order items and order totals for non-eu based customers the `removeVatOutsideEU` config toggle should be set to `true`. It is also important to remember to add the `EvCheckout.PreStageComponent:: processBasketVATFromAddress` to the app based EvCheckout stage process array.

# Status Update Emails

## Default emails
When order statuses are updated, they can trigger an email to be sent. By default emails are sent when an order's status is changed to paid or shipped. The emails are created using EvEmail and so therefore can be edited via the CMS.

The update to paid emails are the same as the original emails that were sent when an order was markedAsPaid. To be more flexible, the original emails were broken up into elements and can be used on any future template.

## Creating A New Template
So you have a custom order status and want to set up sending emails when it is updated to? Emails can be added through the ev_checkout config.

First create an email template in the CMS. You will want to remember the System Name you have added to use later. For a better explanation of how to use EvEmail templates visit the plugin.
If you want to add formatted data then you will need to add a "{==content==}" token to the email content. This is the part of the email that is replaced with the rendered templates.

For other tokens just add what you want it to be called inside curly braces.

The rest of the work will be done inside the ev_checkout config file. For examples use the base templates inside the EvCheckout config.
The jist is as follows:

	'statusUpdateEmails' => array(
		YourCustomStatus => array(
			array(
				'target' => 'admin',
				'emailTemplate' => (Enter Your EvEmail Template System Name Here),
				'headViewTemplates' => array(
					(Enter your own templates here using their directory listing - These will be rendered in the head of the email),
					'EvCheckout.Emails/html/Orders/orderDeliveryAddress'
				),
				'viewTemplates' => array(
					(Enter your own templates here using their directory listing),
					'EvCheckout.Emails/html/Orders/orderDeliveryAddress'
				),
				'viewLayout' => 'default',
				'tokens' => array(
					(Enter your own token name here) => (Enter the value from the order aray you wish to display in dot notation)
					'customerName' => 'Order.name',
					'orderNumber' => 'Order.id'
				)
			)
		)
	)

## Addresses

From version 2.4.10.* order addresses will also be stored in JSON format in the database to preserve access to individual fields. They can be found in the OrderData with names `delivery_address` and `billing_address`.

## Returning Items

Items can be returned if the `EvCheckout.allowReturningItems` flag is set to true. This creates a link on the CMS order page where the user can enter a quantity to return. The return works by adding an `OrderItemData` named `quantity_returned` with the number returned. If the inventory plugin is loaded, EvCheckout will attempt to increase the stock level if it can find the original variant from the `OrderItem`. If it can't, an additional `OrderItemData` will be saved with the name `return_stock_updated` set to false and show a warning message in the CMS.

## Order Notes

There is a flag in the config that will switch the order notes from using 2 columns for all the notes to using a separate table. The admin_edit and view order pages will update to reflect the change to notes.

## Editing Orders

When used with the latest versions of `EvBasket` and `EvDiscount`, there is now the ability to edit order items. This is done by deconstructing an order back in to a basket, modifying the items either using the basket on the front end or via another project specific means, and then updating the existing order using the modified basket. To help with this, new actions have been added to the `OrdersController` and `OrderManagerComponent`. Order editing begins by clicking the edit order items button in the toolbar when viewing an order in the CMS.

### Order editing sessions

When editing an order, a new temporary basket is created and assigned to an editing session. These sessions will last either a set amount of time as defined in the config (1 hour by default) or until the session is ended manually. Only one editing session per order should be active at a time. A warning is displayed in the CMS when an order is currently being edited. `OrderEditingComponent::checkForEditOrder` can be used to check for an active editing session. `EvBasket` will automatically detect an editing session and load the order basket in place of the regular basket. While an order is being edited on the front end, the `EvCheckout.Orders/edit-order-banner` element should be added to the top of the page. To allow the banner to work properly, the output of `OrderEditingComponent::checkForEditOrder` needs to be set as view variable named `editOrder`.

### Saving changes

When the basket is ready to be saved, direct the user to `EditOrderItemsController::admin_save`. To exit a session without saving anything send the user to `EditOrderItemsController::admin_cancel`.

### Emailing the customer

After a session is saved, the admin user is directed to a confirmation page with the option to notify the user via email. The email template should have been added automatically through a migration. To configure the email further, update the `EvCheckout.orderUpdatedEmail` section in the config. This uses the same format as the status emails.

### Setup
- Ensure `EvBasket` is added to the project.
- Enable editing orders in the EvCheckout config.
- Set the order editing component in `EvBasket`. This will usually be set to "EvCheckout.OrderEditing"
- `OrderEditingComponent` should be loaded in `AppController` so it can be used to detect order editing sessions.
- In `AppController::beforeRender` the output of `OrderEditingComponent::checkForEditOrder` should be set as a view variable named `editOrder`
- Add the order editing banner (`EvCheckout.Orders/edit-order-banner`) to the layout.
- The front end will need to be adapted to cope with editing orders. The main change will be swapping the last step of the basket process to save the basket changes rather than progressing to the checkout. See above for how to do this.
- If using overridden controllers or to use a different flow, it may be necessary to adjust some of the paths in the config.

### Limitations

- While applied discounts will be respected, this also allows for any existing discount code to be applied. As this functionality is limited to cms users it shouldn't be a problem.
- If a discount has changed (e.g. 10% off to 15% off) the new value of the discount will be used when the basket is updated.
- Promotions are not accounted for so currently behave with the same rules as making a new order. If a promotion has expired it will not apply to the basket. In the same way, new valid promotions that were not previously applied will apply to the new basket.
- This currently only works on single currency sites. To implement multi-currency support in future, the editing session will need to be set to the order currency.
- This is only for editing order items, their associated data and recalculating the totals. Addresses and other order data will not be updated.
- Depending on the site's basket implementation, the delivery address may need to be scraped from the order to calculate delivery properly in the order editing basket.

### Additional Google Analytics stuff
If you have additional Google Analytics stuff that you should only be fired once, you can put it in the
`additional-ecommerce-tracking` view block which is called from `Elements/ga-ecommerce-tracking`.
