<?php

App::uses('AppHelper', 'View/Helper');

class OrderHelper extends AppHelper {

	public $helpers = ['Number'];

/**
 * Get a specific item of order data from the order data array.
 *
 * @param array $orderData The order data array as returned as part of an order.
 * @param array $dataName  The name of the data you want to get.
 * @return string|array The data if found, false if not found and null if no order data was provided.
 */
	public function getOrderData($orderData, $dataName) {
		if (empty($orderData)) {
			return null;
		}

		return ArrayUtil::findFirst($orderData, 'name', $dataName, 'data');
	}

/**
 * Get a specific item of order item data from the order item data array. Order item data
 * is organised in the afterfind so that the keys are the names so just need to check and pull
 * out directly.
 *
 * @param array $orderData The order item data array as returned as part of an order item.
 * @param array $dataName  The name of the item data you want to get.
 * @return string|array The data if found, false if not found and null if no order data was provided.
 */
	public function getOrderItemData($orderItemData, $itemDataName) {
		if (empty($orderItemData[$itemDataName])) {
			return null;
		}

		return $orderItemData[$itemDataName];
	}

/**
 * Get a total from the order totals array.
 *
 * @param array  $orderTotals The order totals array as returned as part of an order.
 * @param string $totalName   The name of the total you want to get.
 * @param string $currency    The currency code.
 * @param bool   $includeTax  If the total shown should be the total including tax.
 * @return string The formatted total or null if not found.
 */
	public function getOrderTotal($orderTotals, $totalName, $currency, $includeTax = true) {
		$value = $this->getOrderTotalValue($orderTotals, $totalName, $includeTax);
		if ($value !== null) {
			return $this->Number->currency($value, $currency);
		}

		return null;
	}

/**
 * Gets the value of an order total
 *
 * @param array  $orderTotals The order totals array as returned as part of an order.
 * @param string $totalName   The name of the total you want to get.
 * @param bool   $includeTax  If the total shown should be the total including tax.
 * @return float
 */
	public function getOrderTotalValue($orderTotals, $totalName, $includeTax = true) {
		if (empty($orderTotals)) {
			return null;
		} else {
			foreach ($orderTotals as $total) {
				if ($total['name'] === $totalName) {
					if ($includeTax && !empty($total['display_inc_tax']) && $total['display_inc_tax'] > 0) {
						$total = $total['display_inc_tax'];
					} else {
						$total = $total['amount'];
					}
					return $total;
				}
			}
		}

		return null;
	}

/**
 * Get the price to display for an item. The item only needs to be provided if you want to only get the price of the
 * item. If EvBulkDiscount is installed then you can use the additional parameters to get the original price.
 * @param  array   $item             The array of a single order item.
 * @param  boolean $getOriginalPrice True if you want the original price of an item. False by default.
 * @param  array   $orderData        Only provide if you want the original price. The order data array as provided
 *                                   as part of an order.
 * @return string                    The price to display.
 */
	public function displayOrderItemUnitPrice($item, $getOriginalPrice = false, $orderData = []) {
		//Get the original prices from the order data
		if ($getOriginalPrice && !empty($item['model']) && !empty($item['model_id']) && !empty($orderData)) {
			$orderItemPath = $item['model'] . '.' . $item['model_id'];

			$originalPriceData = $this->getOrderData($orderData, 'original_price');

			if (!empty($originalPriceData) && !is_array($originalPriceData)) {
				$originalPriceData = json_decode($originalPriceData, true);

				if (!empty($originalPriceData[$orderItemPath]['unit_price_incTax'])) {
					return $originalPriceData[$orderItemPath]['unit_price_incTax'];
				}
			}
		}

		//Get the unit price for an item. If the taxed value is available then use that. There are potentially 2 tax
		//fields because the basket incTax field is calculated so doesn't include the underscore whereas the order
		//in_tax field is a database field so is underscored. This helper needs to work with both for compatability.
		$price = $item['unit_price'];

		if (CakeSession::read('EvCheckout.omitOrderVat') !== true) {
			if (!empty($item['unit_price_inc_tax']) && $item['unit_price_inc_tax'] > 0) {
				$price = $item['unit_price_inc_tax'];
			} elseif (!empty($item['unit_price_incTax']) && $item['unit_price_incTax'] > 0) {
				$price = $item['unit_price_incTax'];
			}
		}

		return $price;
	}

/**
 * Get a set of notes based on whether the admin notes or customer notes are required. Customer notes are
 * extracted by default.
 *
 * @param array $notes An array of all notes to extract from.
 * @param bool  $admin Whether the admin notes or customer notes want to be returned.
 * @return array Extracted notes.
 */
	public function getOrderNotes($notes, $admin = false) {
		return Hash::extract($notes, '{n}[is_admin_note=' . ($admin ? '1' : '0') . ']');
	}

/**
 * Inject order form/tabs on to an admin page. Requires the order to be set and for it to include the order
 * notes to display.
 *
 * @return string Html string of the form content to inject.
 */
	public function injectAdminForm() {
		$injectHtml = '';

		if (!empty(Configure::read('EvCheckout.useSeparateOrderNotes'))) {
			$injectHtml .= $this->element(
				'form_tab',
				[
					'title' => 'Admin Notes',
					'tabContent' => $this->element(
						'EvCheckout.Orders/separate-order-notes',
						[
							'admin' => true,
						]
					),
					'divOptions' => [
						'class' => 'field-order-admin-notes',
					]
				]
			);

			$injectHtml .= $this->element(
				'form_tab',
				[
					'title' => 'Customer Notes',
					'tabContent' => $this->element(
						'EvCheckout.Orders/separate-order-notes',
						[
							'admin' => false,
						]
					),
					'divOptions' => [
						'class' => 'field-order-notes',
					]
				]
			);
		}

		return $injectHtml;
	}
}
