<?php
App::uses('EvCheckoutAppModel', 'EvCheckout.Model');

/**
 * Order Model
 *
 * @property OrderStatus $OrderStatus
 * @property OrderItem $OrderItem
 * @property OrderTotal $OrderTotal
 */
class Order extends EvCheckoutAppModel {

/**
 * Display field
 *
 * @var string
 */
	public $displayField = 'name';

/**
 * inject currency details
 * @var array
 */
	public $actsAs = array(
		'EvCurrency.Currency' => array(
			'formInject' => true
		)
	);

/**
 * Validation rules
 *
 * @var array
 */
	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
		'email' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
			'email' => array(
				'rule' => array('email'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
		'order_status_id' => array(
			'numeric' => array(
				'rule' => array('numeric'),
				//'message' => 'Your custom message here',
				//'allowEmpty' => false,
				//'required' => false,
				//'last' => false, // Stop validation after this rule
				//'on' => 'create', // Limit validation to 'create' or 'update' operations
			),
		),
	);

	//The Associations below have been created with all possible keys, those that are not needed can be removed

/**
 * belongsTo associations
 *
 * @var array
 */
	public $belongsTo = array(
		'OrderStatus' => array(
			'className' => 'EvCheckout.OrderStatus'
		),
		'Currency' => array(
			'className' => 'EvCurrency.Currency'
		)
	);

/**
 * hasMany associations
 *
 * @var array
 */
	public $hasMany = array(
		'OrderItem' => array(
			'className' => 'EvCheckout.OrderItem'
		),
		'OrderTotal' => array(
			'className' => 'EvCheckout.OrderTotal'
		),
		'OrderUpdate' => array(
			'className' => 'EvCheckout.OrderUpdate'
		),
		'OrderData' => array(
			'className' => 'EvCheckout.OrderData'
		),
		'OrderNote' => array(
			'className' => 'EvCheckout.OrderNote'
		),
		'OrderEditingSession' => [
			'className' => 'EvCheckout.OrderEditingSession',
		]
	);

/**
 * Add the condition for session expiry in at runtime
 */
	public function __construct($id = false, $table = null, $ds = null) {
		// Sessions can only last for a given amount of time. Set in the config.
		$this->hasMany['OrderEditingSession']['conditions']['created > '] = date('Y-m-d H:i:s', strtotime(Configure::read('EvCheckout.editOrderSessionExpiry')));

		return parent::__construct($id, $table, $ds);
	}

	/**
	 * update the order, marking it as paid
	 *
	 * @param 	int 	$orderId 	The order ID to update
	 * @return 	bool
	 */
	public function paid($orderId) {
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		return $this->save(
			array(
				'Order' => array(
					'id' => $orderId,
					'is_paid' => 1,
					'order_status_id' => $OrderStatus->getStatusIdBySlug(Configure::read('EvCheckout.statusSlugs.paid'))
				)
			)
		);
	}

	/**
	 * update the order, amended order status
	 *
	 * @param 	int 	$orderId 	The order ID to update
	 * @param 	int 	$statusId 	The status ID to set
	 * @return 	bool
	 */
	public function updateStatus($orderId, $statusId) {
		return $this->save(
			array(
				'Order' => array(
					'id' => $orderId,
					'order_status_id' => $statusId
				)
			)
		);
	}

	/**
	 * Base code for retrieving a single record for edit or view
	 *
	 * @param integer $id ID of row to edit
	 * @param array $query The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	protected function _getItem($id, $query = array()) {
		$defaultParams = array(
			'contain' => array(
				'OrderItem' => array(
					'OrderItemData'
				),
				'OrderTotal' => array(
					'order' => 'OrderTotal.sequence ASC'
				),
				'OrderStatus',
				'OrderData',
				'OrderNote' => array(
					'User'
				),
			),
		);

		$query = array_merge_recursive($query, $defaultParams);

		$itemContains = $this->_buildItemContains(
			Configure::read('EvCheckout.OrderItemContains'),
			Configure::read('EvCheckout.OrderItemBelongsTo')
		);
		$itemContains[] = 'OrderItemData';
		if (!empty($query['contain']['OrderItem'])) {
			$itemContains = Hash::merge($itemContains, $query['contain']['OrderItem']);
		}
		$query['contain']['OrderItem'] = $itemContains;

		return parent::_getItem($id, $query);
	}

	/**
	 * redefine the readForView
	 *
	 * @param integer $id ID of row to edit
	 * @param array $params The db query array - can be used to pass in additional parameters such as contain
	 * @return array
	 */
	public function readForEdit($id, $suppliedParams = array()) {
		$editParams = [];
		if (!empty(Configure::read('EvCheckout.enableEditOrderItems'))) {
			$editParams['contain']['OrderEditingSession'] = ['User'];
		}

		$params = array_merge_recursive($editParams, $suppliedParams);

		return parent::readForEdit($id, $params);
	}

	protected function _buildItemContains($itemContains, $relationships) {
		if (empty($itemContains)) {
			$itemContain = array();

			foreach ($relationships as $key => $value) {
				if (is_array($value)) {
					$itemContain[] = $key;
				} else {
					$itemContain[] = $value;
				}
			}
		}

		return $itemContains;
	}

	/**
	 * afterFind to transform the transactionData
	 * into easier to use format
	 *
	 * @param mixed $results The results of the find operation
	 * @param bool $primary Whether this model is being queried directly (vs. being queried as an association)
	 * @return mixed Result of the find operation
	 * @link http://book.cakephp.org/2.0/en/models/callback-methods.html#afterfind
	 */
	public function afterFind($results, $primary = false) {
		foreach ($results as $key => $order) {
			if (!empty($order['OrderItem'])) {
				foreach ($order['OrderItem'] as $itemKey => $item) {
					if (!empty($item['OrderItemData'])) {
						$results[$key]['OrderItem'][$itemKey]['OrderItemData'] = Hash::combine(
							$results[$key]['OrderItem'][$itemKey]['OrderItemData'],
							'{n}.name',
							'{n}.item_data'
						);
					}
				}
			}
		}

		return $results;
	}

	/**
	 * Executed before a save() operation.
	 *
	 * @param $Model
	 * @param array $options
	 * @return  boolean
	 */
	public function beforeSave($options = array()) {
		# If we're editing an existing object, save off a copy of
		# the object as it exists before any changes.
		if (!empty( $this->id )) {
			$this->_original = $this->find('first', array(
				'conditions' => array(
					'id' => $this->id
				)
			));
		}
		return parent::beforeSave($options);
	}

	public function afterSave($created, $options = array()) {
		// We are looking for changes so don't do anything if created
		if (!$created) {
			// Get the updated version of the model
			$data = $this->find('first', array(
				'conditions' => array(
					'id' => $this->id
				)
			));

			// Compare the order_status_id with the before save version
			if ($data['Order']['order_status_id'] != $this->_original['Order']['order_status_id']) {
				// The order status has been updated so log the change
				$this->OrderUpdate->create();

				$orderUpdate = array(
					'OrderUpdate' => array(
						'order_id' => $this->id,
						'field' => 'order_status_id',
						'previous_value' => $this->_original['Order']['order_status_id'],
						'new_value' => $data['Order']['order_status_id']
					)
				);

				if ($this->OrderUpdate->save($orderUpdate)) {
					$this->getEventManager()->dispatch(
						new CakeEvent(
							'EvCheckout.Model.Order.statusChange.to.' . $this->OrderStatus->getStatusSlugById($data['Order']['order_status_id']),
							$this,
							[
								'orderId' => $data['Order']['id'],
								'fromStatusId' => $this->_original['Order']['order_status_id'],
							]
						)
					);
				}
			}
		}

		return parent::afterSave($created, $options);
	}

/**
 * Add a note to an order. If notes already exist for an order then add a line to separate them and add the
 * new note on the end. Prefix the new note with the current date and time.
 *
 * @param int    $id            The id of the order to add a note to.
 * @param string $newNote       The note to add to the order.
 * @param string $noteSeparator What to use to separate the new note from the old notes. Default to a new line.
 * @param string $dateSeparator What to use to separate the date and time from the note. Default to hyphen.
 * @return bool Wether the order was successfully updated or not.
 */
	public function addNote($id, $newNote, $noteSeparator = "\n", $dateSeparator = ' - ') {
		//Append the current date and time to the new note.
		$noteDate = new DateTime();
		$newNote = $noteDate->format('d-m-Y H:i') . $dateSeparator . $newNote;

		//Check if we need to add or append the new note to the order.
		$order = $this->findById($id);
		if (!empty($order['Order']['notes'])) {
			$newNote = $order['Order']['notes'] . $noteSeparator . $newNote;
		}

		return $this->save(
			[
				'Order' => [
					'id' => $id,
					'notes' => $newNote,
				]
			]
		);
	}

/**
 * Add a separate order note to an order.
 *
 * @param int    $id      The id of the order to save the note to.
 * @param string $note    The note string.
 * @param bool   $isAdmin True if the note is an admin order note. False otherwise.
 * @param int    $userId  The id of the user who wrote the note.
 * @return bool True if the note was added successfully. False otherwise.
 */
	public function addOrderNote($id, $note, $isAdmin = true, $userId = null) {
		$this->OrderNote->create();
		return $this->OrderNote->save(
			[
				'order_id' => $id,
				'note' => $note,
				'is_admin_note' => $isAdmin,
				'user_id' => $userId,
			]
		);
	}

}
