<?php

App::uses('OrderBuilderInterface', 'EvCheckout.Lib');
App::uses('OrderBuilder', 'EvCheckout.Lib');
App::uses('BasketLib', 'EvBasket.Lib');

class BasketBuilder extends OrderBuilder implements OrderBuilderInterface {

/**
 * Takes a transaction Id and processes the order, if the transaction is already attached to an order it just marks it
 * as paid, otherwise it creates an order from the basket contents.
 *
 * @param int $transactionId Transaction Id of the successful transaction
 * @return int                Order ID
 */
	public function build($transactionId) {
		if (!CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot build order from basket');
			return false;
		}

		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}

		$TransactionModel = EvClassRegistry::init('EvTransactions.Transaction');
		$transaction = $TransactionModel->readForView($transactionId);

		// If there is an unpaid order associated with the transation then simply mark it as paid
		if ($transaction['Transaction']['model'] == 'EvCheckout.Order') {
			$this->_controller->OrderManager->markAsPaid($transaction['Transaction']['model_id']);
			$orderId = $transaction['Transaction']['model_id'];

		} else {
			// It is an old transaction made before changing over so we need to create the order from the basket
			$orderId = $this->buildOrderFromBasket($transactionId);
			$this->_controller->OrderManager->markAsPaid($orderId);
		}

		if ($orderId) {
			// Whichever action we have take we still need to delete the basket to avoid confusion
			// get the basket Id
			$this->_deleteBasket($orderId);
		}

		return $orderId;
	}

/**
 * Delete the basket belonging to an order
 *
 * @param int $orderId Order ID
 * @return void
 */
	protected function _deleteBasket($orderId) {
		$OrderData = EvClassRegistry::init('EvCheckout.OrderData');
		$basketId = $OrderData->find('first', [
			'conditions' => [
				'name' => 'basket_id',
				'order_id' => $orderId
			]
		]);

		if ($basketId) {
			// clear the basket
			BasketLib::deleteBasket($basketId['OrderData']['data']);
			// force a rebuild of the basket cache
			$this->_controller->BasketManager->getBasket(true);
		}

		if (! empty(Configure::read('EvCheckout.customStockHandling')) || ! CakePlugin::loaded('EvInventory')) {
			return;
		}

		$this->_controller->OrderManager->reduceStockForOrder($orderId);
	}

/**
 * build an order from a basket
 *
 * @param 	int 	$transactionId 		The successful transaction ID just completed
 * @return  int 	$orderId 			The id number of the order we just created
 */
	public function buildOrderFromBasket($transactionId) {
		if (! CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot process basket');
			return false;
		}

		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		// load the mbasket manager so we can get the basket
		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}
		// get the addresses
		$TransactionModel = EvClassRegistry::init('EvTransactions.Transaction');
		$transaction = $TransactionModel->readForView($transactionId);

		//Load the user, we are pulling the user from the transaction if possible as it is safer than
		//assuming the user is logged in (timeouts, payment callbacks etc)
		$UserModel = EvClassRegistry::init('EvCore.User');

		// If the user_id isn't set on the transaction table then
		if ($transaction['Transaction']['user_id']) {
			$user = $UserModel->findById($transaction['Transaction']['user_id']);
		} else {
			$user = $this->_controller->Auth->user();
		}

		$addresses = array();
		if (! empty($transaction['TransactionData']['addresses'])) {
			$addresses = json_decode($transaction['TransactionData']['addresses'], true);
		}

		// Set the basket HASH as we can't always get it from the cookie (Worldpay isn't authenticated)
		$BasketModel = EvClassRegistry::init($transaction['Transaction']['model']);
		$basket = $BasketModel->findById($transaction['Transaction']['model_id']);

		$this->_controller->BasketManager->hash = $basket['Basket']['hash'];

		$orderId = $this->createFromBasket(
			$this->_controller->BasketManager->getBasket(),
			$user['User'],
			$addresses
		);

		// update the transaction to reference the order
		$this->_controller->OrderManager->assignToTransaction(
			$orderId,
			$transactionId
		);

		return $orderId;
	}

/**
 * create and setup an order from a basket
 *
 * @param 	array 		$basket 	Array of Basket Data
 * @param 	array 		$user 		array of user details
 * @param 	array 		$addresses 	array of address details from Address Model
 * @param   boolean 	$deleteBasket	If set to true it removes the basket after creating the order
 * @return 	int|bool 	$orderId 	Return the order ID number
 */
	public function createFromBasket($basket, $user, $addresses, $deleteBasket = true) {
		$this->_controller->OrderManager->setupOrder($user, $addresses, $basket['Basket']['currency_id']);

		foreach ($basket['BasketItem'] as $item) {
			$itemData = $this->_controller->OrderManager->setupOrderItemData($item);

			$this->_controller->OrderManager->setupOrderItem(
				BasketLib::getItemName($item),
				$item['quantity'],
				$item['unit_price'],
				$item['unit_price_incTax'],
				$item['row_total'],
				$item['row_total_incTax'],
				$item['tax_rate'],
				$item['model'],
				$item['model_id'],
				$itemData
			);
		}

		foreach ($basket['BasketTotal'] as $total) {
			$this->_controller->OrderManager->setupOrderTotal(
				$total['name'],
				$total['amount'],
				$total['sequence'],
				$total['display_ex_tax'],
				$total['display_inc_tax']
			);
		}

		foreach ($basket['BasketData'] as $data) {
			$this->_controller->OrderManager->setupOrderData(
				$data['name'],
				$data['data'],
				$data['is_visible']
			);
		}

		// Add the basket id to the order_data so we know what basket to delete when it is successful
		$this->_controller->OrderManager->setupOrderData(
			'basket_id',
			$basket['Basket']['id'],
			false
		);

		// Check if the delivery address is in the EU and whether we should be swallowing the VAT
		if (
			! empty($addresses['delivery']['Country']) &&
			$addresses['delivery']['Country']['is_eu'] !== true
		) {
			// swallow or remove the order VAT
			if (Configure::read('EvCheckout.swallowVatOutsideEU') === true) {
				// If the order is from outside the EU and if the flag is set we will swallow the VAT
				// Keeping the order total the same but zero'ing the VAT
				$this->_controller->OrderManager->zeroOrderItemTax();
			} elseif (Configure::read('EvCheckout.removeVatOutsideEU') === true) {
				// If the order is from outside the EU and if the flag is set then
				// remove the VAT from all order items and order totals
				$this->_controller->OrderManager->removeOrderVAT();
			}
		}

		$orderId = $this->_controller->OrderManager->saveOrder(EvClassRegistry::init('EvCheckout.Order'));

		if ($orderId !== false) {
			if (! empty($basket['Basket']['id']) && $deleteBasket) {
				$Basket = EvClassRegistry::init('EvBasket.Basket');
				$Basket->delete($basket['Basket']['id']);
			}

			return $orderId;
		}

		return false;
	}
}
