<?php

App::uses('EvCheckoutAppController', 'EvCheckout.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');
App::uses('OrderLib', 'EvCheckout.Lib');

class OrdersController extends EvCheckoutAppController {

	public $helpers = [
		'EvCheckout.Order',
	];

/**
 * Called before the controller action. You can use this method to configure and customize components
 * or perform logic that needs to happen before each controller action.
 *
 * @return void.
 */
	public function beforeFilter() {
		$this->adminActions[] = 'admin_pay';
		$this->adminActions[] = 'admin_status';
		$this->adminActions[] = 'admin_invoice';
		$this->adminActions[] = 'admin_notes';
		$this->adminActions[] = 'admin_return_item';
		$this->adminActions[] = 'admin_resendOrderConfirmationEmail';

		parent::beforeFilter();

		$this->Auth->allow(array('complete', 'cancel'));

		// specify whether order items can have their own, individual,
		// order item statuses
		$this->hasOrderItemStatuses = Configure::read('EvCheckout.allowOrderItemStatuses');
	}

/**
 * Complete an order.
 * This could display success or fail messages.
 * This is where the payment is checked and turned into an order
 *
 * @return void.
 */
	public function complete() {
		if (CakePlugin::loaded('EvTransactions') && ! empty($this->request->query['transaction'])) {
			$this->TransactionsManager = $this->loadComponent('EvTransactions.Transactions');
			$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');
			$this->OrderPages = $this->loadComponent('EvCheckout.OrderPages');

			// Security check - Does this transaction belong to the current user session?
			// If not, check the result but don't show any order info
			$authorised = $this->_checkTransactionAccess($this->request->query['transaction']);

			if ($this->TransactionsManager->checkPayment()) {
				// Before we build the order we need to see if this transaction id refers to an already successful order,
				// in which case we wont want to build a new order
				$orderId = $this->_transactionAlreadyProcessed($this->request->query['transaction']);
				if (!$orderId) {
					// build the order. OrderBuilder is specified within Config
					// builds from an EvBasket Row as default
					$orderId = $this->OrderManager->buildOrder($this->request->query['transaction']);

					// trigger the success hook
					$this->getEventManager()->dispatch(
						new CakeEvent('EvCheckout.Controller.Order.success', $this, array(
							'orderId' => $orderId
						))
					);
				}

				if ($authorised) {
					$pageId = $this->OrderPages->getOrderSuccessPageId($orderId);
					$this->OrderPages->setOrderSuccessPage($orderId, $pageId);
				} else {
					// Not authorised - just show a bare bones order success page with no private info
					return $this->OrderPages->showUnauthorisedSuccess($this->request->query['transaction']);
				}
			} else {
				// trigger the fail hook
				$this->getEventManager()->dispatch(
					new CakeEvent('EvCheckout.Controller.Order.fail', $this)
				);

				if ($authorised) {
					if (!empty(Configure::read('EvCheckout.failedOrder.redirect'))) {
						if (!empty(Configure::read('EvCheckout.failedOrder.message'))) {
							$this->Flash->fail(Configure::read('EvCheckout.failedOrder.message'));
						} else {
							$this->Flash->fail('Payment failed. Please check your card details and try again.');
						}

						$this->redirect(Configure::read('EvCheckout.failedOrder.redirect'));

					}

					$pageId = $this->OrderPages->getOrderFailPageId();
					$this->OrderPages->setOrderFailPage($pageId);
				} else {
					// Not authorised. The session has likely been lost so just show a generic fail page with no private information
					return $this->OrderPages->showUnauthorisedFail($this->request->query['transaction']);
				}
			}
		}

		if (empty($this->view)) {
			// trigger the fail hook
			$this->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.Controller.Order.fail', $this)
			);

			$pageId = $this->OrderPages->getOrderFailPageId();
			$this->OrderPages->setOrderFailPage($pageId);
		}

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$this->_setupCompletePage($pageId, [
				'orderId' => isset($orderId) ? $orderId : 0,
			]);
		}
	}

/**
 * Checks that the transaction given is accessible in the current session
 *
 * @param int $transactionId The transaction ID to check
 * @return bool True if accessible, false if not.
 */
	protected function _checkTransactionAccess($transactionId) {
		return OrderLib::checkTransactionAccess($transactionId, $this->Auth->user('User.id'));
	}

/**
 * Set up the view for the complete page.
 *
 * @param int   $pageId The page ID to fetch
 * @param array $params Any params to use for rendering the page.
 * @return void.
 */
	protected function _setupCompletePage($pageId, $params = []) {
		$pageData = $this->assignPage($pageId);

		$pageData = $this->_modifyCompletePageData($pageData, $params);
		$this->set('data', $pageData);

		$this->view = $this->Tpl->getTemplate($pageData, 'Page');
	}

/**
 * When a complete order page is setup, page data is pulled in to display additional content.
 * Before it is rendered the page data is parsed to replace any instance of ORDERID with the
 * actual ID of the order.
 *
 * @param array $pageData   Data from the assigned order complete page.
 * @param array $pageParams The parameters for setting up the complete order page.
 * @return array Modified page data.
 */
	protected function _modifyCompletePageData($pageData, $pageParams = []) {
		if (empty($pageData) || empty($pageParams['orderId'])) {
			return $pageData;
		}

		if (!empty($pageData['Page']['title'])) {
			$pageData['Page']['title'] = str_replace('ORDERID', $pageParams['orderId'], $pageData['Page']['title']);
		}

		if (!empty($pageData['Page']['body'])) {
			$pageData['Page']['body'] = str_replace('ORDERID', $pageParams['orderId'], $pageData['Page']['body']);
		}

		return $pageData;
	}

	/**
	 * cancel an order.
	 * usually used on gateways for the "cancel and return to store" links
	 *
	 */
	public function cancel() {
		if (CakePlugin::loaded('EvTransactions') && ! empty($this->request->query['transaction'])) {
			$this->TransactionsManager = $this->loadComponent('EvTransactions.Transactions');

			/*$this->TransactionsManager->failTransaction(
				$this->request->query['transaction'],
				'The order was cancelled by the user'
			);*/
		}

		// trigger the cancelled hook
		$this->getEventManager()->dispatch(
			new CakeEvent('EvCheckout.Controller.Order.cancel', $this)
		);

		$this->view = 'EvCheckout./Fallbacks/Order/cancel';
		$pageId = Configure::read('EvCheckout.pageIds.cancel');

		// check if we have a specific content page set
		// if so - set the template
		if (! empty($pageId)) {
			$pageData = $this->assignPage($pageId);

			if (! empty($pageData)) {
				$this->view = $this->Tpl->getTemplate($pageData, 'Page');
			}
		}
	}

/**
 * Checks to see if there is already a successful order for the given transaction id
 * This stops use creating duplicate orders for the same transaction if they user
 * refreshs the page etc
 * @param $transactionId The Transction ID you want to check
 * @return mixed The order id if there is one otherwise false
 */
	protected function _transactionAlreadyProcessed($transactionId) {
		$this->Transaction = EvClassRegistry::init('EvTransactions.Transaction');
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$transaction = $this->Transaction->find('first', array(
				'conditions' => array(
					'Transaction.id' => $transactionId,
					'model' => 'EvCheckout.Order',
					'status' => 'success'
				),
				'contain' => array(
					'Order'
				)
			)
		);

		$unPaidStatusId = $OrderStatus->getStatusIdBySlug(Configure::read('EvCheckout.statusSlugs.unpaid'));

		if (!empty($transaction[$this->Transaction->alias]['model_id']) && ($transaction['Order']['order_status_id'] != $unPaidStatusId)) {
			return $transaction[$this->Transaction->alias]['model_id'];
		}

		return false;
	}

	/**
	 * redefine index columns
	 * to add more order info
	 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$columns = ArrayUtil::addAfter(
			$columns,
			'Order.name',
			array(
				'Order.email' => array(
					'type' => 'string',
					'label' => __('Email')
				),
				'Order.phone' => array(
					'type' => 'string',
					'label' => __('Phone')
				),
				'OrderStatus.name' => array(
					'type' => 'string',
					'label' => __('Status')
				),
				'Order.is_paid' => array(
					'type' => 'display_boolean',
					'label' => __('Paid')
				),
				'Currency.name' => array(
					'type' => 'string',
					'label' => __('Currency')
				)
			)
		);

		unset($columns['Order.modified']);
		return $columns;
	}

	/*
		----------------------------------------------------------------
		Admin methods
		----------------------------------------------------------------
	*/

	/**
	 * redefine to remove the add button
	 *
	 */
	protected function _adminIndexToolbar() {
		return array();
	}

	/**
	 * redefine index paginate
	 * to contain status
	 */
	protected function _adminIndexPaginate() {
		$paginate = parent::_adminIndexPaginate();

		$paginate['contain'][] = 'OrderStatus';
		$paginate['contain'][] = 'Currency';
		$paginate['order'] = 'Order.id DESC';

		return $paginate;
	}

/**
 * Defines which actions are available to each row of the admin_index table results
 *
 * Override in your controller to customise
 * Customise the handlers for these actions in /view/Admin/Elements/index_results.ctp
 *
 * @return array
 */
	protected function _adminIndexActions() {
		$actions = parent::_adminIndexActions();

		//The "View" action links to the frontend view for the order which admin users don't have permission to view.
		//Removing to avoid confusion.
		if (isset($actions['View'])) {
			unset($actions['View']);
		}

		return $actions;
	}

	/**
	 * Define an array of actions that can be performed on multiple items on the
	 * admin listing page.
	 *
	 * @return array
	 */
	protected function _adminMultiEditActions() {
		$actions = array('PrintInvoices' => 'Print Invoices');
		if (Configure::check('EvCheckout.dispatchedStatusId')) {
			$actions['MarkDispatched'] = 'Mark Dispatched';
		}
		return $actions;
	}

	/**
	 * define index filters
	 * to make it more usable
	 *
	 */
	protected function _adminFilterFields() {
		//Adding ID filter as first filter
		$fields = [
			'Order.ref' => [
				'label' => __('Id'),
				'type' => 'integer',
				'compare' => [
					'Order.id LIKE' => '%s'
				]
			]
		];

		$fields = array_merge($fields, parent::_adminFilterFields());

		$fields = ArrayUtil::addAfter(
			$fields,
			'Order.name',
			array(
				'Order.email' => array(
					'label' => __('Email'),
					'type' => 'string',
					'compare' => array(
						'Order.email LIKE' => '%%%s%%'
					)
				),
				'Order.order_status_id' => array(
					'label' => __('Status'),
					'type' => 'select',
					'compare' => array(
						'Order.order_status_id' => '%s'
					)
				),
				'Order.is_paid' => array(
					'label' => __('Paid'),
					'type' => 'select',
					'compare' => array(
						'Order.is_paid' => '%s'
					)
				),
				'Order.currency_id' => array(
					'label' => __('Currency'),
					'type' => 'select',
					'compare' => array(
						'Order.currency_id' => '%s'
					)
				)
			)
		);

		unset($fields['Order.modified']);

		return $fields;
	}

	/**
	 * redefine filter populate
	 * to bring in order status list
	 *
	 */
	protected function _adminPopulateFilterLookups() {
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$orderStatuses = $OrderStatus->find('list', array(
			'conditions' => array(
				'is_selectable' => 1,
				'is_active' => 1
			)
		));

		$this->set(
			'orderStatuses',
			$orderStatuses
		);

		$this->set(
			'isPaids',
			array(
				'1' => __('Yes'),
				'0' => __('No')
			)
		);
	}

	/**
	 * redefine filter populate
	 * to bring in order status list
	 *
	 */
	protected function _adminPopulateLookups() {
		$OrderStatus = EvClassRegistry::init('EvCheckout.OrderStatus');

		$this->set(
			'orderStatuses',
			$OrderStatus->getForDropDown()
		);

		if (! empty($this->hasOrderItemStatuses)) {
			$OrderItemStatus = EvClassRegistry::init('EvCheckout.OrderItemStatus');
			$this->set(
				'orderItemStatuses',
				$OrderItemStatus->getForDropdown()
			);
		}
	}

	/**
	 * redefine to remove the add button
	 *
	 */
	protected function _adminFormToolbar($id = null) {
		$Model = $this->{$this->modelClass};

		$toolbar = parent::_adminFormToolbar($id);

		unset($toolbar['Add New']);

		$toolbar['<i class="fa fa-file"></i> Print Invoice'] = array(
			'url' => array(
				'admin' => true,
				'plugin' => 'ev_checkout',
				'controller' => 'orders',
				'action' => 'admin_invoice',
				$id
			),
			'class' => 'toolbar__btn',
			'target' => '_blank'
		);

		if (! empty(Configure::read('EvCheckout.enableEditOrderItems')) && $Model->OrderStatus->isEditableState($this->request->data['OrderStatus']['slug'])) {
			$toolbar['Edit Order Items'] = [
				'class' => 'toolbar__btn--edit',
				'url' => [
					'plugin' => 'ev_checkout',
					'controller' => 'edit_order_items',
					'action' => 'edit',
					$id
				]
			];
		}

		if (Configure::read('EvCheckout.enableResendOrderConfirmationEmail')) {
			$toolbar['Resend Confirmation Email'] = [
				'class' => 'toolbar__btn--copy',
				'url' => [
					'plugin' => 'ev_checkout',
					'controller' => 'orders',
					'action' => 'resendOrderConfirmationEmail',
					$id
				]
			];
		}

		return $toolbar;
	}

	/**
	 * redefine the admin edit to have a nicer order management screen
	 *
	 * @param 	int 	$id 	The order ID
	 */
	public function admin_edit($id = null) {
		$this->helpers[] = 'EvCheckout.OrderPayment';

		$this->toInject('helpers', 'EvCheckout.Order');

		if ($id === null) {
			$this->Flash->fail(
				array(
					'title' => 'Error',
					'description' => 'No order was selected for viewing.'
				)
			);

			$this->redirect(
				array('action' => 'index')
			);
		}

		parent::admin_edit($id);

		// assign template variable to specify whether order items can have their
		// own individual statuses
		$this->set('hasOrderItemStatuses', $this->hasOrderItemStatuses);

		// If EvTranssaction is installed and we can find a transaction for this order, lets pull it in!
		if (CakePlugin::loaded('EvTransactions')) {
			$this->loadModel('EvTransactions.Transaction');
			$transactions = $this->Transaction->find('all', array(
				'conditions' => array(
					'OR' => array(
						array('model' => 'EvCheckout.Order'),
						array('model' => 'EvCheckoutOrder') // Gotta account for sites that override stuff!
					),
					'model_id' => $id
				)
			));

			if (! empty($transactions)) {
				$this->set(compact('transactions'));
			}
		}

		$this->view = 'EvCheckout.Orders/admin_edit';
	}

	/**
	 * save order notes
	 *
	 * @param  integer $id order id
	 * @return Redirect
	 */
	public function admin_notes($id) {
		$result = false;

		if ($this->request->is('post') || $this->request->is('put')) {
			$this->{$this->modelClass}->id = $id;

			$result = $this->{$this->modelClass}->save(array(
				'admin_notes' => $this->request->data['Order']['admin_notes'],
				'notes' => $this->request->data['Order']['notes']
			), array('deep' => true));

			if ($result) {
				$this->Flash->success(
					array(
						'title' => 'Notes Updated',
						'description' => 'The notes for this order have been updated.'
					)
				);
			}
		}

		if (! $result) {
			$this->Flash->fail(
				array(
					'title' => 'Failed',
					'description' => 'There was a problem updating your order notes, please try again.'
				)
			);
		}

		return $this->redirect(array(
			'plugin' => 'ev_checkout',
			'admin' => true,
			'controller' => 'orders',
			'action' => 'edit',
			$id
		));
	}

	/**
	 * mark an order as paid
	 *
	 * @param 	int 	$id 	the order ID to mark as paid
	 */
	public function admin_pay($id) {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		if ($this->OrderManager->markAsPaid($id)) {
			$this->Flash->success(
				array(
					'title' => 'Paid',
					'description' => 'The order (#' . $id . ') has been marked as paid'
				)
			);
		} else {
			$this->Flash->fail(
				array(
					'title' => 'Failed',
					'description' => 'There was a problem marking the order (#' . $id . ') as paid'
				)
			);
		}

		$this->redirect(
			array(
				'action' => 'edit',
				$id
			)
		);
	}

	/**
	 * change an orders status
	 *
	 * @param 	int 	$id 	the order ID to change
	 */
	public function admin_status($id) {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		$statusSet = ! empty($this->request->data['Order']['order_status_id']);

		if ($statusSet && $this->OrderManager->updateStatus($id, $this->request->data['Order']['order_status_id'])) {
			$this->Flash->success(
				array(
					'title' => 'Status Changed',
					'description' => 'The order status for #' . $id . ' has been updated'
				)
			);
		} else {
			$this->Flash->fail(
				array(
					'title' => 'Failed',
					'description' => 'There was a problem updating the status for order #' . $id
				)
			);
		}

		$this->redirect(
			array(
				'action' => 'edit',
				$id
			)
		);
	}

	/**
	 * view printable picking / invoice list
	 *
	 * @param 	int 	$id 	the order ID to view
	 */
	public function admin_invoice($id) {
		parent::admin_edit($id);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'Order #' . $id . ' Invoice',
				)
			),
			'EvCore.Page'
		);

		$currencyModel = EvClassRegistry::init('EvCurrency.Currency');
		$this->set('currencyList', $currencyModel->getForDropdown());

		$this->view = 'EvCheckout.Orders/admin_invoice';
		$this->layout = 'print';
	}

	/**
	 * frontend - show users order history
	 *
	 */
	public function history() {
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		$this->set(
			'orders',
			$this->OrderManager->getOrdersByUser($this->Auth->user('User.id'))
		);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'Order History'
				)
			),
			'EvCore.Page'
		);

		$this->view = 'EvCheckout.Orders/history';
	}

	/**
	 * frontend - show the user their order
	 *
	 * @param 	int 	$orderId 	The id of the order to view
	 */
	public function view($id) {
		$OrderModel = EvClassRegistry::init('EvCheckout.Order');
		$order = $OrderModel->readForView($id);

		if (! empty($order['Order']['user_id']) && (int)$order['Order']['user_id'] !== (int)$this->Auth->user('User.id')) {
			$route = $this->Routable->getItemRoute('EvCheckout', 'Order');
			$route['action'] = 'history';

			$this->Flash->fail('You do not have permission to view that order');

			$this->redirect($route);
		}

		$this->set(
			'order',
			$order
		);

		$this->Meta->set(
			array(
				'Page' => array(
					'title' => 'View Order #' . $id
				)
			),
			'EvCore.Page'
		);

		$this->view = 'EvCheckout.Orders/view';
	}

	/**
	 * Takes multiple orders and generates the invoices for them ready to print
	 * @return null
	 */
	protected function _processMultiEditPrintInvoices() {
		$Model = $this->{$this->modelClass};

		$orderIds = array_keys($this->request->data[$Model->alias]['id']);

		$orders = $Model->find('all', array(
			'contain' => array(
				'OrderItem' => array(
					'OrderItemData'
				),
				'OrderTotal' => array(
					'order' => 'OrderTotal.sequence ASC'
				),
				'OrderStatus',
				'OrderData'
			),
			'conditions' => array(
				'Order.id' => $orderIds
			)
		));

		$this->set('orders', $orders);

		$currencyModel = EvClassRegistry::init('EvCurrency.Currency');
		$this->set('currencyList', $currencyModel->getForDropdown());

		$this->layout = 'print';
		echo $this->render('EvCheckout.Orders/admin_bulk_invoice');
		die();
	}

/**
 * Takes multiple orders and updates their status to dispatched
 * @return null
 */
	protected function _processMultiEditMarkDispatched() {
		$Model = $this->{$this->modelClass};
		$orderIds = array_keys($this->request->data[$Model->alias]['id']);

		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');

		foreach ($orderIds as $id) {
			$this->OrderManager->updateStatus($id, Configure::read('EvCheckout.dispatchedStatusId'));
		}
	}

/**
 * Marks an item returned by adding an OrderItemData with name quantity_returned set to the number returned
 * If stock couldn't be automatically updated it adds a OrderItemData with name return_stock_updated set to 0
 */
	public function admin_return_item() {
		// Only allow delete via post otherwise a crawler could potentially delete everything
		if ($this->request->is('post') || $this->request->is('put')) {
			if (!empty($this->request->data['OrderItem']['id'])) {

				$id = $this->request->data['OrderItem']['id'];

				$OrderItem = EvClassRegistry::init('EvCheckout.OrderItem');
				$item = $OrderItem->find('first', [
					'conditions' => [
						'id' => $id,
					],
					'contain' => [
						'OrderItemData'
					]
				]);

				if (empty($item)) {
					$this->Flash->fail('There was a problem updating the item');
					$this->redirect( $this->referer() );
				}

				// Sort the item data into an easier to read format
				if (!empty($item['OrderItemData'])) {
					$item['OrderItemData'] = Hash::combine($item['OrderItemData'], '{n}.name', '{n}.item_data');
				}

				// Calculate the total number that will have been returned and check that it is not more than the original quantity
				$numToReturn = $this->request->data['OrderItem']['quantity'];
				if (!empty($item['OrderItemData']['quantity_returned'])) {
					$numToReturn += $item['OrderItemData']['quantity_returned'];
				}
				if ($numToReturn > $item['OrderItem']['quantity']) {
					$this->Flash->fail('You cannot return more than were originally purchased');
					return $this->redirect( $this->referer() );
				}

				// Remove the old entry and save a new one
				$OrderItem->OrderItemData->deleteAll([
					'OrderItemData.name' => 'quantity_returned',
					'OrderItemData.order_item_id' => $id
				]);

				$OrderItem->OrderItemData->save([
					'name' => 'quantity_returned',
					'item_data' => $numToReturn,
					'order_item_id' => $id,
				]);

				// Update the inventory
				if (CakePlugin::loaded('EvShop') && CakePlugin::loaded('EvInventory') && !empty($item['OrderItemData']['id'])) {
					$Variant = EvClassRegistry::init('EvShop.Variant');
					$variant = $Variant->findById($item['OrderItemData']['id']);

					$success = false;

					if (!empty($variant['Inventory'])) {
						$variant['Inventory']['stock'] += $this->request->data['OrderItem']['quantity'];
						$success = $Variant->Inventory->save([
							'id' => $variant['Inventory']['id'],
							'stock' => $variant['Inventory']['stock']
						]);
					}

					// If this already has a not updated notice just leave that one there
					// otherwise add a notice to say the inventory was not updated.
					if (empty($success)) {
						if (empty($item['OrderItemData']['return_stock_updated'])) {
							$OrderItem->OrderItemData->save([
								'OrderItemData' => [
									'name' => 'return_stock_updated',
									'item_data' => '0',
									'order_item_id' => $id,
								]
							]);
						}
					}
				}

				$this->Flash->success('Items have been marked as returned');
				return $this->redirect( $this->referer() );
			}
		}
	}

/**
 * Sets up a restricted basket on the front end and allows modifying the order
 *
 * @param int $id The ID of the order to edit
 * @throws NotFoundException When the order id is invalid
 * @return void Redirects to the basket page.
 */
	public function admin_edit_order_items($id) {
		// Start an order editing session
		$result = $this->OrderManager->beginEditOrderSession($id);

		if (!empty($result)) {
			return $this->redirect(Configure::read('EvCheckout.editOrderItemRedirects.beginSession'));
		} else {
			$this->Flash->fail(['title' => 'Could not begin an editing session for this order']);
			return $this->redirect($this->referer());
		}
	}

/**
 * Cancels editing the order
 *
 * @param int $id The order ID. If not defined it will get it from the current session.
 * @return void Redirects to the order.
 */
	public function admin_cancel_edit_order_items($id = null) {
		// If not given an ID try to find one from the session
		if (empty($id)) {
			$editOrder = $this->OrderManager->checkForEditOrder();
			if (!empty($editOrder)) {
				$id = $editOrder['order_id'];
			}
		}

		if (!empty($id)) {
			// End the editing session
			$this->OrderManager->endEditOrderSession($id);

			$path = Configure::read('EvCheckout.editOrderItemRedirects.endSession');
			if (!empty(Configure::read('EvCheckout.editOrderItemRedirects.appendOrderIdToEndSession'))) {
				$path[] = $id;
			}

			// Redirect back to the edit page
			return $this->redirect($path);
		}

		// Nothing to do, redirect to admin index
		return $this->redirect(Configure::read('EvCheckout.editOrderItemRedirects.endSessionWithoutId'));
	}

/**
 * Saves the order currently being edited
 *
 * @throws NotFoundException if there is no order currently being edited
 * @return void Redirects back on fail or to a success page on success.
 */
	public function admin_save_edit_order_items() {
		// Without basket there's nothing we can do by default
		if (CakePlugin::isLoaded('EvBasket')) {
			// If we have an order being edited, save that first ignoring the $id param
			$editOrder = $this->OrderManager->checkForEditOrder();
			if (empty($editOrder)) {
				throw new NotFoundException();
			}

			// Get the current basket ensuring basketItemData is contained
			if (empty($this->Basket)) {
				$this->loadModel('Basket');
			}
			$basket = $this->Basket->getFullBasket(
				$editOrder['basket_hash'],
				Configure::read('EvBasket.BasketItemContains')
			);

			// Update the order using the basket
			$success = $this->OrderManager->updateOrderFromBasket($editOrder['order_id'], $basket);

			if (!$success) {
				// Redirect back the basket on fail
				return $this->redirect($this->referer());
			} else {
				// Add a note to the order to say it was edited.
				$note = Configure::read('EvCheckout.editedOrderAdminNote');
				if (!empty($note)) {
					$this->{$this->modelClass}->addOrderNote($editOrder['order_id'], $note, true, $this->Auth->user()['User']['id']);
				}

				// We've finished so go to the complete page
				$this->OrderManager->endEditOrderSession();
				return $this->redirect([
					'admin' => false,
					'action' => 'edit_order_complete',
					$editOrder['order_id']
				]);
			}
		} else {
			throw new NotFoundException();
		}
	}

/**
 * The order has been saved successfully
 *
 * @param int $id The order ID to show
 * @return void.
 */
	public function edit_order_complete($id) {
		$Model = $this->{$this->modelClass};

		if ($this->request->is('post')) {
			if (!empty($this->request->data['Order']['notify_user'])) {
				if ($this->OrderManager->sendOrderUpdatedEmail($id)) {
					$this->Flash->success(['title' => 'Email Sent', 'description' => 'An email was sent to the user with details of the changed order']);
				} else {
					$this->Flash->success(['title' => 'Could not send email', 'description' => 'There was a problem sending the email']);
				}
			}

			$path = Configure::read('EvCheckout.editOrderItemRedirects.endSession');
			if (!empty(Configure::read('EvCheckout.editOrderItemRedirects.appendOrderIdToEndSession'))) {
				$path[] = $id;
			}

			// Return to the order in the CMS
			return $this->redirect($path);
		}

		$order = $Model->readForView($id);
		$this->set('order', $order);

		$this->view = 'EvCheckout.Orders/edit-order-complete';
	}

/**
 * Resend the order confirmation email
 *
 * @param $id
 *
 * @return CakeResponse|null
 */
	public function admin_resendOrderConfirmationEmail($id) {
		$fail = false; // Used to redirect to the index if there's a problem
		$routePrefix = array(
			'plugin' => 'ev_checkout',
			'admin' => true,
			'controller' => 'orders',
		);

		// Check if this action is enabled
		if (! Configure::read('EvCheckout.enableResendOrderConfirmationEmail')) {
			$this->Flash->fail('This action has been disabled.');
			$fail = true;
		}

		// Verify the order exists
		$order = $this->{$this->modelClass}->findById($id);
		if (empty($order)) {
			$this->Flash->fail('The order you selected could not be found.');
			$fail = true;
		}

		// Redirect to the index if the fail flag has been triggered
		if ($fail) {
			return $this->redirect(array_merge($routePrefix, array('action' => 'index')));
		}

		// Trigger the email, and then redirect back to the order edit page
		$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');
		$this->OrderManager->sendOrderConfirmEmails($id);

		$this->Flash->success('The confirmation email has been resent.');
		return $this->redirect(array_merge($routePrefix, array('action' => 'edit', $id)));
	}
}
