<?php

App::uses('EvCheckoutAppController', 'EvCheckout.Controller');
App::uses('ArrayUtil', 'EvCore.Lib');

class EditOrderItemsController extends EvCheckoutAppController {

	public $uses = 'EvCheckout.Order';

	public $helpers = [
		'EvCheckout.Order'
	];

	public function beforeFilter() {
		$this->adminActions[] = 'admin_edit';
		$this->adminActions[] = 'admin_save';
		$this->adminActions[] = 'admin_cancel';

		parent::beforeFilter();
	}

/**
 * Sets up a restricted basket on the front end and allows modifying the order
 *
 * @param int $id The ID of the order to edit
 * @throws NotFoundException When the order id is invalid
 * @return void Redirects to the basket page.
 */
	public function admin_edit($id = null) {
		if (!empty($id)) {
			// Make sure the component is loaded
			if (empty($this->OrderEditing)) {
				$this->OrderEditing = $this->loadComponent('EvCheckout.OrderEditing');
			}

			// Start an order editing session
			$result = $this->OrderEditing->beginEditOrderSession($id);
		}

		if (!empty($result)) {
			return $this->redirect(Configure::read('EvCheckout.editOrderItemRedirects.beginSession'));
		} else {
			$this->Flash->fail(['title' => 'Could not begin an editing session for this order']);
			return $this->redirect($this->referer());
		}
	}

/**
 * Cancels editing the order
 *
 * @param int $id The order ID. If not defined it will get it from the current session.
 * @return void Redirects to the order.
 */
	public function admin_cancel($id = null) {
		// Make sure the component is loaded
		if (empty($this->OrderEditing)) {
			$this->OrderEditing = $this->loadComponent('EvCheckout.OrderEditing');
		}

		// If not given an ID try to find one from the session
		if (empty($id)) {
			$editOrder = $this->OrderEditing->checkForEditOrder();
			if (!empty($editOrder)) {
				$id = $editOrder['order_id'];
			}
		}

		if (!empty($id)) {
			// End the editing session
			$this->OrderEditing->endEditOrderSession($id);

			$path = Configure::read('EvCheckout.editOrderItemRedirects.endSession');
			if (!empty(Configure::read('EvCheckout.editOrderItemRedirects.appendOrderIdToEndSession'))) {
				$path[] = $id;
			}

			// Redirect back to the edit page
			return $this->redirect($path);
		}

		// Nothing to do, redirect to admin index
		return $this->redirect(Configure::read('EvCheckout.editOrderItemRedirects.endSessionWithoutId'));
	}

/**
 * Saves the order currently being edited
 *
 * @throws NotFoundException if there is no order currently being edited
 * @return void Redirects back on fail or to a success page on success.
 */
	public function admin_save() {
		// Without basket there's nothing we can do by default
		if (CakePlugin::loaded('EvBasket')) {
			// Make sure the component is loaded
			if (empty($this->OrderEditing)) {
				$this->OrderEditing = $this->loadComponent('EvCheckout.OrderEditing');
			}

			// If we have an order being edited, save that first ignoring the $id param
			$editOrder = $this->OrderEditing->checkForEditOrder();
			if (empty($editOrder)) {
				throw new NotFoundException();
			}

			// Get the current basket ensuring basketItemData is contained
			if (empty($this->Basket)) {
				$this->loadModel('EvBasket.Basket');
			}
			$basket = $this->Basket->getFullBasket(
				$editOrder['basket_hash'],
				Configure::read('EvBasket.BasketItemContains')
			);

			// Update the order using the basket
			$success = $this->OrderEditing->updateOrderFromBasket($editOrder['order_id'], $basket);

			if (!$success) {
				// Redirect back the basket on fail
				return $this->redirect($this->referer());
			} else {
				// Add a note to the order to say it was edited.
				$note = Configure::read('EvCheckout.editedOrderAdminNote');
				if (!empty($note)) {
					$this->Order->addOrderNote($editOrder['order_id'], $note, true, $this->Auth->user()['User']['id']);
				}

				// We've finished so go to the complete page
				$this->OrderEditing->endEditOrderSession();
				$path = Configure::read('EvCheckout.editOrderItemRedirects.editItemsComplete');
				$path[] = $editOrder['order_id'];
				return $this->redirect($path);
			}
		} else {
			throw new NotFoundException();
		}
	}

/**
 * The order has been saved successfully
 *
 * @param int $id The order ID to show
 * @return void.
 */
	public function complete($id) {
		$Model = $this->{$this->modelClass};

		if ($this->request->is('post')) {
			if (!empty($this->request->data['Order']['notify_user'])) {

				if (empty($this->OrderManager)) {
					$this->OrderManager = $this->loadComponent('EvCheckout.OrderManager');
				}

				if ($this->OrderManager->sendOrderUpdatedEmail($id)) {
					$this->Flash->success(['title' => 'Email Sent', 'description' => 'An email was sent to the user with details of the changed order']);
				} else {
					$this->Flash->success(['title' => 'Could not send email', 'description' => 'There was a problem sending the email']);
				}
			}

			$path = Configure::read('EvCheckout.editOrderItemRedirects.endSession');
			if (!empty(Configure::read('EvCheckout.editOrderItemRedirects.appendOrderIdToEndSession'))) {
				$path[] = $id;
			}

			// Return to the order in the CMS
			return $this->redirect($path);
		}

		$order = $Model->readForView($id);
		$this->set('order', $order);

		$this->view = 'EvCheckout.Orders/edit-order-complete';
	}
}
