<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('BasketLib', 'EvBasket.Lib');

class PostStageComponent extends AppComponent {

	/**
	 * check for a process any registration data
	 *
	 * @return bool
	 */
	public function processRegister() {
		$return = true;

		if (isset($this->_controller->request->data['Form']['action']) && $this->_controller->request->data['Form']['action'] == 'register') {

			if (empty($this->_controller->UserManager)) {
				$this->_controller->UserManager = $this->_controller->loadComponent('EvCore.UserManager');
			}

			// move the array data to 'User' section, separated to prevent
			// login / register forms from both populating

			$UserModel = EvClassRegistry::init('EvCore.User');
			$result = $this->_controller->UserManager->register($UserModel);

			if ($result['result'] === true) {
				//Don't show the success message if they are a guest user.
				if (empty($this->_controller->request->data['User']['is_guest_user'])) {
					$this->_controller->Flash->success($result['data']['message']);
				}
			} else {

				$return = false;
				$this->_controller->Flash->fail($result['data']['errors']);
			}
		}

		return $return;
	}

	/**
	 * check for a process any login data
	 *
	 * @return bool
	 */
	public function processLogin() {
		$return = true;

		if (isset($this->_controller->request->data['Form']['action']) && $this->_controller->request->data['Form']['action'] == 'login') {

			if (empty($this->_controller->UserManager)) {
				$this->_controller->UserManager = $this->_controller->loadComponent('EvCore.UserManager');
			}

			// move the array data to 'User' section, separated to prevent
			// login / register forms from both populating
			$this->_controller->request->data['User'] = $this->_controller->request->data['Login'];

			$UserModel = EvClassRegistry::init('EvCore.User');
			$result = $this->_controller->UserManager->login($UserModel);

			if ($result['result'] === true) {
				$this->_controller->Flash->success($result['data']['message']);
			} else {
				$return = false;
				$this->_controller->Flash->fail($result['data']['errors']);
			}
		}

		return $return;
	}

	/**
	 * process the delivery addresses
	 *
	 * @return 	bool
	 */
	public function processDeliveryAddress() {
		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		$return = true;

		// check to see if we are wanting to add and select an address
		if (isset($this->_controller->request->data['Form']['action']) && $this->_controller->request->data['Form']['action'] == 'addAddress') {
			if (empty($this->_controller->AddressManager)) {
				$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
			}

			$AddressModel = EvClassRegistry::init('EvAddressBook.Address');
			$result = $this->_controller->AddressManager->add(
				$AddressModel,
				'EvCore.User',
				$this->_controller->Auth->user('User.id')
			);

			// if successful select the address
			if ($result['result'] === true) {
				$this->_controller->Flash->success($result['data']['message']);
				$this->_controller->AddressManager->selectAddress(
					'delivery',
					$result['data']['Address']['id']
				);
			} else {
				$return = false;
				$this->_controller->Flash->fail($result['data']['errors']);
			}
		}

		// check to see if we are wanting to select an existing address
		if (isset($this->_controller->request->data['Form']['action']) && $this->_controller->request->data['Form']['action'] == 'selectAddress') {
			if (empty($this->_controller->AddressManager)) {
				$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
			}

			if (! empty($this->_controller->request->data['Address']['id'])) {
				$AddressModel = EvClassRegistry::init('EvAddressBook.Address');
				$result = $this->_controller->AddressManager->addressBelongsToUser(
					$AddressModel,
					$this->_controller->request->data['Address']['id'],
					$this->_controller->Auth->user('User.id')
				);

				if ($result['result'] === true) {
					$result = $this->_controller->AddressManager->addressHasShipping(
						$AddressModel,
						$this->_controller->request->data['Address']['id']
					);

					if ($result['result'] === true) {
						$this->_controller->AddressManager->selectAddress(
							'delivery',
							$this->_controller->request->data['Address']['id']
						);
					} else {
						$return = false;
						$this->_controller->Flash->fail($result['data']['errors']);
					}

				} else {
					$return = false;
					$this->_controller->Flash->fail($result['data']['errors']);
				}
			} else {
				$return = false;
				$this->_controller->Flash->fail(
					array(
						'title' => 'No Address Selected',
						'description' => 'You did not select an address, please either check one of the addresses before selecting or use the other form to enter a new address'
					)
				);
			}
		}

		return $return;
	}

	/**
	 * build a transaction from a basket
	 *
	 * @return bool
	 */
	public function buildTransactionFromBasket() {
		if (! CakePlugin::loaded('EvBasket')) {
			$this->_controller->Flash->fail('EvBasket is not loaded. Cannot process basket');
			return false;
		}

		if (! CakePlugin::loaded('EvAddressBook')) {
			$this->_controller->Flash->fail('EvAddressBook is not loaded. Cannot retrieve addresses');
			return false;
		}

		if (! CakePlugin::loaded('EvTransactions')) {
			$this->_controller->Flash->fail('EvTransactions is not loaded. Cannot setup transaction');
			return false;
		}

		// load the transactions component
		if (empty($this->_controller->TransactionsManager)) {
			$this->_controller->TransactionsManager = $this->_controller->loadComponent('EvTransactions.Transactions');
		}

		// load the address manager component
		if (empty($this->_controller->AddressManager)) {
			$this->_controller->AddressManager = $this->_controller->loadComponent('EvAddressBook.AddressManager');
		}

		// load the basket manager so we can get the basket
		if (empty($this->_controller->BasketManager)) {
			$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
		}

		// load the order manager component
		if (empty($this->_controller->OrderManager)) {
			$this->_controller->OrderManager = $this->_controller->loadComponent('EvCheckout.OrderManager');
		}

		//Check the basket is sensible before proceeding to transaction gateway otherwise redirect back to basket with message
		$canPurchase = $this->_controller->BasketManager->isBasketPurchasable();
		if ($canPurchase !== true) {
			$this->_controller->Flash->fail($canPurchase);

			$this->_controller->redirect([
				'plugin' => 'ev_basket',
				'controller' => 'basket',
				'action' => 'index'
			]);
		}

		// assign the additional data array, ready to pass through to _processPaymentTaking
		$additionalData = array();

		// find the addresses to save to the transaction
		$additionalData['addresses'] = $this->_controller->AddressManager->findAddresses();

		$currencies = Configure::read('currencies');
		if (isset($currencies[$this->_controller->BasketManager->basketCache['Basket']['currency_id']])) {
			$additionalData['Currency'] = $currencies[$this->_controller->BasketManager->basketCache['Basket']['currency_id']];
		} else {
			return false;
		}

		// Build the transaction data from the basket data and the address
		$transactionData = array();

		foreach ($this->_controller->BasketManager->getBasketData() as $basketData) {
			$transactionData[$basketData['name']] = $basketData['data'];
		}

		// Create an order from the basket. We do this now so on external payment
		// gateways they don't alter the order whilst on the payment site to get
		// free items etc.
		$selectedBuilder = Configure::read('EvCheckout.orderBuilder');

		$OrderBuilder = $this->_controller->OrderManager->loadBuilder($selectedBuilder);

		if (is_object($OrderBuilder)) {
			$modelId = $OrderBuilder->createFromBasket(
				$this->_controller->BasketManager->basketCache,
				$this->_controller->Auth->user('User'),
				$additionalData['addresses'],
				false // Don't delete the basket
			);
		}

		// store a decoded version of the transaction address data
		$transactionData['address'] = json_encode($additionalData['addresses']);

		// Set the order ID in the session so we can verify guest users on the callback
		$this->_controller->Session->write('EvCheckout.last_order_id', $modelId);

		// go out to additional method for the processing of payment
		// to allow for easy extension of the functionality.
		// this way one, or more than one, payment process can be used
		$this->_processPaymentTaking(
			$transactionData,
			$modelId,
			$this->_controller->request->data,
			$additionalData
		);

		return true;
	}

/**
 * Passes the transaction data to transaction manager to take payment, this also
 * allows for extension so more than one payment method can be handled
 * i.e. SagePay and PayPal
 *
 * @param array $transactionData Contains the transaction data
 * @param int $modelId The order id for the newly created basket
 * @return void
 */
	protected function _processPaymentTaking(
		$transactionData = array(),
		$modelId = null,
		$requestData = array(),
		$additionalData = array()
	) {
		// get the order totals; NOT basket
		$transactionTotals = $this->_controller->OrderManager->buildTransactionTotals();

		// take the payment
		$this->_controller->TransactionsManager->takePayment(
			array(
				'return' => $this->_controller->OrderManager->getCompleteUrl(),
				'cancel' => $this->_controller->OrderManager->getCancelledUrl()
			),
			array(
				'model' => 'EvCheckout.Order',
				'model_id' => $modelId,
			),
			array(
				'amount' => $transactionTotals['grandtotal'],
				'currency' => $additionalData['Currency']
			),
			$this->_controller->OrderManager->buildTransactionItems(),
			$this->_controller->OrderManager->buildTransactionExtras(
				$additionalData['addresses'],
				$this->_controller->Auth->user(),
				$requestData
			),
			$transactionData
		);
	}

}
