<?php

App::uses('AppComponent', 'Controller/Component');
App::uses('OrderStatus', 'EvCheckout.Model');
App::uses('BasketLib', 'EvBasket.Lib');
App::uses('RouterUtil', 'Routable.Lib');
App::uses('AddressBookLib', 'EvAddressBook.Lib');

class OrderEditingComponent extends AppComponent {

	protected $_orderEditingSession = null;

/**
 * Check for an editing session on load and send an event if one is found.
 *
 * @param Controller $controller The controller
 * @return void
 */
	public function initialize(Controller $controller) {
		parent::initialize($controller);

		$editOrder = $this->checkForEditOrder();
		if (!empty($editOrder)) {
			$controller->getEventManager()->dispatch(
				new CakeEvent('EvCheckout.EditOrder.currentlyEditing', $this, array(
					'editOrder' => $editOrder,
				))
			);
		}
	}

/**
 * Updates a saved order to match a basket
 *
 * @param int $orderId An order ID
 * @param Basket $basket A basket that contains BasketItem, BasketItemData, BasketData, BasketTotal.
 * @return bool True for a successful save, false for an unsuccessful one.
 */
	public function updateOrderFromBasket($orderId, $basket) {
		// Load the required model / component
		if (empty($this->_controller->Order)) {
			$this->_controller->loadModel('EvCheckout.Order');
		}
		if (empty($this->_controller->OrderManager)) {
			$this->_controller->OrderManager = $this->_controller->loadComponent('EvCheckout.OrderManager');
		}

		$order = $this->_controller->Order->readForEdit($orderId);

		$orderItemsUpdated = [];
		$orderTotalsUpdated = [];
		$orderDataUpdated = [];
		$orderItems = Hash::combine($order, ['%s:%s', 'OrderItem.{n}.model', 'OrderItem.{n}.model_id'], 'OrderItem.{n}');
		$orderTotals = Hash::combine($order, 'OrderTotal.{n}.name', 'OrderTotal.{n}');
		$orderData = Hash::combine($order, 'OrderData.{n}.name', 'OrderData.{n}');

		// Start a transaction so if anything doesn't save correctly the whole change is reverted.
		$this->_controller->Order->getDataSource()->begin();
		$success = true;

		try {
			// Update / create the order items and their data
			foreach ($basket['BasketItem'] as $basketItem) {
				$itemKey = $basketItem['model'] . ':' . $basketItem['model_id'];
				$orderItem = !empty($orderItems[$itemKey]) ? $orderItems[$itemKey] : null;
				$orderItemId = $this->_updateOrderItemFromBasketItem($orderId, $orderItem, $basketItem);

				$orderItemsUpdated[] = $orderItemId;
			}

			// Delete any order items that have been removed
			$this->_controller->Order->OrderItem->deleteAll([
				'NOT' => ['OrderItem.id' => $orderItemsUpdated],
				'OrderItem.order_id' => $orderId
			]);

			// Update the order totals
			foreach ($basket['BasketTotal'] as $basketTotal) {
				$orderTotal = !empty($orderTotals[$basketTotal['name']]) ? $orderTotals[$basketTotal['name']] : null;
				$orderTotalId = $this->_updateOrderTotalFromBasketTotal($orderId, $orderTotal, $basketTotal);
				$orderTotalsUpdated[] = $orderTotalId;
			}

			// Delete any order totals that have been removed
			$this->_controller->Order->OrderTotal->deleteAll([
				'NOT' => ['OrderTotal.id' => $orderTotalsUpdated],
				'OrderTotal.order_id' => $orderId
			]);

			// Update the order data
			foreach ($basket['BasketData'] as $basketData) {
				$orderData = !empty($orderData[$basketData['name']]) ? $orderData[$basketData['name']] : null;
				$orderDataId = $this->_updateOrderDataFromBasketData($orderId, $orderData, $basketData);
				$orderDataUpdated[] = $orderDataId;
			}

			// Delete any order data that have been removed
			$this->_controller->Order->OrderData->deleteAll([
				'NOT' => ['OrderData.id' => $orderDataUpdated],
				'OrderData.order_id' => $orderId
			]);

			// Done

		} catch (Exception $e) {
			$success = false;
		}

		// If all went well then commit
		if ($success) {
			$this->_controller->Order->getDataSource()->commit();
			$this->_controller->Flash->success(['title' => 'The order was updated']);
		} else {
			$this->_controller->Order->getDataSource()->rollback();
			$this->_controller->Flash->fail(['title' => 'Saving the order failed']);
		}

		return $success;
	}

/**
 * Update an order item using a basket item. This will save the output.
 *
 * @param int $orderId The order id that the item belongs to
 * @param OrderItem $orderItem An order item. If orderItem is null a new order item will be created
 * @param BasketItem $basketItem A basket item
 * @return int the updated item id
 */
	protected function _updateOrderItemFromBasketItem($orderId, $orderItem, $basketItem) {
		$defaultItemData = $this->_controller->OrderManager->setupOrderItemData($basketItem);
		$defaultItemData = Hash::combine($defaultItemData, '{n}.name', '{n}.item_data');

		$updatedItem = [
			'quantity' => $basketItem['quantity'],
			'unit_price' => $basketItem['unit_price'],
			'unit_price_inc_tax' => $basketItem['unit_price'] + $basketItem['unit_tax'],
			'row_total' => $basketItem['row_total'],
			'row_total_inc_tax' => $basketItem['row_total'] + ($basketItem['quantity'] * $basketItem['unit_tax']),
			'tax_rate' => $basketItem['tax_rate']
		];

		if (!empty($orderItem)) {
			$updatedItem['id'] = $orderItem['id'];
		} else {
			// This is a new order item. Fill in the rest of the data
			$updatedItem = $this->_addDataToNewOrderItem($orderId, $basketItem, $updatedItem);
		}

		$this->_controller->Order->OrderItem->clear();
		$this->_controller->Order->OrderItem->save(['OrderItem' => $updatedItem]);
		$orderItemId = $this->_controller->Order->OrderItem->id;

		$this->_updateOrderItemDataFromBasketItemData(
			$orderItemId,
			!empty($orderItem['OrderItemData']) ? $orderItem['OrderItemData'] : [],
			!empty($basketItem['BasketItemData']) ? array_merge(Hash::combine($basketItem['BasketItemData'], '{n}.name', '{n}.item_data'), $defaultItemData) : $defaultItemData
		);

		return $orderItemId;
	}

/**
 * When a new item has been added to the basket while it was being edited, extra data needs to be added to the updated
 * order item. The new data is added here.
 *
 * @param int   $orderId     The id of the order that is being edited.
 * @param array $basketItem  The item that was added to the basket during editing.
 * @param array $updatedItem The new item to be added to the order.
 * @return array Modified `$updatedItem`.
 */
	protected function _addDataToNewOrderItem($orderId, $basketItem, $updatedItem) {
		$updatedItem['name'] = BasketLib::getItemName($basketItem);
		$updatedItem['order_id'] = $orderId;
		$updatedItem['model'] = $basketItem['model'];
		$updatedItem['model_id'] = $basketItem['model_id'];

		return $updatedItem;
	}

/**
 * Updates the item data. Expects Key value pair arrays.
 * - This will remove any data that exist in order that don't exist in basket
 * - It will add and data that exist in basket but not in order
 * - It will update any that exist both using the value in basket
 *
 * @param int $orderItemId The order item id that the data belongs to
 * @param array $orderItemData Key => Value array of current orderItemData
 * @param array $basketItemData Key => Value array of updated data
 * @return void
 */
	protected function _updateOrderItemDataFromBasketItemData($orderItemId, $orderItemData, $basketItemData) {
		$dataSource = $this->_controller->Order->OrderItem->OrderItemData->getDataSource();

		foreach ($basketItemData as $name => $data) {
			if (isset($orderItemData[$name])) {
				// Both exist, update the value
				$this->_controller->Order->OrderItem->OrderItemData->updateAll([
					'OrderItemData.item_data' => $dataSource->value($data),
				], [
					'OrderItemData.name' => $name,
					'OrderItemData.order_item_id' => $orderItemId
				]);
			} else {
				// This is new data
				$this->_controller->Order->OrderItem->OrderItemData->clear();
				$this->_controller->Order->OrderItem->OrderItemData->save([
					'OrderItemData' => [
						'order_item_id' => $orderItemId,
						'name' => $name,
						'item_data' => $data,
					]
				]);
			}
		}

		// Remove any removed data
		$removedNames = array_diff(array_keys($orderItemData), array_keys($basketItemData));
		$this->_controller->Order->OrderItem->OrderItemData->deleteAll([
			'OrderItemData.order_item_id' => $orderItemId,
			'OrderItemData.name' => $removedNames
		]);
	}

/**
 * Update an order total using a basket total. This will save the output.
 *
 * @param int $orderId The order id this total should belong to
 * @param OrderTotal $orderTotal An order total. If orderTotal is null a new order total will be created
 * @param BasketTotal $basketTotal A basket total
 * @return int The updated total id
 */
	protected function _updateOrderTotalFromBasketTotal($orderId, $orderTotal, $basketTotal) {
		$updatedTotalData = [
			'amount' => $basketTotal['amount'],
			'display_inc_tax' => $basketTotal['display_inc_tax'],
			'display_ex_tax' => $basketTotal['display_ex_tax'],
			'sequence' => $basketTotal['sequence'],
		];

		if (!empty($orderTotal)) {
			$updatedTotalData['id'] = $orderTotal['id'];
		} else {
			// This is a new order item. Fill in the rest of the data
			$updatedTotalData['name'] = $basketTotal['name'];
			$updatedTotalData['order_id'] = $orderId;
		}
		$this->_controller->Order->OrderTotal->clear();
		$this->_controller->Order->OrderTotal->save(['OrderTotal' => $updatedTotalData]);

		return $this->_controller->Order->OrderTotal->id;
	}

/**
 * Update an order Data using a basket Data. This will save the output.
 *
 * @param int $orderId The order id this Data should belong to
 * @param OrderData $orderData An order Data. If orderData is null a new order Data will be created
 * @param BasketData $basketData A basket Data
 * @return int The updated Data id
 */
	protected function _updateOrderDataFromBasketData($orderId, $orderData, $basketData) {
		$updatedData = [
			'data' => $basketData['data']
		];

		if (!empty($orderData)) {
			$updatedData['id'] = $orderData['id'];
		} else {
			// This is a new order item. Fill in the rest of the data
			$updatedData['name'] = $basketData['name'];
			$updatedData['order_id'] = $orderId;
			$updatedData['is_visible'] = $basketData['is_visible'];
		}
		$this->_controller->Order->OrderData->clear();
		$this->_controller->Order->OrderData->save(['OrderData' => $updatedData]);

		return $this->_controller->Order->OrderData->id;
	}

/**
 * Starts editing an order in the current session
 *
 * @param int $orderId The order to begin editing
 * @return array|bool An OrderEditingSession of false on failure
 */
	public function beginEditOrderSession($orderId) {
		if (CakePlugin::loaded('EvBasket')) {
			// Load the order
			if (empty($this->_controller->Order)) {
				$this->_controller->loadModel('EvCheckout.Order');
			}
			$order = $this->_controller->Order->readForEdit($orderId);

			if (!empty($order['OrderEditingSession'])) {
				// There is already an editing session going on
				return false;
			}

			if (empty($order) || !$this->_controller->Order->OrderStatus->isEditableState($order['OrderStatus']['slug'])) {
				// This order is not editable
				return false;
			}

			// Create a temporary basket from the current order
			if (empty($this->_controller->BasketManager)) {
				$this->_controller->BasketManager = $this->_controller->loadComponent('EvBasket.Basket');
			}
			$basketHash = $this->_controller->BasketManager->createNewBasketFromOrder($order);

			// Start the session
			if (empty($this->_controller->OrderEditingSession)) {
				$this->_controller->loadModel('EvCheckout.OrderEditingSession');
			}
			$editSession = [
				'OrderEditingSession' => [
					'user_id' => $this->_controller->Auth->user()['User']['id'],
					'order_id' => $orderId,
					'basket_hash' => $basketHash
				]
			];
			$this->_orderEditingSession = $this->_controller->OrderEditingSession->save($editSession);

			if (!empty($this->_orderEditingSession)) {
				$this->_controller->Session->write('EvCheckout.editOrderSessionId', $this->_orderEditingSession['OrderEditingSession']['id']);
				return $this->_orderEditingSession['OrderEditingSession'];
			}
		}
		return false;
	}

/**
 * Checks for an order that is being edited
 *
 * @return array|bool Will return either false for not editing an order or the edit order details to be passed to the view
 */
	public function checkForEditOrder() {
		// Check for an editing session the first time. If not found this will be set to false and not checked again on this request
		if ($this->_orderEditingSession === null) {
			$editOrderSessionId = $this->_controller->Session->read('EvCheckout.editOrderSessionId');

			if (!empty($editOrderSessionId)) {
				// Check this exists in the db and has not been ended elsewhere
				if (empty($this->_controller->OrderEditingSession)) {
					$this->_controller->loadModel('EvCheckout.OrderEditingSession');
				}
				$this->_orderEditingSession = $this->_controller->OrderEditingSession->findById($editOrderSessionId);
			} else {
				// Don't bother checking again this request
				$this->_orderEditingSession = false;
			}
		}

		// Check there is a current editing session for the correct user and it has not expired.
		if (!empty($this->_orderEditingSession)) {
			if (
				$this->_orderEditingSession['OrderEditingSession']['created'] > date('Y-m-d H:i:s', strtotime(Configure::read('EvCheckout.editOrderSessionExpiry'))) &&
				$this->_controller->Auth->user()['User']['id'] == $this->_orderEditingSession['OrderEditingSession']['user_id']
			) {
				return $this->_orderEditingSession['OrderEditingSession'];
			} else {
				$this->_removeEditingSession($this->_orderEditingSession['OrderEditingSession']);
			}
		}

		return false;
	}

/**
 * Ends the session for editing an order
 *
 * @param int $orderId If set, ends all sessions for this order ID. If not set, it will end the current session.
 * @return void
 */
	public function endEditOrderSession($orderId = null) {
		if (empty($this->_controller->OrderEditingSession)) {
			$this->_controller->loadModel('EvCheckout.OrderEditingSession');
		}

		if (!empty($orderId)) {
			// There may be more than one if something has not gone according to plan
			$editingSessions = $this->_controller->OrderEditingSession->find('all', [
				'conditions' => [
					'order_id' => $orderId
				]
			]);

			if (!empty($editingSessions)) {
				foreach ($editingSessions as $editingSession) {
					$this->_removeEditingSession($editingSession['OrderEditingSession']);
				}
			}
		} else {
			// End the editing session in the current user session.
			$editingSession = $this->checkForEditOrder();
			if (!empty($editingSession)) {
				$this->_removeEditingSession($editingSession);
			}
		}
	}

/**
 * Removes an editing session. This should be used within this component instead of endEditOrderSession to avoid infinite loops
 * Also clears any expired sessions.
 *
 * @param array $editingSession The session to end. This should be the same as the output from checkForEditOrder.
 * @return void
 */
	protected function _removeEditingSession($editingSession) {
		if (CakePlugin::loaded('EvBasket')) {
			// Delete the temporary basket
			if (empty($this->_controller->Basket)) {
				$this->_controller->loadModel('EvBasket.Basket');
			}
			$this->_controller->Basket->deleteAll([
				'hash' => $editingSession['basket_hash']
			]);
		}

		if (!empty($this->_orderEditingSession) && $this->_orderEditingSession['OrderEditingSession']['id'] == $editingSession['id']) {
			// We are ending the current session. Clear out the session vars too.
			$this->_controller->Session->delete('EvCheckout.editOrderSessionId');
			$this->_orderEditingSession = false;
		}

		if (empty($this->_controller->OrderEditingSession)) {
			$this->_controller->loadModel('EvCheckout.OrderEditingSession');
		}
		// Clear out any old sessions too
		$this->_controller->OrderEditingSession->deleteAll([
			'OR' => [
				'OrderEditingSession.created < ' => date('Y-m-d H:i:s', strtotime(Configure::read('EvCheckout.editOrderSessionExpiry'))),
				'OrderEditingSession.id' => $editingSession['id']
			]
		]);
	}
}
